=head1 NAME

Geneos::API - Handy Perl interface to ITRS Geneos XML-RPC Instrumentation API

=head1 VERSION

Version 0.11

=head1 SYNOPSIS

    use Geneos::API;

    # open API to NetProbe running on example.com and port 7036
    my $api = Geneos::API->new("http://example.com:7036/xmlrpc");

    # get the sampler "Keeper" in the managed entity "Zoo"
    my $sampler = $api->get_sampler("Zoo", "Keeper");

    # create view "Monkeys" in the group "Animals"
    my $view = $sampler->create_view("Monkeys", "Animals");

    # prepare some data
    my $monkeys = [
        ["Name",   "Type"  ],
        ["Funky",  "Monkey"],
        ["Cheeky", "Dog"   ]
    ];

    # populate the view
    $view->update_entire_table($monkeys);

=head1 DESCRIPTION

C<Geneos::API> is a Perl module that implements ITRS XML-RPC Instrumentation API

=head1 METHODS

=head2 Constructor

=head3 $api->new($url, $options)

$url is required and must be in the form:

C<http://host:port/path>

For example:

    my $api = Geneos::API->new("http://localhost:7036/xmlrpc");

XML-RPC Client is initialized upon call to the API constructor

Options

The constructor accepts a reference to the options hash as optional second parameter:

    my $api = Geneos::API->new("http://localhost:7036/xmlrpc", {
        api => {raise_error => 1,},
        ua  => {timeout => 5,},
    });

=head4 api - XML-RPC options

=over

=item * C<< raise_error >>

Force errors to raise exceptions via C<Carp::croak>

=item * C<< print_error >>

Force errors to raise warnings via C<Carp::carp>

=back

raise_error take precedence over print_error. If neither is set, the errors won't be 
reported and C<$api->error> method will need to be called to check 
if the latest call generated an error or not.

=head4 ua - UserAgent options

=over

=item * C<< any options supported by L<LWP::UserAgent> >>

=back

=head2 Error handling

=head3 $api->raise_error()

Get the raise_error attribute value

Returns true is the raise_error attribute is set or false othersise

If the raise_error attribute is set, errors generated by API calls will be passed to C<Carp::croak>

If none of the above are set

=head3 $api->print_error()

Get the print_error attribute value

Returns true is the print_error attribute is set or false othersise

If the print_error attribute is set, errors generated by API calls will be passed to C<Carp::carp>

print_error attribute is ignored if raise_error is set.

=head3 $api->status_line

Returns the string "<code> <message>". Returns undef if there is no error.

=head3 $api->error

Get the error produced by the last api call.

Returns reference to the error hash or undef if the last call produced no error.
The hash contains three elements:

=over

=item * code

HTTP or XML-RPC error code.

=item * message

Error string.

=item * class

The component that produced the error: C<HTTP> or C<XML-RPC>.

=back

=head2 API Function Calls

There are two classes that represent Samplers and Views. First, a sampler object must be created:

$sampler = $api->get_sampler(managedEntity, samplerName)

This will create a Sampler object representing a sampler with the name "samplerName"
in the managed entity "managedEntity". You can call any method from the section "Sampler calls" on this object.

Views are represented by the internal Geneos::API::Sampler::View class. 
In order to create an instance of this class, you can use:

$view = $sampler->get_view(viewName, groupHeading)

if the view already exists, or

$view = $sampler->create_view(viewNmae, groupHeading)

if the view does not exist yet and you want to create one.

Once the view object is created, you can call any of the "View calls" on it.

These methods are described below:

=head2 Sampler calls

=head3 $api->get_sampler(managedEntity, sampler)

This method doesn't check whether the sampler exists.

Returns sampler object.

=head3 $sampler->create_view(viewName, groupHeading)

Creates a new, empty view in the specified sampler under the specified groupHeading.
This method will create a view and returns the object representing it. An error will be produced if the view already exists.

Returns OK on successful completion.

=head3 $sampler->get_view(viewName, groupHeading)

The view must already exist. This method will NOT check that the view extists or not. 
Use C<$sampler->view_exists> method for that.

Returns an object representing the view viewName.

=head3 $sampler->view_exists(viewName)

Checks whether a particular view exists in this sampler. viewName should be in the form group-view.

Returns true if the view exists, false otherwise.

=head3 $sampler->remove_view(viewName)

Removes a view that has been created with createView.

Returns OK on successful completion.

=head3 $sampler->get_parameter(parameterName)

Retrieves the value of a sampler parameter that has been defined in the gateway configuration.

Returns the parameter text written in the gateway configuration.

=head3 $sampler->sign_on(period)

period - The maximum time between updates before samplingStatus becomes FAILED

Commits the API client to provide at least one heartbeat or update to the view within the time period specified.

Returns OK on successful completion.

=head3 $sampler->sign_off()

Cancels the commitment to provide updates to a view.

Returns OK on successful completion.

=head3 $sampler->heartbeat()

Prevents the sampling status from becoming failed when no updates are needed to a view and the client is signed on.

Returns on successful completion.

=head2 View calls

=head3 $view->add_table_row(rowName,data)

Adds a new, table row to the specified view and populates it with data Returns OK on successful completion.

=head3 $view->remove_table_row(rowName)

Removes an existing row from the specified view.

Returns OK on successful completion.

=head3 $view->add_headline(headlineName)

Adds a headline variable to the view.

Returns OK on successful completion.

=head3 $view->remove_headline(headlineName)

Removes a headline variable from the view.

Returns OK on successful completion.

=head3 $view->update_variable(variableName, newValue)

Can be used to update either a headline variable or a table cell.
If the variable name contains a period (.) then a cell is assumed, otherwise a headline variable is assumed.

Returns OK on successful completion.

=head3 $view->update_headline(headlineName, newValue)

Updates a headline variable.

Returns OK on successful completion.

=head3 $view->update_table_cell(cellName, newValue))

Updates a single cell in a table. The standard row.column format should be used to reference a cell.

Returns OK on successful completion.

=head3 $view->update_table_row(rowName, newValue))

Updates an existing row from the specified view with the new values provided.

Returns OK on successful completion.

=head3 $view->add_table_column(columnName)

Adds another column to the table.

Returns OK on successful completion.

=head3 $view->update_entire_table(newTable)

Updates the entire table for a given view. This is useful if the entire table 
will change at once or the table is being created for the first time. 
The array passed should be two dimensional. The first row should be the column 
headings and the first column of each subsequent row should be the name of the row.
The array should be at least 2 columns by 2 rows. Once table columns have been defined, 
they cannot be changed by this method.

Returns OK on successful completion.

=head3 $view->column_exists(columnName)

Check if the headline variable exists.

Returns true if the column exists, false otherwise.

=head3 $view->row_exists(rowName)

Check if the headline variable exists.

Returns true if the row exists, false otherwise.

=head3 $view->headline_exists(headlineName)

Check if the headline variable exists.

Returns true if the headline variable exists, false otherwise.

=head3 $view->get_column_count()

Return the column count of the view.

Returns the number of columns in the view. This includes the rowName column.

=head3 $view->get_row_count()

Return the headline count of the view.

Returns the number of headlines in the view. This includes the samplingStatus headline.

=head3 $view->get_headline_count()

Returns the number of headlines in the view. This includes the samplingStatus headline.

=head3 $view->get_column_names()

Returns the names of existing columns in the view. This includes the rowNames column name.

=head3 $view->get_row_names()

Returns the names of existing rows in the view

=head3 $view->get_headline_names()

Returns the names of existing headlines in the view. This includes the samplingStatus headline.

=head3 $view->get_row_names_older_than(epoch)

epoch - The timestamp against which to compare row update time.
The timestamp should be provided as Unix timestamp, i.e. number of seconds elapsed since UNIX epoch.

Returns the names of rows whose update time is older than the time provided.

=head2 NetProbe Function Calls

=head3 $api->managed_entity_exists(managedEntity)

Checks whether a particular Managed Entity exists on this NetProbe containing any API or API-Streams samplers.
Returns true if the Managed Entity exists, false otherwise

=head3 $api->sampler_exists(managedEntity, sampler)

Checks whether a particular API or
API-Streams sampler exists on this NetProbe Returns true if sampler exists, false otherwise

=head3 $api->gateway_connected()

Checks whether the Gateway is connected to this NetProbe Returns true if the Gateway is connected, false otherwise

=head2 Gateway Function Calls

=head3 $api->add_managed_entity($managed_entity, $data_section)

Adds the managed entity to the particular data section Returns true on success, false otherwise

=head1 ONLINE RESOURCES AND SUPPORT

=over 4

=item * L<http://www.itrsgroup.com/> ITRS Group - Application Performance Management in Real-Time

=item * L<http://xmlrpc.scripting.com/> XML-RPC resource

=back

=head1 AUTHOR

Ivan Dmitriev, E<lt>tot@cpan.orgE<gt>

=head1 COPYRIGHT AND LICENSE

Copyright (C) 2014 by Ivan Dmitriev 

This library is free software; you can redistribute it and/or modify
it under the same terms as Perl itself, either Perl version 5.12.3 or,
at your option, any later version of Perl 5 you may have available.

=cut

    ###############################################################
    #
    # package Geneos::API::XMLRPC::Response
    #
    # Parses XML-RPC response and converts it into Perl structure
    #
    ###############################################################

    package Geneos::API::XMLRPC::Response;

    use strict;
    use warnings;

    use XML::LibXML qw(:libxml);

    sub new {
        my $this = shift;
        my $class = ref($this) || $this;
        my $self = {
            _response => {},
            _error => undef,
        };

        bless $self, $class;
        $self->_init(@_);
    }

    sub _init {
        my ($self, $response) = @_;

        # Check if the HTTP request succeeded
        if ($response->is_success) {

            my $dom = XML::LibXML->load_xml(string => $response->decoded_content);
            process_node($self->{_response}, $dom->documentElement);

            if (exists $self->{_response}{fault}) {
                my $code = exists $self->{_response}{fault}{faultCode}
                         ? $self->{_response}{fault}{faultCode}
                         : -1;

                my $str  = exists $self->{_response}{fault}{faultString}
                         ? $self->{_response}{fault}{faultString}
                         : 'NO_ERROR_STRING';

                $self->error({class=>"XML-RPC", code=>$code, message=>$str,});
            }

        }
        else {
            $self->error({class=>"HTTP", code=>$response->code, message=>$response->message,});
        }

        return $self;
    }

    sub is_success {!shift->error}

    sub params {shift->{_response}{params}}

    sub error {
        my ($self, $error) = @_;
        $self->{_error} = $error if $error;

        return $self->{_error};
    }

    # ---------------
    # Response parser

    sub process_node {
        my ($r, $node) = @_;

        for my $child ($node->childNodes) {

            if ($child->nodeName eq "struct") {
                process_struct($r, $child);
            }
            elsif ($child->nodeName eq "fault") {
                process_fault($r, $child);
            }
            elsif ($child->nodeName eq "params") {
                process_params($r, $child);
            }
            elsif ($child->nodeName eq "array") {
                process_array($r, $child);
            }
            elsif ($child->nodeName =~ m/^i4|int|boolean|string|double|dateTime\.iso8601|base64$/) {
                $$r = $child->textContent;
            }
            elsif ($child->nodeType == 3
                && $node->nodeName eq "value"
                && $node->childNodes->size == 1
            ) {
                $$r = $child->textContent;
            }
            else {
                process_node($r, $child);
            }
        }
    }

    sub process_fault {
        my ($r, $node) = @_;

        my ($value) = $node->findnodes("./value");

        process_node(\$r->{fault}, $value);
    }

    sub process_struct {
        my ($r, $node) = @_;

        foreach my $member ( $node->findnodes("./member") ) {
            my ($name)  = $member->findnodes("./name");
            my ($value) = $member->findnodes("./value");

            process_node(\$$r->{$name->textContent}, $value);
        }
    }

    sub process_array {
        my ($r, $node) = @_;

        foreach my $value ( $node->findnodes("./data/value") ) {
            process_node(\$$r->[++$#{$$r}], $value);
        }
    }

    sub process_params {
        my ($r, $node) = @_;

        $r->{params} = [];

        foreach my $param ( $node->findnodes("./param") ) {
            my ($value) = $param->findnodes("./value");
            process_node(\$r->{params}[++$#{$r->{params}}], $value);
        }
    }

    ###########################################
    #
    # package Geneos::API::XMLRPC::Request
    #
    # Converts method and Perl data structure
    # into an XML-RPC request
    #
    ###########################################

    package Geneos::API::XMLRPC::Request;

    use XML::LibXML;

    sub new {
        my $this = shift;
        my $class = ref($this) || $this;
        my $self = {};
        bless $self, $class;
        $self->_init(@_);
    }

    # Private methods

    sub _init {
        my ($self, $method, @args) = @_;

        $self->{doc} = XML::LibXML::Document->new('1.0', 'utf-8');

        my $root = $self->{doc}->createElement("methodCall");
        $self->{doc}->setDocumentElement($root);

        # ------------------
        # Add the methodName

        my $methodName = $self->{doc}->createElement("methodName");
        $methodName->appendTextNode($method);
        $root->appendChild($methodName);

        # --------------
        # Add the params
        my $params = $self->{doc}->createElement("params");
        $root->appendChild($params);

        # ---------------------
        # Process the agruments
        foreach (@args) {
            my $param = $self->{doc}->createElement("param");
            $params->addChild($param);
            $self->parse($param, $_);
        }

        return $self;
    }

    # Public methods

    sub content {shift->{doc}->toString}

    sub parse {
        my ($self, $node, $p) = @_;

        my $value = $self->{doc}->createElement("value");
        $node->addChild($value);

        if ( ref($p) eq 'HASH' ) {
            $self->parse_hash($value,$p);
        }
        elsif ( ref($p) eq 'ARRAY' ) {
            $self->parse_array($value,$p);
        }
        elsif ( ref($p) eq 'CODE' ) {
            $self->parse_code($value,$p);
        }
        else {
            $self->parse_scalar($value,$p);
        }
    }

    # It seems that Geneos treats everything as a string
    # no need for anything sophisticated here

    sub parse_scalar {
        my ($self, $node, $scalar) = @_;

        $scalar ||= "";

        if (( $scalar =~ m/^[\-+]?\d+$/) && (abs($scalar) <= (0xffffffff >> 1))) {
            my $i = $self->{doc}->createElement("i4");
            $i->appendTextNode($scalar);
            $node->appendChild($i);
        }
        elsif ( $scalar =~ m/^[\-+]?\d+\.\d+$/ ) {
            my $d = $self->{doc}->createElement("double");
            $d->appendTextNode($scalar);
            $node->appendChild($d);
        }
        else {
            my $s = $self->{doc}->createElement("string");
            $s->appendTextNode($scalar);
            $node->appendChild($s);
        }
    }

    sub parse_hash {
        my ($self, $node, $hash) = @_;

        my $struct = $self->{doc}->createElement("struct");
           $node->appendChild($struct);

           foreach (keys %$hash) {
               my $member = $self->{doc}->createElement("member");
               $struct->appendChild($member);

               my $name = $self->{doc}->createElement("name");
               $name->appendTextNode($_);
               $member->appendChild($name);

               $self->parse($member, $hash->{$_});
           }
    }

    sub parse_array {
        my ($self, $node, $args) = @_;

        my $array = $self->{doc}->createElement("array");
        $node->appendChild($array);

        my $data = $self->{doc}->createElement("data");
        $array->appendChild($data);

        $self->parse($data, $_) for @$args;
    }

    sub parse_code {
        my ($self, $node, $code) = @_;

        my ($type, $data) = $code->();

        my $e = $self->{doc}->createElement($type);
        $e->appendTextNode($data);
        $node->appendChild($e);
    }

    ########################################################################
    #
    # package Geneos::API::XMLRPC
    #
    # XML-RPC client
    # The reason for yet another XML-RPC implementation is that
    # because Geneos XML-RPC does not conform to the XML-RPC standard:
    #
    # * '-' is used in the method names
    # * the values do not default to type 'string'
    #
    # Among other reasons, ensuring that HTTP1.1 is used to take advantage
    # of the keep_alive feature supported by Geneos XML-RPC server
    #
    ########################################################################

    package Geneos::API::XMLRPC;

    use LWP::UserAgent;

    # -----------
    # Constructor

    sub new {
        my $this = shift;

        my $class = ref($this) || $this;
        my $self = {};
        bless $self, $class;

        $self->_init(@_);
    }

    # ---------------
    # Private methods

    sub _init {
        my ($self, $url, $opts) = @_;

        $self->{_url} = $url;

        $opts ||= {};
        $opts->{ua} ||= {};

        # set up the UserAgent
        $self->{_ua} = LWP::UserAgent->new(%{$opts->{ua}});

        return $self;
    }

    # --------------
    # Public methods

    sub request {
        my ($self, $method, @params) = @_;

        # prepare the XML-RPC request
        my $x_request = Geneos::API::XMLRPC::Request->new($method, @params);

        my $req = HTTP::Request->new("POST",$self->{_url});

        $req->header('Content-Type' => 'text/xml');
        $req->add_content_utf8($x_request->content);

        Geneos::API::XMLRPC::Response->new(
            $self->{_ua}->request($req)
        );
    }

    ##############################################################################
    # package Geneos::Base                                                       #
    #                                                                            #
    # This base class implements error handling and interface to                 #
    # Geneos::API::XMLRPC  that is used by Geneos::API, Geneos::API::Sampler and #
    # Geneos::API::Sampler::View classes                                         #
    #                                                                            #
    ##############################################################################

    package Geneos::Base;

    use Carp;

    our $VERSION = '0.11';

    sub new {bless({_error=>undef,}, shift)->_init(@_)}

    sub status_line {
        my $self = shift;
        $self->{_error}?join(" ", $self->{_error}->{code}, $self->{_error}->{message}):undef;
    }

    sub error {shift->{_error}}

    sub _error {
        my ($self, $error) = @_;
        $self->{_error} = $error if $error;

        return $self->{_error};
    }

    sub call {
        my ($self, $method, @params) = @_;

        $self->_reset_error;

        # send the XMLRPC request to the NetProbe
        my $response = $self->api->request($self->_method($method), @params);

        # check the response
        if ($response->is_success) {
            $response->params->[0];
        }
        else {
            $self->_handle_error($response->error);
        }
    }

    sub _reset_error {shift->{_error} = undef}

    sub _handle_error {
        my $self = shift;

        # record the error
        $self->_error(@_);

        croak $self->status_line if $self->api->raise_error;
        carp $self->status_line if $self->api->print_error;

        # always return undef
        return;
    }

    ######################################
    #
    # package Geneos::API::Sampler::View
    #
    # Implements all View methods
    #
    ######################################

    package Geneos::API::Sampler::View;

    use base 'Geneos::Base';

    sub _init {
        my ($self, $sampler, $view, $group) = @_;

        $self->{_sampler} = $sampler;
        $self->{_view}    = $view;
        $self->{_group}   = $group;

        return $self;
    }

    sub _method {
        my $self = shift;
        join(".", $self->{_sampler}->entity, $self->{_sampler}->sampler, "$self->{_group}-$self->{_view}", @_);
    }

    sub api {shift->{_sampler}->api}

    # API calls

    # ---------------------------------------------
    # Combines addTableRow and updateTableRow calls
    #
    sub add_table_row {
        my ($self, $name, $data) = @_;

        return unless $self->_add_table_row($name);

        # if there is data - add it to the row
        $data ? $self->update_table_row($name, $data) : 1;
    }

    # -----------------------------------------------------
    # Each method below is an XML-RPC call to the NetProbe
    #
    # The first argument passed to the call method is the
    # XML-RPC method name. The rest are parameters passed
    # with that call to the XML-RPC server
    #

    sub _add_table_row {shift->call("addTableRow", @_)}

    sub remove_table_row {shift->call("removeTableRow", @_)}

    sub add_headline {shift->call("addHeadline", @_)}

    sub remove_headline {shift->call("removeHeadline", @_)}

    sub update_variable {shift->call("updateVariable", @_)}

    sub update_headline {shift->call("updateHeadline", @_)}

    sub update_table_cell {shift->call("updateTableCell", @_)}

    sub update_table_row {shift->call("updateTableRow", @_)}

    sub add_table_column {shift->call("addTableColumn", @_)}

    sub update_entire_table {shift->call("updateEntireTable", @_)}

    sub column_exists {shift->call("columnExists", @_)}

    sub row_exists {shift->call("rowExists", @_)}

    sub headline_exists {shift->call("headlineExists", @_)}

    sub get_column_count {shift->call("getColumnCount")}

    sub get_row_count {shift->call("getRowCount")}

    sub get_headline_count {shift->call("getHeadlineCount")}

    sub get_column_names {shift->call("getColumnNames")}

    sub get_row_names {shift->call("getRowNames")}

    sub get_headline_names {shift->call("getHeadlineNames")}

    sub get_row_names_older_than {shift->call("getRowNamesOlderThan", @_)}

    ##################################
    #
    # package Geneos::API::Sampler
    #
    # Implements all sampler methods
    #
    ##################################

    package Geneos::API::Sampler;

    use base 'Geneos::Base';

    sub _init {
        my ($self, $api, $entity, $sampler) = @_;

        $self->{_api}     = $api;
        $self->{_entity}  = $entity;
        $self->{_sampler} = $sampler;

        return $self;
    }

    #
    # XML-RPC methodName for the sampler calls looks like this:
    # entity.sampler.action
    #

    sub _method {
        my $self = shift;
        join(".", $self->{_entity}, $self->{_sampler}, @_);
    }

    sub api {shift->{_api}}
    sub sampler {shift->{_sampler}}
    sub entity {shift->{_entity}}

    # -------------------------------------
    #

    sub get_view {
        my $self = shift;
        $self->_reset_error;
        Geneos::API::Sampler::View->new($self, @_)
    }

    #############
    # API calls #
    #############

    sub create_view {
        my $self = shift;
        $self->call("createView", @_) ? Geneos::API::Sampler::View->new($self, @_) : undef;
    }

    # -------------------------------------------------------
    # Checks whether a particular view exists in this sampler
    #
    # Returns true if the view exists, false otherwise
    #

    sub view_exists {
        my ($self, $view, $group) = @_;
        $self->call("viewExists", "${group}-${view}");
    }

    sub remove_view {shift->call("removeView", @_)}

    # -----------------------------------------------
    # Retrieves the value of a sampler parameter that
    # has been defined in the gateway configuration
    #
    # Returns the parameter text written in the gateway configuration
    #

    sub get_parameter {shift->call("getParameter", @_)}

    sub sign_on {shift->call("signOn", @_)}

    sub sign_off {shift->call("signOff")}

    sub heartbeat {shift->call("heartbeat")}

    ######################################
    #
    # package Geneos::API
    #
    # Implements the Geneos XML-RPC API
    #
    ######################################

    package Geneos::API;

    our $VERSION = '0.11';

    use base 'Geneos::Base';

    sub _init {
        my ($self, $url, $opts) = @_;

        # process the options
        $opts ||= {};
        $opts->{ua} ||= {};
        $opts->{api} ||= {};

        $self->{_xmlrpc} = Geneos::API::XMLRPC->new($url, $opts);
        $self->{_opts} = $opts;

        return $self;
    }

    sub _method {shift;@_}

    sub raise_error {shift->{_opts}{api}{raise_error}}

    sub print_error {shift->{_opts}{api}{print_error}}

    sub api{shift}

    sub request {shift->{_xmlrpc}->request(@_)}

    #############
    # API calls #
    #############

    # ------------------------
    # Creates a sampler object

    sub get_sampler {
        my $self = shift;
        $self->_reset_error;
        Geneos::API::Sampler->new($self,@_)
    }

    # ------------------------------------------------------------------------------
    # Checks whether a particular API or API-Streams sampler exists on this NetProbe
    #
    # Returns true if the sampler exists, false otherwise
    #

    sub sampler_exists {
        my ($self, $me, $sampler) = @_;
        $self->call("_netprobe.samplerExists", "$me.$sampler");
    }

    # ---------------------------------------------------------
    # Checks whether the Gateway is connected to this NetProbe
    #
    # Returns true if the Gateway is connected, false otherwise
    #

    sub gateway_connected {shift->call("_netprobe.gatewayConnected")}

    # ------------------------------------------------------
    # Adds the managed entity to the particular data section
    #
    # Returns true on success, false otherwise
    #

    sub add_managed_entity {shift->call("_gateway.addManagedEntity", @_)}

    # ------------------------------------------------------------------
    # Checks whether a particular Managed Entity exists on this NetProbe
    # containing any API or API-Streams samplers
    #
    # Returns true if the Managed Entity exists, false otherwise
    #

    sub managed_entity_exists {shift->call("_netprobe.managedEntityExists", @_)}

    1;

__END__
