#############################################################################
# output the a Graph::Easy as SVG (Scalable Vector Graphics)
#
# (c) by Tels 2004-2005.
#############################################################################

package Graph::Easy::As_svg;

use vars qw/$VERSION/;

$VERSION = '0.07';

use Graph::Easy;

#############################################################################
#############################################################################

package Graph::Easy;

sub EM () { 11; }

use strict;

my $devs = {
  'ah' => 
     "  <!-- open arrow head -->\n  <g id="
   . '"ah" stroke-linecap="round" stroke-width="1">' . "\n"
   . '   <line x1="-8" y1="-4" x2="1" y2="0" />'. "\n"
   . '   <line x1="1" y1="0" x2="-8" y2="4" />'. "\n"
   . "  </g>\n",

  'ahc' => 
     "  <!-- closed arrow head -->\n  <g id="
   . '"ahc" stroke-linecap="round" stroke-width="1">' . "\n"
   . '   <polygon points="-8,-4, 1,0, -8,4"/>'. "\n"
   . "  </g>\n",

  'ahf' => 
     "  <!-- filled arrow head -->\n  <g id="
   . '"ahf" stroke-linecap="round" stroke-width="1">' . "\n"
   . '   <polygon points="-8,-4, 1,0, -8,4"/>'. "\n"
   . "  </g>\n",

  # point-styles
  'diamond' =>
     "  <g id="
   . '"diamond" stroke-linecap="round">' . "\n"
   . '   <polygon points="0,-6, 6,0, 0,6, -6,0"/>'. "\n"
   . "  </g>\n",
  'circle' =>
     "  <g id="
   . '"circle">' . "\n"
   . '   <circle r="6" />'. "\n"
   . "  </g>\n",
  'star' =>
     "  <g id="
   . '"star" stroke-linecap="round">' . "\n"
   . '   <line x1="0" y1="-5" x2="0" y2="5" />'. "\n"
   . '   <line x1="-5" y1="0" x2="5" y2="0" />'. "\n"
   . '   <line x1="-3" y1="-3" x2="3" y2="3" />'. "\n"
   . '   <line x1="-3" y1="3" x2="3" y2="-3" />'. "\n"
   . "  </g>\n",
  'square' =>
     "  <g id="
   . '"square">' . "\n"
   . '   <rect width="10" height="10" />'. "\n"
   . "  </g>\n",
  'dot' =>
     "  <g id="
   . '"dot">' . "\n"
   . '   <circle r="1" />'. "\n"
   . "  </g>\n",
  'cross' =>
     "  <g id="
   . '"cross" stroke-linecap="round">' . "\n"
   . '   <line x1="0" y1="-5" x2="0" y2="5" />'. "\n"
   . '   <line x1="-5" y1="0" x2="5" y2="0" />'. "\n"
   . "  </g>\n",
  };

sub _as_svg
  {
  # convert the graph to SVG
  my ($self, $options) = @_;

  $self->layout() unless defined $self->{score};

  my ($rows,$cols,$max_x,$max_y,$cells) = $self->_prepare_layout('svg');

  my $txt;

  if ($options->{standalone})
    {
    $txt .= <<EOSVG
<?xml version="1.0" standalone="no"?>
<!DOCTYPE svg PUBLIC "-//W3C//DTD SVG 1.1//EN" "http://www.w3.org/Graphics/SVG/1.1/DTD/svg11.dtd">

EOSVG
;
    }

  # XXX TODO: that should use the padding/margin attribute from the graph
  my $xl = int(EM * 1); my $yl = int(EM * 1);
  my $xr = int(EM * 1); my $yr = int(EM * 1);

  my $mx = $max_x + $xl + $xr;
  my $my = $max_y + $yl + $yr;

  $txt .=
     "<svg viewBox=\"0 0 $mx $my\" version=\"1.1\">"
    ."\n<!-- Generated by Graph::Easy v$Graph::Easy::VERSION" .
	    " at " . scalar localtime() . " -->\n\n";

  $txt .= " <defs>\n##devs##";

  # which attributes must be output as what name:
  my $mutator = {
    background => 'fill',
    'text-align' => 'text-anchor',
    'color' => 'stroke',
    };
  my $skip = qr/^(
   label|
   linkbase|
   (auto)?(link|title)|
   nodeclass|
   border-style|
   padding.*|
   margin.*|
   line-height|
   letter-spacing|
   font-family|
   border|
   width|
   color|
   title|
   text-align|
   border-color|
   shape|
   rows|
   cols|
   size|
   label-color|
   arrow-style
   )\z/x;

  my $overlay = {
    node => {
      "text-anchor" => 'middle',
    },
  };
  # generate the class attributes first
  my $style = $self->_class_styles( $skip, $mutator, '', '  ', $overlay);

  ## output groups first, with their nodes
  #foreach my $gn (sort keys %{$self->{groups}})
  #  {
  #  my $group = $self->{groups}->{$gn};
  #  $txt .= $group->as_txt();		# marks nodes as processed if nec.
  #  $count++;
  #  }

  $txt .= 
    "\n  <!-- class definitions -->\n"
   ."  <style type=\"text/css\"><![CDATA[\n$style  "
   # include a pseudo-class ".text" to shorten output
   .".text {\n   text-anchor: middle;\n   font-size: 14;\n  }\n"
   ."  ]]></style>\n"
    if $style ne '';
 
  $txt .=" </defs>\n\n";

  # XXX: output cells that belong to one edge/node in order, e.g.
#  for my $n (@nodes)
#     {
#     # output node      
#    for my $e (@edges)
#       {
#       # output edges leaving this node  
#       }
#     }

  # now output all the occupied cells
  foreach my $n (@$cells)
    {
    # exclude filler cells
    if ($n->{minw} != 0 && $n->{minh} != 0)
      {
      # get position from cell
      my $x = $cols->{ $n->{x} } + $xl;
      my $y = $rows->{ $n->{y} } + $yl;

      my $class = $n->{class}; $class =~ s/\./-/;	# node.city => node-city
      my $indent = '  ';
      $txt .= "<g class=\"$class\">\n";
      $txt .= $n->as_svg($x,$y,'  ');	# output cell, indented
      $txt =~ s/\n\z/<\/g>\n\n/;
      }
    }

  # include the used definitions into <devs>
  my $d = '';
  #for my $key (keys %$used)
  for my $key (qw/ah ahc ahf circle square dot cross star diamond/)
    {
    $d .= $devs->{$key};
    }
  $txt =~ s/##devs##/$d/;

  $txt . "</svg>";			# finish
  }

=pod

=head1 NAME

Graph::Easy::As_svg - Output a Graph::Easy as Scalable Vector Graphics (SVG)

=head1 SYNOPSIS

	use Graph::Easy;
	
	my $graph = Graph::Easy->new();

	$graph->add_edge ('Bonn', 'Berlin');

	print $graph->as_svg_file();


=head1 DESCRIPTION

C<Graph::Easy::As_svg> contains just the code for converting a
L<Graph::Easy|Graph::Easy> object to a SVG text.

=head1 EXPORT

Exports nothing.

=head1 SEE ALSO

L<Graph::Easy>.

=head1 AUTHOR

Copyright (C) 2004 - 2005 by Tels L<http://bloodgate.com>

See the LICENSE file for information.

=cut

package Graph::Easy::Node;

sub as_svg
  {
  # output a node as SVG
  my ($self,$x,$y,$indent) = @_;

  my $name = $self->{att}->{label}; $name = $self->{name} unless defined $name;

  my $em = Graph::Easy::EM;		# multiplication factor chars * em = units (pixels)

  # the attributes of the element we will finally output
  my $att = $self->_svg_attributes($x,$y);
  
  my $shape = $att->{shape};				# rect, circle etc
  delete $att->{shape};

  return '' if $shape eq 'invisible';

  # set a potential title
  my $title = $self->title();
  $att->{title} = $title if $title ne '';

  my $s = $self->attribute('shape') || 'rectangle';
  my $svg = "$indent<!-- $name, $s -->\n";

  if ($shape eq 'point')
    {
    # include the point-style
    my $s = $self->attribute('point-style') || 'star';
    my $a = { };
    for my $key (keys %$att)
      {
      $a->{$key} = $att->{$key};
      }
    $a->{stroke} = $self->attribute('border-color') || 'black';
    $a->{fill} = $a->{stroke} if $s eq 'dot';

    my $att_txt = $self->_svg_attributes_as_txt($a);
    my $xt = int($x + $self->{w} / 2);
    my $yt = int($y + $self->{h} / 2);
    $yt -= 5 if $s eq 'square';
    $xt -= 5 if $s eq 'square';
    $svg .= "$indent<use$att_txt xlink:href=\"#$s\" x=\"$xt\" y=\"$yt\"/>\n\n";
    }
  else
    {
    my $att_txt = $self->_svg_attributes_as_txt($att);

    $svg .= "$indent<$shape$att_txt />\n";

    ###########################################################################
    # include the label/name/text

    my $label = $self->label();

    my $xt = int($x + $em * 0.05 + $self->{w} / 2);
    my $yt = int($y + $self->{h} / 2 + $em / 2 - $em * 0.1);

    my $color = $self->attribute('color') || 'black';

    $svg .= "$indent<text class=\"text\" x=\"$xt\" y=\"$yt\" fill=\"$color\">$label</text>\n\n";
    }

  $svg;
  }

my $strokes = {
  'dashed' => '4, 2',
  'dotted' => '2, 2',
  'dot-dash' => '2, 2, 4, 2',
  'dot-dot-dash' => '2, 2, 2, 2, 4, 2',
  'double-dash' => '4, 2',
  };

sub _svg_attributes
  {
  # Return a hash with attributes for the node, like "x => 1, y => 1, w => 1, h => 1"
  # Especially usefull for shapes other than boxes.
  my ($self,$x,$y) = @_;

  my $att = {};

  my $shape = $self->shape();

  my $w2 = $self->{w} / 2;
  my $h2 = $self->{h} / 2;

  my $x2 = $x + $self->{w};
  my $y2 = $y + $self->{h};
  if ($shape eq 'point')
    {
    }
  elsif ($shape eq 'circle')
    {
    $att->{cx} = $x + $w2;
    $att->{cy} = $y + $h2;
    $att->{r} = $self->{minw} > $self->{minh} ? $self->{minw} : $self->{minh};
    $att->{r} /= 2;
    }
  elsif ($shape eq 'diamond')
    {
    my $x1 = $x + $w2;
    my $y1 = $y + $h2;

    $shape = "polygon points=\"$x,$y1, $x1,$y, $x2,$y1, $x1,$y2\" stroke-linecap=\"round\"";
    }
  elsif ($shape eq 'house')
    {
    my $x1 = $x + $w2;;
    my $y1 = sprintf("%0.4f", $y + $self->{h} * 0.33);

    $shape = "polygon points=\"$x1,$y, $x2,$y1, $x2,$y2, $x,$y2, $x,$y1\"";
    }
  elsif ($shape eq 'triangle')
    {
    my $x1 = $x + $w2;;
    my $y1 = sprintf("%0.4f", $y + $self->{h} * 0.33);

    $shape = "polygon points=\"$x1,$y, $x2,$y2, $x,$y2\"";
    }
  elsif ($shape eq 'invtriangle')
    {
    my $x1 = $x + $w2;;
    my $y1 = sprintf("%0.4f", $y + $self->{h} * 0.33);

    $shape = "polygon points=\"$x,$y, $x2,$y, $x1,$y2\"";
    }
  elsif ($shape eq 'ellipse')
    {
    $att->{cx} = $x + $w2;
    $att->{cy} = $y + $h2;
    $att->{rx} = $w2;
    $att->{ry} = $h2;
    }
  else
    {
    if ($shape eq 'rounded')
      {
      # round corners by 10%
      $att->{ry} = '10%';
      $att->{rx} = '10%';
      $shape = 'rect';
      }
    $att->{x} = $x;
    $att->{y} = $y;
    $att->{width} = $self->{w};
    $att->{height} = $self->{h};

    # XXX TODO: other shapes than rectangles like invhouse etc
    }
  $att->{shape} = $shape;

  my $border_style = $self->attribute('border-style') || 'solid';
  my $border_color = $self->attribute('border-color') || 'black';
  my $border_width = $self->attribute('border-width') || '1';

  $att->{'stroke-width'} = $border_width if $border_width ne '1';
  $att->{stroke} = $border_color;

  # black is the default, so save space
#  delete $att->{stroke} if $border_color eq 'black';

  if ($border_style !~ /^(none|solid)/)
    {
    $att->{'stroke-dasharray'} = $strokes->{$border_style}
     if exists $strokes->{$border_style};
    }

  $att->{'stroke-width'} = 3 if $border_style eq 'bold';
  if ($border_style eq 'none')
    {
    delete $att->{'stroke-width'};
    delete $att->{stroke};
    }
  my $background = $self->attribute('fill') || '';
  $att->{fill} = $background if $background ne '';

  $att;
  }

sub _svg_attributes_as_txt
  {
  # convert hash with attributes to text to be included in SVG tag
  my ($self, $att) = @_;

  my $att_line = '';				# attributes as text (cur line)
  my $att_txt = '';				# attributes as text (all)
  foreach my $e (sort keys %$att)
    {
    # skip these
    next if $e =~ /^(arrow-style|label-color|rows|cols|size|offset|origin)\z/;

    $att_line .= " $e=\"$att->{$e}\"";
    if (length($att_line) > 75)
      {
      $att_txt .= "$att_line\n  "; $att_line = '';
      }
    }
  $att_txt .= $att_line;
  $att_txt =~ s/\n\z//;		# avoid a "  >" on last line
  $att_txt;
  }
 
sub _correct_size_svg
  {
  # Correct {w} and {h} after parsing.
  my $self = shift;

  my $em = Graph::Easy::EM;		# multiplication factor chars * em = units (pixels)

  return if defined $self->{w};

  my $shape = $self->shape();
  if ($shape eq 'point')
    {
    $self->{w} = $em * 3;
    $self->{h} = $em * 3;
    return;
    }

  my $txt = $self->label();

  my ($w,$h) = $self->dimensions();
  # XXX TODO: that should use a changable padding factor (like "0.2 em" or "4")
  $self->{w} = int($w * $em + 0.2 * $em);
  $self->{h} = int($h * $em + 0.8 * $em);
  my $border = $self->attribute('border-style') || 'none';
  if ($border ne 'none')
    {
    # XXX TODO: that should use the border width (like "1 pixel" * 2)
    $self->{w} += 2;
    $self->{h} += 2;
    }
 
  # for triangle or invtriangle:
  $self->{w} *= 2 if $shape =~ /triangle/;

  if ($shape =~ /^(diamond|circle|triangle)$/)
    {
    # the min size is either w or h, depending on which is bigger
    my $max = $self->{w}; $max = $self->{h} if $self->{h} > $max;
    $self->{h} = $max;
    $self->{w} = $max;
    }
  }
 
1;

package Graph::Easy::Edge::Cell;

#############################################################################
#############################################################################
# Line drawing code for edges

# define the line lengths for the different edge types

sub LINE_HOR () { 0x0; }
sub LINE_VER () { 0x1; }

sub LINE_MASK () { 0x0F; }
sub LINE_DOUBLE () { 0x10; }

  # edge type       line type  spacing left/top
  #				    spacing right/bottom

my $draw_lines = {

  EDGE_N_W_S()	=> [ LINE_HOR, 0.2, 0.2 ],			# v--+  loop, northwards
  EDGE_S_W_N()	=> [ LINE_HOR, 0.2, 0.2 ],			# ^--+  loop, southwards
  EDGE_E_S_W()	=> [ LINE_VER, 0.2, 0.2 ],			# [_    loop, westwards
  EDGE_W_S_E()	=> [ LINE_VER, 0.2, 0.2 ],			# _]    loop, eastwards

  EDGE_VER()	=> [ LINE_VER, 0, 0 ],				# |	vertical line
  EDGE_HOR()	=> [ LINE_HOR, 0, 0 ],				# --	vertical line

  EDGE_CROSS()	=> [ LINE_HOR, 0, 0, LINE_VER, 0, 0  ],		# + crossing

  EDGE_S_E()	=> [ LINE_VER,   0.5, 0, LINE_HOR, 0.5, 0 ],	# |_    corner (N to E)
  EDGE_N_W()	=> [ LINE_VER,   0, 0.5, LINE_HOR, 0, 0.5 ],	# _|    corner (N to W)
  EDGE_N_E()	=> [ LINE_VER,   0, 0.5, LINE_HOR, 0.5, 0 ],	# ,-    corner (S to E)
  EDGE_S_W()	=> [ LINE_VER,   0.5, 0, LINE_HOR, 0, 0.5 ],	# -,    corner (S to W)

  EDGE_S_E_W	=> [ LINE_HOR, 0, 0, LINE_VER, 0.5, 0 ],	# -,-   three-sided corner (S to W/E)
  EDGE_N_E_W	=> [ LINE_HOR, 0, 0, LINE_VER, 0, 0.5 ],	# -'-   three-sided corner (N to W/E)
  EDGE_E_N_S	=> [ LINE_VER, 0, 0, LINE_HOR, 0.5, 0 ],	#  |-   three-sided corner (E to S/N)
  EDGE_W_N_S	=> [ LINE_VER, 0, 0, LINE_HOR, 0, 0.5 ],	# -|    three-sided corner (W to S/N)
 };

sub _svg_arrow
  {
  my ($self, $att, $x, $y, $type, $dis, $indent) = @_;

  my $w = $self->{w};
  my $h = $self->{h};

  my ($x1,$x2, $y1,$y2);

  # For the things to be "used" define these attributes, so if they
  # match, we can skip them, generating shorter output:

  my $DEF = { 
    "stroke-linecap" => 'round',
    };

  my $arrow_style = $att->{"arrow-style"} || '';
  my $class = "ah" . substr($arrow_style,0,1);

  my $a = {};
  for my $key (keys %$att)
    {
    next if $key =~ /^(stroke-dasharray|arrow-style)\z/;
    $a->{$key} = $att->{$key}
     unless exists $DEF->{$key} && $DEF->{$key} eq $att->{$key};
    }
  if ($arrow_style eq 'closed')
    {
    $a->{fill} = $self->attribute('background');
    $a->{fill} = $self->{graph}->attribute('graph', 'background') if $a->{fill} eq 'inherit';
    $a->{fill} = 'white' if $a->{fill} eq 'inherit';
    }
  elsif ($arrow_style eq 'filled')
    {
    $a->{fill} = $self->attribute('fill');
    }

  my $att_txt = $self->_svg_attributes_as_txt($a);

  my $svg = "$indent<use$att_txt xlink:href=\"#$class\" ";

  if ($type == EDGE_END_N)
    {
    $x1 = $x + $w / 2;		# the arrow tip
    $dis *= $h if $dis < 1;
    $y1 = $y + $dis;

    $svg .= "transform=\"translate($x1 $y1) rotate(-90)\"";
    }
  elsif ($type == EDGE_END_S)
    {
    $x1 = $x + $w / 2;		# the arrow tip
    $dis *= $h if $dis < 1;
    $y1 = $y + $h - $dis;

    $svg .= "transform=\"translate($x1 $y1) rotate(90)\"";
    }
  elsif ($type == EDGE_END_W)
    {
    $dis *= $w if $dis < 1;
    $x1 = $x + $dis;		# the arrow tip
    $y1 = $y + $h / 2;

    $svg .= "transform=\"translate($x1 $y1) rotate(180)\"";
    }
  else	# $type == EDGE_END_E
    {
    $dis *= $w if $dis < 1;
    $x1 = $x + $w - $dis;		# the arrow tip
    $y1 = $y + $h / 2;
    $svg .= "x=\"$x1\" y=\"$y1\"";
    }
  $svg .= "/>\n";

  $svg;
  }

sub _svg_line_straight
  {
  # Generate SVG tags for a vertical/horizontal line, bounded by (x,y), (x+w,y+h).
  # $l and $r shorten the line left/right, or top/bottom, respectively. If $l/$r < 1,
  # in % (aka $l * w), otherwise in units.
  # "$s" means there is a starting point, so the line needs to be shorter. Likewise
  # for "$e", only on the "other" side. 
  # VER: s = north, e = south, HOR: s = left, e= right
  my ($self, $att, $x, $y, $type, $l, $r, $indent, $s, $e) = @_;

  my $att_txt = $self->_svg_attributes_as_txt($att);

  my $w = $self->{w};
  my $h = $self->{h};

  my ($x1,$x2, $y1,$y2, $x3, $x4, $y3, $y4);

  my $ltype = $type & LINE_MASK;
  if ($ltype == LINE_HOR)
    {
    $l += $s if $s;
    $r += $e if $e;
    $l *= $w if $l < 1;
    $r *= $w if $r < 1;
    $x1 = $x + $l; $x2 = $x + $w - $r;
    $y1 = $y + $h / 2; $y2 = $y1;
    if (($type & LINE_DOUBLE) != 0)
      {
      $y1--; $y2--; $y3 = $y1 + 2; $y4 = $y3;
      $x1 += 1; $x2 -= 2;
      $x3 = $x1; $x4 = $x2;
      }
    }
  else
    {
    $l += $s if $s;
    $r += $e if $e;
    $l *= $h if $l < 1;
    $r *= $h if $r < 1;
    $x1 = $x + $w / 2; $x2 = $x1;
    $y1 = $y + $l; $y2 = $y + $h - $r;
    if (($type & LINE_DOUBLE) != 0)
      {
      $x1--; $x2--; $x3 = $x1 + 2; $x4 = $x3;
      $y1 += 1; $y2 -= 1;
      $y3 = $y1; $y4 = $y2;
      }
    }

  my $txt = 
  "$indent<line x1=\"$x1\" y1=\"$y1\" x2=\"$x2\" y2=\"$y2\"$att_txt />\n";

  # for a double line
  if (defined $x3)
    {
    $txt .=
     "$indent<line x1=\"$x3\" y1=\"$y3\" x2=\"$x4\" y2=\"$y4\"$att_txt />\n";
    }
  $txt;
  }

#############################################################################
#############################################################################

my $dimensions = {
  EDGE_VER()	=> [ 1, 4 ],	# |
  EDGE_HOR()	=> [ 4, 1 ],	# -

  EDGE_CROSS()	=> [ 4, 4 ],	# +	crossing

  EDGE_N_E()	=> [ 4, 4 ],	# |_    corner (N to E)
  EDGE_N_W()	=> [ 4, 4 ],	# _|    corner (N to W)
  EDGE_S_E()	=> [ 4, 4 ],	# ,-    corner (S to E)
  EDGE_S_W()	=> [ 4, 4 ],	# -,    corner (S to W)

  EDGE_S_E_W	=> [ 4, 2 ],	# -,-   three-sided corner (S to W/E)
  EDGE_N_E_W	=> [ 4, 2 ],	# -'-   three-sided corner (N to W/E)
  EDGE_E_N_S	=> [ 2, 4 ],	#  |-   three-sided corner (E to S/N)
  EDGE_W_N_S	=> [ 2, 4 ], 	# -|    three-sided corner (W to S/N)
 };

sub _correct_size_svg
  {
  my ($self,$format) = @_;

  my $em = Graph::Easy::EM;		# multiplication factor chars * em = units (pixels)

  return if defined $self->{w};

  #my $border = $self->{edge}->attribute('border-style') || 'none';

  # set the minimum width/height
  my $type = $self->{type} & EDGE_TYPE_MASK();
  my $dim = $dimensions->{$type} || [ 2, 2 ];
  ($self->{w}, $self->{h}) = ($dim->[0] * $em, $dim->[1] * $em);
  }

#############################################################################
#############################################################################

sub _svg_attributes
  {
  # Return a hash with attributes for the cell.
  my ($self) = @_;

  my $att = {};

  $att->{stroke} = $self->attribute('color') || 'black';
  $att->{'stroke-width'} = 1;

  my $style = $self->{style};
  if ($style ne 'solid')				# solid line
    {
    $att->{'stroke-dasharray'} = $strokes->{$style}
     if exists $strokes->{$style};
    }

  $att->{'stroke-width'} = 3 if $style eq 'bold';

  $att->{'arrow-style'} = $self->attribute('arrow-style') || '';
  $att;
  }

sub as_svg
  {
  my ($self,$x,$y, $indent) = @_;

  my $em = Graph::Easy::EM;		# multiplication factor chars * em = units (pixels)

  # the attributes of the element we will finally output
  my $att = $self->_svg_attributes();
  
  # set a potential title
  my $title = $self->title();
  $att->{title} = $title if $title ne '';

  my $att_txt = $self->_svg_attributes_as_txt($att);

  my $type = $self->{type} & EDGE_TYPE_MASK();
  my $end = $self->{type} & EDGE_END_MASK();
  my $start = $self->{type} & EDGE_START_MASK();

  my $edge = $self->{edge};
  my $from = $edge->{from}->{name};
  my $to = $edge->{to}->{name};
  my $svg = "$indent<!-- " . edge_type($type) . ", from $from to $to -->\n";
 
  # for each line, include a SVG tag

  my $lines = [ @{$draw_lines->{$type}} ];	# make copy

  while (@$lines > 0)
    {
    my ($type, $l, $r) = splice (@$lines, 0, 3);

    # start/end points
    my ($s,$e) = (undef,undef);

    # VER: s = north, e = south, HOR: s = left, e= right
    if ($type == LINE_VER)
      {
      $e = 0.1 if ($end & EDGE_END_S);
      $s = 0.1 if ($end & EDGE_END_N);
      $e = 0.1 if ($start & EDGE_START_S);
      $s = 0.1 if ($start & EDGE_START_N);
      }
    else # $type == LINE_HOR
      {
      $e = 0.1 if ($end & EDGE_END_E);
      $s = 0.1 if ($end & EDGE_END_W);
      $e = 0.1 if ($start & EDGE_START_E);
      $s = 0.1 if ($start & EDGE_START_W);
      }

    $type += LINE_DOUBLE if $self->{style} =~ /^double/;
    $svg .= $self->_svg_line_straight($att, $x, $y, $type, $l, $r, $indent, $s, $e);
    }

  my $arrow = $self->{type} & EDGE_END_MASK;
  # depending on end points, add the arrow

  $svg .= $self->_svg_arrow($att, $x, $y, $arrow, 0.1, $indent) unless $arrow == 0;

  ###########################################################################
  # include the label/name/text if we are the label cell
 
  if (($self->{type} & EDGE_LABEL_CELL()))
    {
    my $label = $self->label(); $label = '' unless defined $label;

    if ($label ne '')
      {
      my $xt = int($x + $em * 0.2 + $self->{w} / 4);
      my $yt = int($y + $em + $em * 0.2);
      # XXX TODO: use attribute "label-color"
      my $color = $att->{color} || 'black';

      $svg .= "$indent<text x=\"$xt\" y=\"$yt\" style=\"font-size:$em\" fill=\"$color\">$label</text>\n";
      }
    }

  $svg. "\n";
  }

