#!perl
#
# Documentation, copyright and license is at the end of this file.
#

package  Archive::TarGzip;

use 5.001;
use strict;
use warnings;
use warnings::register;

use vars qw($VERSION $DATE $FILE);
$VERSION = '0.01';
$DATE = '2003/08/04';
$FILE = __FILE__;

use vars qw(@ISA @EXPORT_OK);
require Exporter;
use Archive::Tar;
@ISA= qw(Exporter Archive::Tar);
@EXPORT_OK = qw(tar untar);

# use SelfLoader;

use Cwd;

######
#
#
sub new
{
     my $class = shift @_;
     $class = ref($class) if( ref($class) );

     ######
     # Parse the arguments
     #
     my ($tar_file, $compress, $args_p);
     if( ref($_[-1]) eq 'ARRAY') {
          $args_p = $_[-1];
          my %args = @{$args_p};
          $args_p = \%args;
     }
     elsif (ref($_[-1]) eq 'HASH') {
          $args_p = pop @_;
     }
     else {
          $tar_file = shift @_;
          $compress = shift @_;
          my %args = @_;
          $args_p = \%args;
     }

     #######
     # tar object
     #
     my $self = $class->Archive::Tar::new($tar_file, $compress);
     $self->{TarGzip} = $args_p;
     $self;
}


######
#
#
sub taropen 
{
     my $self = shift @_;

     ######
     # Parse the arguments
     #
     my ($tar_file, $compress, $args_p);
     if( ref($_[-1]) eq 'ARRAY') {
          $args_p = $_[-1];
          my %args = @{$args_p};
          $args_p = \%args;
     }
     elsif (ref($_[-1]) eq 'HASH') {
          $args_p = pop @_;
     }
     else {
          $tar_file = shift @_;
          $compress = shift @_;
          my %args = @_;
          $args_p = \%args;
     }

     my $key;
     foreach $key (%$args_p) {
         $self->{TarGzip}->{$key} = $args_p->{$key};
     }

     $self->{TarGzip}->{compress} = $compress if $compress;

     $tar_file = $self->{TarGzip}->{tar_file} unless $tar_file;
     unless( $tar_file ) {
         warn( "No tar file\n");
         return undef;
     }
     $self->{TarGzip}->{tar_file} = $tar_file;

     #######
     # Try to find a file by adding extension for the tar and gz
     #
     my $flag = $self->{TarGzip}->{tar_flag};
     if( $flag eq '>' ) {
         if( $self->{TarGzip}->{compress} ) {
             $tar_file .= '.gz' if substr($tar_file, -4, 4) eq '.tar';
             $tar_file .= '.tar.gz' unless substr($tar_file, -7, 7) eq '.tar.gz';
         }
         else {
             $tar_file .= '.tar' unless substr($tar_file, -4, 4) eq '.tar';
         }
     }
     else {
         unless( -e $tar_file ) {
             if( $self->{TarGzip}->{compress} ) {
                 $tar_file .= '.gz' if substr($tar_file, -4, 4) eq '.tar';
                 $tar_file .= '.tar.gz' unless substr($tar_file, -7, 7) eq '.tar.gz';
             }
             else {
                 $tar_file .= '.tar' unless substr($tar_file, -4, 4) eq '.tar';
             }
             $self->{TarGzip}->{tar_file} = $tar_file;
             unless( -e $tar_file ) {
                 warn("Cannot find $tar_file\n");
                 return undef;
             }
         }
     }

     ########
     # Open files
     # 
     if( $self->{TarGzip}->{compress} ) {
         require Compress::Zlib;
         Compress::Zlib->import qw(&gzopen $gzerrno Z_STREAM_END);
         $self->{TarGzip}->{handle} = gzopen($tar_file, ($flag eq '>' ) ? 'wb' : 'rb');
     }
     else {

         ######
         # Open tar file
         #
         unless (open TAR, "$flag $tar_file") {
             warn( "Cannot open $flag $tar_file\n");
             return undef;
         }
         binmode TAR;
         $self->{TarGzip}->{handle} = \*TAR;
     }
     $self->{TarGzip}->{handle};
}

#######
# Close the TAR file
#
sub tarclose
{

     my ($self) = @_;
     my $handle = $self->{TarGzip}->{handle};
     return undef unless $handle;
     my $blank_header = "\0" x 1024;  
     my $write_flag = $self->{TarGzip}->{tar_flag} eq '>';
     $self->{TarGzip}->{handle} = '';
     if( $self->{TarGzip}->{compress} ) {
          $handle->gzwrite($blank_header) if $write_flag;
          $handle->gzclose( );
     }
     else {
          print $blank_header if $write_flag; 
          close $handle;
     }

}

# 1

# __DATA__


#######
# add a file to the TAR file
#
sub taradd
{
     my ($self, $file_name, $file_contents) = @_;
     my $tar = $self->{tar};
     unless( defined $file_contents ) {
         unless (open FILE, $file_name) {
             warn "Cannot open $file_name\n";
             return undef;
         }
         binmode FILE;
         $file_contents = join '', <FILE>;
         close FILE;
     }
     return undef unless $self->add_data($file_name, $file_contents);
     $self->tarwrite();

     1
}


#######
# 
#
sub tarwrite
{
     my ($self) = @_;
     my $tar_content = &Archive::Tar::format_tar_entry(shift @{$self->{'_data'}});
     my $handle = $self->{TarGzip}->{handle};
     if( $self->{TarGzip}->{compress} ) {
          my $gz = $handle;
          unless ($gz->gzwrite($tar_content) ) {
              warn "error gzwrite $self->{TarGzip}->{tar_file}.gz: $Compress::Zlib::gzerrno\n" ;
              $gz->gzclose( );
              return undef;
          }
     }
     else {
          unless( print $handle $tar_content ) {
              warn "print error\n" ;
              close $self->{TarGzip}->{handle};
              return undef;
          }
     }

     1
}



######
# This is taken directly from big loop in Archive::Tar::read_tar
#
#
sub parse_header
{
     return undef unless(defined($_[0]));      
     shift @_ if $_[0] eq 'Archive::TarGzip' || ref($_[0]);  # drop self on object call 

     unless(@_) {
         warn "No arguments.\n";
         return undef;
     }

     my ($header) = @_;

     # Apparently this should really be two blocks of 512 zeroes,
     # but GNU tar sometimes gets it wrong. See comment in the
     # source code (tar.c) to GNU cpio.
     return { end_of_tar => 1 } if $header eq "\0" x 512; # End of tar file
        
     my ($name,		# string
	 $mode,		# octal number
	 $uid,		# octal number
	 $gid,		# octal number
	 $size,		# octal number
	 $mtime,		# octal number
	 $chksum,		# octal number
	 $typeflag,		# character
	 $linkname,		# string
	 $magic,		# string
	 $version,		# two bytes
	 $uname,		# string
	 $gname,		# string
	 $devmajor,		# octal number
	 $devminor,		# octal number
	 $prefix) = unpack($Archive::Tar::tar_unpack_header, $header);
	
     $mode = oct $mode;
     $uid = oct $uid;
     $gid = oct $gid;
     $size = oct $size;
     $mtime = oct $mtime;
     $chksum = oct $chksum;
     $devmajor = oct $devmajor;
     $devminor = oct $devminor;
     $name = $prefix."/".$name if $prefix;
     $prefix = "";
     # some broken tar-s don't set the typeflag for directories
     # so we ass_u_me a directory if the name ends in slash
     $typeflag = 5 if $name =~ m|/$| and not $typeflag;
		
     my $error = '';
     substr($header,148,8) = "        ";
     $error .= "$name: checksum error.\n" unless (unpack("%16C*",$header) == $chksum);
     $error .= "$name: wrong header length\n" unless( $Archive::Tar::tar_header_length == length($header));

     my $end_of_tar = 0;
     # Guard against tarfiles with garbage at the end
     $end_of_tar = 1 if $name eq '';

     warn( $error ) if $error;

     return {
         name => $name,
	 mode => $mode,
	 uid => $uid,
	 gid => $gid,
	 size => $size,
	 mtime => $mtime,
	 chksum => $chksum,
	 typeflag => $typeflag,
	 linkname => $linkname,
	 magic => $magic,
	 version => $version,
	 uname => $uname,
	 gname => $gname,
	 devmajor => $devmajor,
	 devminor => $devminor,
	 prefix => $prefix,
         error => $error,
         end_of_tar => $end_of_tar,
         header_only => 0,
         skip_file => 0,
         data => ''};
}


#####
#
#
sub tarread
{

     my $self = shift @_;

     #######
     # Parse out any options
     #
     my $options_p;
     if( ref($_[-1]) eq 'ARRAY') {
          $options_p = $_[-1];
          my %options = @{$options_p};
          $options_p = \%options;
     }
     elsif (ref($_[-1]) eq 'HASH') {
          $options_p = pop @_;
     }

     foreach my $key (keys %$options_p) {
         $self->{TarGzip}->{$key} = $options_p->{$key}; 
     }

     ####### 
     # Add any @files to the extract selection hash
     # 
     my $extract_p = $self->{TarGzip}->{extract_files};
     FileList2Sel(@_, $extract_p);
     my $extract_count = keys %$extract_p;

     ########
     # Read header
     #
     my $data;
     return undef unless $self->target(\$data, $Archive::Tar::tar_header_length );

     ########
     # Parse header
     #
     my $file_position = undef;
     $file_position = tell $self->{TarGzip}->{handle} unless( $self->{TarGzip}->{compress} );
     my $header = parse_header( $data );
     return undef if $header->{end_of_tar};
     $header->{file_position} = $file_position if $file_position;

     #######
     # Process header_only option
     # 
     $header->{data} = '';
     $header->{header_only} = 0;
     $header->{skip_file} = 0;
     my $buffer_p = \$header->{data};
     if ($self->{TarGzip}->{header_only}) {
         $buffer_p = undef;
         $header->{header_only} = 1;
     }

     #######
     # Process extract file list
     # 
     if( $extract_count && !$extract_p->{$header->{name}} ) {
         $buffer_p = undef;
         $header->{skip_file} = 1;
         $header->{name} = '';
     }

     #######
     # Process exclude file list
     # 
     my $exclude_p = $self->{TarGzip}->{extract_files};
     my $exclude_count = scalar(keys %$exclude_p);
     if( $exclude_count && $exclude_p->{$header->{name}} ) {
         $buffer_p = undef;
         $header->{skip_file} = 1;
         $header->{name} = '';
     }
    
     #######
     # Read file contents
     # 
     my $size = $header->{size};
     return $header unless $size;
     return undef unless $self->target($buffer_p, $header->{size});

     ######
     # Trash bye padding to put on 512 byte boundary
     #
     $size = ($size % 512);
     return $header unless $size;
     $self->target(undef, 512 - $size);
     $header;

}



#####
#
#
sub target
{
     my ($self, $buffer_p, $size) = @_;
     my $handle = $self->{TarGzip}->{handle};
     $handle = $self->{TarGzip}->{handle};
     my $bytes;
     if( $self->{TarGzip}->{compress} ) {
         my $trash;
         unless( $buffer_p ) {
             $buffer_p = \$trash;
         }
         $$buffer_p = '';
         $bytes = $handle->gzread($$buffer_p, $size);
         if( $bytes == -1) {
              warn "error gzwrite $self->{TarGzip}->{tar_file}\.gz: $Archive::Zlib::gzerrno\n" ;
              $handle->gzclose( );
              return undef;
         }
         return undef unless $bytes == $size || !$buffer_p;
     }
     elsif( $buffer_p ) {
         $$buffer_p = '';
         $bytes = read( $handle, $$buffer_p, $size);
         return undef unless $bytes == $size;
     }
     else {
         seek $handle, $size, 1;
     }

     1
}





#######
# Store a number of files in one archive file in the tar format
#
sub tar
{
     unless(defined($_[0])) {
         warn( "No inputs.\n");
         return undef;
     }
     shift @_ if $_[0] eq 'Archive::TarGzip' || ref($_[0]);  # drop self on object call 

     my (@files) = @_;

     unless(@files) {
         warn "No files.\n";
         return undef;
     }

     ######
     # Check to see if first argument is an option
     #
     my $options_p;
     if( ref($files[-1]) eq 'ARRAY') {
          my %options = @{pop @files};
          $options_p = \%options;
     }
     elsif (ref($files[-1]) eq 'HASH') {
          $options_p = pop @files;
     }

     unless(@files) {
         warn "No files.\n";
         return undef;
     }

     ########
     # Create a new tar file
     #
     $options_p->{tar_flag} = '>';
     my $tar;  
     return undef unless $tar = new Archive::TarGzip($options_p);
     return undef unless $tar->taropen();

     #####
     # Bring in some needed program modules
     #   
     require File::Spec;
     require File::AnySpec;

     ######
     # Process options
     #
     $options_p->{dest_dir} = '' unless $options_p->{dest_dir};
     my $dest_dir = $options_p->{dest_dir};
     $dest_dir = File::AnySpec->os2fspec('Unix',$dest_dir,'nofile');
     $dest_dir .= '/' unless $dest_dir && substr($dest_dir, -1, 1) eq '/';

     $options_p->{src_dir} = '' unless $options_p->{src_dir};
     my $src_dir = $options_p->{src_dir};

     #####
     # change to the source directory
     #
     my $restore_dir = cwd();
     chdir $src_dir if $src_dir;
    
     my $contents;     
     if( $dest_dir ) {
         unless ($tar->taradd($dest_dir, '')) {
             chdir $restore_dir;
             return undef;
         }
     }

     my (%dirs, $dir_name, $file_dir, @file_dirs);
     my ($file_name, $file_contents);

     foreach $file_name (@files) {
 
         #######
         # Add directory path to the archive file
         #
         (undef, $file_dir) = File::Spec->splitpath( $file_name ) ;
         @file_dirs = File::Spec->splitdir($file_dir);
         $dir_name = $dest_dir;
         foreach $file_dir (@file_dirs) {
             $dir_name = File::Spec::Unix->catdir( $dir_name, $file_dir) if $dir_name;
             unless( $dirs{$dir_name} ) {
                 $dirs{$dir_name} = 1;
                 $dir_name .= '/';
                 unless ($tar->taradd($dir_name, '')) { # add a directory name, no content
                     chdir $restore_dir;
                     return undef;
                 }
             }
         }

         ########
         # Read the contents of the file
         #
         unless( open( CONTENT, "< $file_name") ) {
             $file_name = File::Spec->rel2abs($file_name);
             warn "Cannot read contents of $file_name\n";
             chdir $restore_dir;
             $tar->close( );
             return undef;
         }
         binmode CONTENT;
         $file_contents = join '',<CONTENT>;
         close CONTENT;

         #######
         # Add the file to the file archive
         #  
         $file_name = File::AnySpec->os2fspec('Unix', $file_name);
         $file_name =  File::Spec::Unix->catfile($dest_dir,$file_name) if $dest_dir;
         unless ($tar->taradd($file_name, $file_contents)) {
             chdir $restore_dir;
             return undef;
         }
  
     }
     chdir $restore_dir;

     $tar->tarclose( );
     return $options_p->{tar_file};
}



#######
#
#
sub untar
{
     return undef unless(defined($_[0]));      
     shift @_ if $_[0] eq 'Archive::TarGzip';  # drop self on object call 

     unless(@_) {
         warn "No arguments.\n";
         return undef;
     }

     ######
     # Check to see if first argument is an option
     #
     my $args_p;
     if( ref($_[0]) eq 'ARRAY') {
          my %args = @{shift @_};
          $args_p = \%args;
     }
     elsif (ref($_[0]) eq 'HASH') {
          $args_p = shift @_;
     }

     ######
     # Process options
     #
     my $tar_file = $args_p->{tar_file};
     unless( $tar_file ) {
          warn( "No tar file\n" );
          return undef;
     }

     #####
     # Bring in some needed program modules
     #   
     require File::Spec;
     require File::AnySpec;
     require File::Path;
     File::Path->import( 'mkpath' );

     ########
     # Attach to an existing tar file
     #
     $args_p->{tar_flag} = '<';
     my $tar;  
     return undef unless $tar = new Archive::TarGzip($args_p);
     return undef unless $tar->taropen();

     #######
     # Add any inputs files to the extract file hash
     #
     FileList2Sel(@_, $tar->{TarGzip}->{extract_files});

     ########
     # Change to the destination directory where place the 
     # extracted files.
     #
     my $restore_dir = cwd();
     if ($args_p->{dest_dir}) {
         mkpath($args_p->{dest_dir});
         chdir $args_p->{dest_dir};
     }

     my ($tar_dir, $file_name, $dirs);
     while( 1 ) {

         $tar_dir = $tar->tarread( );
         last unless defined $tar_dir;
         last if $tar_dir->{end_of_tar};
         next if $tar_dir->{skip_file};
         my $data = $tar_dir->{data};
         my $typeflag = $tar_dir->{typeflag};
         my $name = $tar_dir->{name};
         $typeflag = 5 if( substr($name,-1,1) eq '/' && !$data);
         $name = File::AnySpec->fspec2os('Unix', $name);
         if( $typeflag == 5 ) {
              mkpath( $name );
              next;
         }

         #######
         # Just in  case the directories where not
         # put in correctly, create them for files
         #
         (undef, $dirs) = File::Spec->splitpath($name);
         mkpath ($dirs);

         ########
         # Extract the file
         #
         open FILE, "> $name";
         binmode FILE;
         print FILE $data;
         close FILE;
     }

     $tar->tarclose( );
     chdir $restore_dir;

     1
}



sub FileList2Sel
{
     my ($list_p, $select_p) = @_;

     #######
     # Add any inputs files to the extract file hash
     #
     $select_p = {} unless $select_p;
     foreach my $item (@$list_p) {
         $item = File::AnySpec->os2fspec( 'Unix', $item );     
         $select_p->{$item} = 1
     };   
}


1


__END__


=head1 NAME

Archive::TarGzip - save and restore file to and from compressed tape archives (tar)

=head1 SYNOPSIS


 use Archive::TarGzip [qw(parse_header tar untar)];

 $tar_file = Archive::TarGzip->tar(@file, [\%options or\@options]);
 $tar_file = tar(@file, [\%options or\@options]); # only if imported

 $success = Archive::TarGzip->untar([@file], \%options or\@options or @options);
 $success = untar([@file], \%options or\@options or @options); # only if imported


 $tar = new Archive::TarGip( ); # best for Archive::Tar methods
 $tar = new Archive::TarGip( $filename or filehandle, [$compress]);

 $tar = new Archive::TarGip( \%options or\@options); # best for Archive::TarGzip methods

 $tar_handle = $tar->taropen( $tar_file, $compress, [\%options or\@options]);
 $success = $tar->taradd($file_name, $file_contents) ;
 $success  = $tar->tarwrite();
 \%tar_header = $tar->tarread(@file, [\%options or\@options]);
 \%tar_header = $tar->tarread(\%options or\@options);
 $status = $tar->target( \$buffer, $size);
 $success = $tar->tarclose();

 \%tar_header = Archive::TarGzip->parse_header($buffer) ;
 \%tar_header = parse_header($buffer);  # only if imported


=head1 DESCRIPTION

The Archive::TarGzip module provides tar subroutine to archive a list of files
in an archive file in the tar format. 
The archve file may be optionally compressed using the gzip compression routines.
The ARchive::TarGzip module also provides a untar subroutine that can extract
the files from the tar or tar/gzip archive files.

The tar and untar top level subroutines use methods from the Archive::TarGzip
class. The Archive::TarGzip class is dervided from its parent Archive::Tar class.
The new methods supplied with the Archive::TarGzip derived class provide means
to access individual files within the archive file without bringing the entire
archive file into memory. When the gzip compression option is active, the
compression is performed on the fly without creating an intermediate uncompressed
tar file. The new methods provide a smaller memory footprint that enhances performance
for very large archive files.

Individual descriptions of the mehods and subroutines follows.

=over 4

=item tar subroutine

 $tar_file = Archive::TarGzip->tar(@file, [\%options or\@options]);
 $tar_file = tar(@file, [\%options or\@options]); # only if imported

The tar subroutine creates a tar archive file containing the files
in the @file list. The name of the file is $option{tar_file}.
The tar subroutine will enforce that the $option{tar_file} has
the .tar or .tar.gz extensions 
(uses the $option{compress} to determine which one).

The tar subroutine will add directories to the @file list in the
correct order needed to create the directories so that they will
be available to extract the @files files from the tar archive file.

If the $option{src_dir} is present, the tar subroutine will change
to the $option{src_dir} before reading the @file list. 
The subroutine will restore the original directory after 
processing.

If the $option{dest_dir} is present, the tar subroutine will
add the $option{dest_dir} to each of the files in the @file list.
The $options{dest_dir} name is only used for the name stored
in the tar archive file and not to access the files from the
site storage.

=item untar subroutine

 $success = Archive::TarGzip->untar([@file], \%options or\@options or @options);
 $success = untar([@file], \%options or\@options or @options); # only if imported

The untar subroutine extracts directories and files from a tar archive file.
The untar subroutine does not assume that the directories are stored in the
correct order so that they will be present as needed to create the files.

The name of the file is $option{tar_file}.
If tar subroutine that cannot find the $option{tar_file},
it will look for file with the .tar or .tar.gz extension 
(uses the $option{compress} to determine which one).

If the $option{dest_dir} is present, the tar subroutine will change
to the $option{dest_dir} before extracting the files from the tar archive file. 
The subroutine will restore the original directory after 
processing.

If the @file list is present or the @{$option{extract_file}} list is present,
the untar subroutine will extract only the files in these lists.

If the @{$option{exclude_file}} list is present, the untar subroutine will not
extract files in this list.

=item new method

 $tar = new Archive::TarGip( );
 $tar = new Archive::TarGip( $filename or filehandle, [$compress]);

 $tar = new Archive::TarGip( \%options or\@options);

The new method creates a new tar object. 
The Archive::TarGzip::new method is the only methods that hides
a  Archive::Tar method with the same name.

The new method passes $filename and $compress inputs to the
Archive::Tar::new method which will read the entire
tar archive file into memory. 

The new method with the $filename is better
when using only the Archive::TarGzip methods.

=item taropen method

 $tar_handle = $tar->taropen( $tar_file, $compress, [\%options or\@options]);

The taropen method opens a $tar_file without bringing
any of the files into memory.

If $options{tar_flag} is '>', the taropen method
creats a new $tar_file; otherwise, 
it opens the $tar_file for reading.

=item taradd method

 $success = $tar->taradd($file_name, $file_contents);

The taradd method appends $file_contents using
the name $file_name 
to the end of the tar archive file taropen for writing.
If $file_contents is undefined, 
the taradd method will use the
contents from the file $file_name.

=item tarwrite method

 $success  = $tar->tarwrite();

The tarwrite method will remove the first file
in the Archive::Tar memory and append it
to the end of the tar archive file taropen for writing.

The tarwrite method uses the $option{compress} to
decide whether use gzip compress or normal writing
of the tar archive file.

=item tarread method

 \%tar_header = $tar->tarread(@file, [\%options or\@options]);
 \%tar_header = $tar->tarread(\%options or\@options);

The tarread method reads the next file from the tar archive file
taropen for reading. 
The tar file header and file contents are returned in
the %tar_header hash along with other information needed
for processing by the Archive::Tar and Archive::TarGzip
classes.

If the $option{header_only} exists the tarread method
skips the file contents and it is not return in the
%tar_header.

If either the @file or the @{$option{extract_files}} list is 
present, the tarread method will check to see if
the file is in either of these lists.
If the file name is not in the @files list or
the @{$option{extract_files}} list,
the tarread method will set the $tar_header{skip_file} key
and all other %tar_header keys are indetermined.

If the @{$option{exclude_files}} list is 
present, the tarread method will check to see if
the file is in this list.
If the file name is in the list,
the tarread method will set the $tar_header{skip_file} key
and all other %tar_header keys are indetermined.

If the tarread method reaches the end of the tar archive
file, it will set the $tar_header{end_of_tar} key and
all other %tar_header keys are indermeined.

The $tar_header keys are as follows:

 name
 mode
 uid
 gid
 size
 mtime
 chksum
 typeflag
 linkname
 magic
 version
 uname
 gname
 devmajor
 devminor
 prefix
 error
 end_of_tar
 header_only
 skip_file
 data

=item target method

 $status = $tar->target( \$buffer, $size);

The target method gets bytes in 512 byte chunks from
the tar archive file taropen for reading.
If \$buffer is undefined, the target method skips
over the $size bytes and any additional bytes to pad out
to 512 byte boundaries.

The target method uses the $option{compress} to
decide whether use gzip uncompress or normal reading
of the tar archive file.

=item tarclose method

 $success = $tar->tarclose();

This closes the tar achive opened by the taropen method.

=item parse_header subroutine

 \%tar_header = Archive::TarGzip->parse_header($buffer) ;
 \%tar_header = parse_header($buffer);  # only if imported

The C<parse_header subroutine> takes the pack 512 byte tar file
header and parses it into a the Archive::Tar header hash
with a few additional hash keys.
This is the return for the C<target method>.

=back

=head1 REQUIREMENTS

The requirements are coming.

=head1 DEMONSTRATION

 ~~~~~~ Demonstration overview ~~~~~

Perl code begins with the prompt

 =>

The selected results from executing the Perl Code 
follow on the next lines. For example,

 => 2 + 2
 4

 ~~~~~~ The demonstration follows ~~~~~

 =>     use File::Package;
 =>     use File::AnySpec;
 =>     use File::SmartNL;
 =>     use File::Spec;

 =>     my $fp = 'File::Package';
 =>     my $snl = 'File::SmartNL';
 =>     my $uut = 'Archive::TarGzip'; # Unit Under Test
 =>     my $loaded;
 => my $errors = $fp->load_package($uut)
 => $errors
 ''

 =>      my @files = qw(
 =>          lib/Data/Str2Num.pm
 =>          lib/Docs/Site_SVD/Data_Str2Num.pm
 =>          Makefile.PL
 =>          MANIFEST
 =>          README
 =>          t/Data/Str2Num.d
 =>          t/Data/Str2Num.pm
 =>          t/Data/Str2Num.t
 =>      );
 =>      my $file;
 =>      foreach $file (@files) {
 =>          $file = File::AnySpec->fspec2os( 'Unix', $file );
 =>      }
 =>      my $src_dir = File::Spec->catdir('TarGzip', 'expected');
 => Archive::TarGzip->tar( @files, {tar_file => 'TarGzip.tar.gz', src_dir  => $src_dir,
 =>             dest_dir => 'Data-Str2Num-0.02', compress => 1} )
 'TarGzip.tar.gz'

 => Archive::TarGzip->untar( dest_dir=>'TarGzip', tar_file=>'TarGzip.tar.gz', compress => 1)
 1

 => $snl->fin(File::Spec->catfile('TarGzip', 'Data-Str2Num-0.02', 'MANIFEST'))
 'lib/Docs/Site_SVD/Data_Str2Num.pm
 MANIFEST
 Makefile.PL
 README
 lib/Data/Str2Num.pm
 t/Data/Str2Num.d
 t/Data/Str2Num.pm
 t/Data/Str2Num.t'

 => $snl->fin(File::Spec->catfile('TarGzip', 'expected', 'MANIFEST'))
 'lib/Docs/Site_SVD/Data_Str2Num.pm
 MANIFEST
 Makefile.PL
 README
 lib/Data/Str2Num.pm
 t/Data/Str2Num.d
 t/Data/Str2Num.pm
 t/Data/Str2Num.t'


=head1 NOTES

=head2 AUTHOR

The holder of the copyright and maintainer is

E<lt>support@SoftwareDiamonds.comE<gt>

=head2 COPYRIGHT NOTICE

Copyrighted (c) 2002 Software Diamonds

All Rights Reserved

=head2 BINDING REQUIREMENTS NOTICE

Binding requirements are indexed with the
pharse 'shall[dd]' where dd is an unique number
for each header section.
This conforms to standard federal
government practices, 490A (L<STD490A/3.2.3.6>).
In accordance with the License, Software Diamonds
is not liable for any requirement, binding or otherwise.

=head2 LICENSE

Software Diamonds permits the redistribution
and use in source and binary forms, with or
without modification, provided that the 
following conditions are met: 

=over 4

=item 1

Redistributions of source code must retain
the above copyright notice, this list of
conditions and the following disclaimer. 

=item 2

Redistributions in binary form must 
reproduce the above copyright notice,
this list of conditions and the following 
disclaimer in the documentation and/or
other materials provided with the
distribution.

=back

SOFTWARE DIAMONDS, http::www.softwarediamonds.com,
PROVIDES THIS SOFTWARE 
'AS IS' AND ANY EXPRESS OR IMPLIED WARRANTIES,
INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT
SHALL SOFTWARE DIAMONDS BE LIABLE FOR ANY DIRECT,
INDIRECT, INCIDENTAL, SPECIAL,EXEMPLARY, OR 
CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED
TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
LOSS OF USE,DATA, OR PROFITS; OR BUSINESS
INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY
OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
OR TORT (INCLUDING USE OF THIS SOFTWARE, EVEN IF
ADVISED OF NEGLIGENCE OR OTHERWISE) ARISING IN
ANY WAY OUT OF THE POSSIBILITY OF SUCH DAMAGE. 

=for html
<p><br>
<!-- BLK ID="NOTICE" -->
<!-- /BLK -->
<p><br>
<!-- BLK ID="OPT-IN" -->
<!-- /BLK -->
<p><br>
<!-- BLK ID="EMAIL" -->
<!-- /BLK -->
<p><br>
<!-- BLK ID="COPYRIGHT" -->
<!-- /BLK -->
<p><br>
<!-- BLK ID="LOG_CGI" -->
<!-- /BLK -->
<p><br>

=cut

### end of file ###