#####################################################################
package Helpers;

use strict;
use warnings;
use utf8;

sub new {
    my ($class) = @_;
    return bless {"Helpers" => {}}, $class;
}

sub makeChars {
    my ($class, $program) = @_;

    my @chars = split("", $program);
    $class->{"Helpers"}->{"Chars"} = \@chars;
    $class->{"Helpers"}->{"length"} = $#chars;
}

sub programLength {
    my ($class) = @_;

    my $programLength = $class->{"Helpers"}->{"length"};
    return $programLength;
}

sub getChar {
    my ($class) = @_;

    my @chars = @{$class->{"Helpers"}->{"Chars"}};
    my $currentChar = shift(@chars);
    $class->{"Helpers"}->{"Chars"} = \@chars;
    return $currentChar;
}

sub nextChar {
    my ($class) = @_;

    my @chars = @{$class->{"Helpers"}->{"Chars"}};
    my $currentChar = $chars[0];
    return $currentChar;
}

sub nextNextChar {
    my ($class) = @_;

    my @chars = @{$class->{"Helpers"}->{"Chars"}};
    my $currentChar = $chars[1];
    return $currentChar;
}

sub putChar {
    my ($class, $char) = @_;

    my @chars = @{$class->{"Helpers"}->{"Chars"}};
    unshift(@chars, $char);
    $class->{"Helpers"}->{"Chars"} = \@chars;
}

1;

#####################################################################
package CharGroups;

use strict;
use warnings;
use utf8;
no warnings qw(experimental::smartmatch);

sub new {
    my ($class) = @_;
    return bless {}, $class;
}

sub isSpaceNewline {
    my ($class, $char) = @_;

    my @spaceNewline = (" ", "\n", "\t", "\r");
    if( $char ~~ @spaceNewline ) {
        return 1;
    } else {
        return 0;
    }
}

sub isDigit {
    my ($class, $char) = @_;

    my @digits = ("0", "1", "2", "3", "4", "5", "6", "7", "8", "9");
    foreach my $digit (@digits) {
        if($char eq $digit) {
            return 1;
        }
    }

    return 0;
}

sub isAlpha {
    my ($class, $char) = @_;
    my @alpha = ();

    for my $char ('a' ... 'z') {
        push @alpha, $char;
    }
    for my $char ('A' ... 'Z') {
        push @alpha, $char;
    }
    push @alpha, "_";

    if( $char ~~ @alpha ) {
        return 1;
    } else {
        return 0;
    }
}

sub isQuote {
    my ($class, $char) = @_;
    if( $char eq '"' ) {
        return 1;
    } else {
        return 0;
    }
}

sub isSpecialCharachter {
    my ($class, $char) = @_;
    my @specialCharachters = ("{", "}", "(", ")", "[", "]", ",", ";", ":", ".", "=", "?");

    if($char ~~ @specialCharachters) {
        return 1;
    } else {
        return 0;
    }
}

sub isOperator {
    my ($class, $char) = @_;
    my @operators = ("+", "-", "|", "*", "/", ">", "<", "!", "&", "%");

    if($char ~~ @operators) {
        return 1;
    } else {
        return 0;
    }
}

1;

#####################################################################
package Lexer;

use strict;
use warnings;
use utf8;

our @ISA = qw(Helpers CharGroups);

sub new {
    my ($class) = @_;
    return bless {}, $class;
}

sub lexer {
    my ($class, $program) = @_;
    my @tokens;

    $class->makeChars($program);

    my $counter = 0;
    my $programLength = $class->programLength();

    while($counter <= $programLength) {
        my $currentChar = $class->getChar();
        $counter++;

        if($class->isSpaceNewline($currentChar)) {
            next;
        }

        if($currentChar eq "#") {
            my $comment = "";
            my $delimiter = "@";

            $currentChar = $class->getChar();
            $counter++;

            while($currentChar ne $delimiter) {
                $comment .= $currentChar;
                $currentChar = $class->getChar();
                $counter++;
            }

            $class->getChar();
            $counter++;

            my $token = { "type" => "Comment", "value" => $comment };
            push(@tokens, $token);
            next;
        }

        if($currentChar eq "(" && $class->nextChar eq "?") {
            my $embedBlock = "";
            $class->getChar();
            $counter++;

            $currentChar = $class->getChar();
            $counter++;
            while($currentChar ne "?" && $class->nextChar() ne ")" ) {
                $embedBlock .= $currentChar;
                $currentChar = $class->getChar();
                $counter++;
                while($currentChar ne "?") {
                    $embedBlock .= $currentChar;
                    $currentChar = $class->getChar();
                    $counter++;
                }
            }

            $class->getChar();
            $counter++;

            my $token = { "type" => "EmbedBlock", "value" => $embedBlock };
            push(@tokens, $token);
            next;
        }

        if($currentChar eq "!" && $class->isAlpha($class->nextChar())) {
            my $object = "";
            $object .= $currentChar;

            $currentChar = $class->getChar();
            $counter++;

            while($class->isAlpha($currentChar)) {
                $object .= $currentChar;
                $currentChar = $class->getChar();
                $counter++;
            }

            $class->putChar($currentChar);
            $counter = $counter - 1;

            my $token = { "type" => "Object", "value" => $object };
            push(@tokens, $token);
            next;
        }

        if($currentChar eq ":" && $class->nextChar() eq ":") {
            $class->getChar();
            $counter++;

            my $token = { "type" => "ObjectColon", "value" => "::" };
            push(@tokens, $token);
            next;
        }

        if($currentChar eq "=" && $class->nextChar() eq "=") {
            $class->getChar();
            $counter++;

            my $token = { "type" => "Equals", "value" => "==" };
            push(@tokens, $token);
            next;
        }

        if($class->isSpecialCharachter($currentChar)) {
            my $token = { "type" => "SpecialCharachter", "value" => $currentChar };
            push(@tokens, $token);
            next;
        }

        if($class->isOperator($currentChar)) {
            if($currentChar eq "&") {
                my $nextChar = $class->nextChar();
                if( $nextChar eq "&" ) {
                    $class->getChar();
                    $counter++;

                    my $token = { "type" => "Operator", "value" => "&&" };
                    push(@tokens, $token);
                    next;
                }
            }
        }

        if($class->isOperator($currentChar)) {
            if($currentChar eq "|") {
                my $nextChar = $class->nextChar();
                if( $nextChar eq "|" ) {
                    $class->getChar();
                    $counter++;

                    my $token = { "type" => "Operator", "value" => "||" };
                    push(@tokens, $token);
                    next;
                }
            }
        }

        if($class->isOperator($currentChar)) {
            if($currentChar eq "!") {
                my $nextChar = $class->nextChar();
                if( $nextChar eq "=" ) {
                    $class->getChar();
                    $counter++;

                    my $token = { "type" => "Operator", "value" => "!=" };
                    push(@tokens, $token);
                    next;
                }
            }
        }

        if($class->isOperator($currentChar)) {
            if($currentChar eq ">") {
                my $nextChar = $class->nextChar();
                if( $nextChar eq "=" ) {
                    $class->getChar();
                    $counter++;

                    my $token = { "type" => "Operator", "value" => ">=" };
                    push(@tokens, $token);
                    next;
                }
            }
        }

        if($class->isOperator($currentChar)) {
            if($currentChar eq "<") {
                my $nextChar = $class->nextChar();
                if( $nextChar eq "=" ) {
                    $class->getChar();
                    $counter++;

                    my $token = { "type" => "Operator", "value" => "<=" };
                    push(@tokens, $token);
                    next;
                }
            }
        }

        if($class->isOperator($currentChar)) {
            if($currentChar eq "*") {
                my $nextChar = $class->nextChar();
                if( $nextChar eq "*" ) {
                    $class->getChar();
                    $counter++;

                    my $token = { "type" => "Operator", "value" => "**" };
                    push(@tokens, $token);
                    next;
                }
            }
        }

        if($class->isOperator($currentChar)) {
            my $token = { "type" => "Operator", "value" => $currentChar };
            push(@tokens, $token);
            next;
        }

        if($class->isQuote($currentChar)) {
            my $string = "";
            my $delimiter = $currentChar;

            $currentChar = $class->getChar();
            $counter++;

            while($currentChar ne $delimiter) {
                $string .= $currentChar;
                $currentChar = $class->getChar();
                $counter++;
            }

            my $token = { "type" => "String", "value" => $string };
            push(@tokens, $token);
            next;
        }

        if($currentChar eq "e" && $class->nextChar() eq "q") {
            $class->getChar();
            $counter++;

            my $token = { "type" => "Operator", "value" => "eq" };
            push(@tokens, $token);
            next;
        }

        if($currentChar eq "n" && $class->nextChar() eq "e") {
            $class->getChar();
            $counter++;

            my $token = { "type" => "Operator", "value" => "ne" };
            push(@tokens, $token);
            next;
        }

        if($class->isAlpha($currentChar)) {
            my $symbol = "";
            $symbol .= $currentChar;

            $currentChar = $class->getChar();
            $counter++;

            while($class->isAlpha($currentChar)) {
                $symbol .= $currentChar;
                $currentChar = $class->getChar();
                $counter++;
            }

            $class->putChar($currentChar);
            $counter = $counter - 1;

            my $token = { "type" => "Symbol", "value" => $symbol };
            push(@tokens, $token);
            next;
        }

        if($class->isDigit($currentChar)) {
            my $number = "";
            $number .= $currentChar;

            $currentChar = $class->getChar();
            $counter++;

            while($class->isDigit($currentChar) || $currentChar eq ".") {
                $number .= $currentChar;
                $currentChar = $class->getChar();
                $counter++;
            }

            $class->putChar($currentChar);
            $counter = $counter - 1;

            my $token = { "type" => "Number", "value" => $number };
            push(@tokens, $token);

            next;
        }

        else {
            my $errorArea = "";
            foreach (@tokens[-3 .. -1]) {
                my %token = %{$_};
                $errorArea .= $token{value} . " ";
            }

            print "Lexical Error at ", $errorArea, "$currentChar \n";
        }
    }

    return @tokens;
}

1;

#####################################################################
package ParserHelpers;

use strict;
use warnings;
use utf8;

sub new {
    my ($class) = @_;
    return bless {"ParserHelpers" => {}}, $class;
}

sub makeTokens {
    my ($class, $tokens) = @_;

    my @tokens = @{$tokens};
    $class->{"ParserHelpers"}->{"Tokens"} = \@tokens;
    $class->{"ParserHelpers"}->{"TokensLength"} = $#tokens;
}

sub tokensLength {
    my ($class) = @_;

    my $tokensLength = $class->{"ParserHelpers"}->{"TokensLength"};
    return $tokensLength;
}

sub getToken {
    my ($class) = @_;

    my @tokens = @{$class->{"ParserHelpers"}->{"Tokens"}};
    my $currentToken = shift(@tokens);
    $class->{"ParserHelpers"}->{"Tokens"} = \@tokens;
    return $currentToken;
}

sub nextToken {
    my ($class) = @_;

    my @tokens = @{$class->{"ParserHelpers"}->{"Tokens"}};
    my $currentToken = $tokens[0];
    return $currentToken;
}

sub putToken {
    my ($class, $token) = @_;

    my @tokens = @{$class->{"ParserHelpers"}->{"Tokens"}};
    unshift(@tokens, $token);
    $class->{"ParserHelpers"}->{"Tokens"} = \@tokens;
}

sub setCurrentClass {
    my ($class, $className) = @_;
    $class->{"ParserHelpers"}->{"currentClassName"} = $className;
}

sub getCurrentClass {
    my ($class) = @_;
    my $currentClassName = $class->{"ParserHelpers"}->{"currentClassName"};
    return $currentClassName;
}

sub setCurrentFunction {
    my ($class, $functionName) = @_;
    $class->{"ParserHelpers"}->{"currentFunctionName"} = $functionName;
}

sub getCurrentFunction {
    my ($class) = @_;
    my $currentFunctionName = $class->{"ParserHelpers"}->{"currentFunctionName"};
    return $currentFunctionName;
}

sub setLastBlock {
    my ($class, $lastBlock) = @_;
    $class->{"ParserHelpers"}->{"lastBlock"} = $lastBlock;
}

sub nextTokens {
    my ($class) = @_;
    my @tokens = @{$class->{"ParserHelpers"}->{"Tokens"}};

    my $nextTokens;
    foreach (@tokens[0 .. 5]) {
        my %token = %{$_};
        $nextTokens .= $token{"value"} . " ";
    }

    return $nextTokens;
}

sub printError {
    my ($class) = @_;

    my $currentClassName = $class->getCurrentClass();
    my $currentFunctionName = $class->getCurrentFunction();
    my $nextTokens = $class->nextTokens();

    print "Error at: \nclassName: ", $currentClassName, "\nfunctionName: "
          , $currentFunctionName, "\nnextTokens: ", $nextTokens, "\n";
}

1;

#####################################################################
package Parser;

use strict;
use warnings;
use utf8;
use Data::Printer;

our @ISA = qw(ParserHelpers);

sub new {
    my ($class) = @_;
    return bless {}, $class;
}

sub parse {
    my ($class, $tokens) = @_;
    $class->makeTokens($tokens);

    my $code = $class->Lang();
    if($code) {
        return $code;
    } else {
        $class->printError();
    }
}

sub Lang {
    # check line return $lang; => return 0;
    my ($class) = @_;
    my $lang = "";

    my $classString = $class->Class();
    if( $class ) {
        $lang .= $classString;
    } else {
        return $lang;
    }

    my $rightLang = $class->Lang();
    if( $rightLang ) {
        $lang .= $rightLang;
    } else {
        return 0;
    }

    return $lang;
}

sub Class {
    my ($class) = @_;
    my $classString = "";

    my $tokenClass = $class->TokenClass();
    if($tokenClass) {
        $classString .= $tokenClass;
    } else {
        return 0;
    }

    my $className = $class->ClassName();
    if($className) {
        $classString .= $className;
        $class->setCurrentClass($className);
    } else {
        return 0;
    }

    my $classBlock = $class->ClassBlock();
    if($classBlock) {
        $classString .= $classBlock;
    } else {
        return 0;
    }

    return $classBlock;
}

sub TokenClass {
    my ($class) = @_;

    my $currentToken = $class->getToken();
    if($currentToken->{"value"} eq "class") {
        return "class";
    } else {
        $class->putToken($currentToken);
        return 0;
    }
}

sub ClassName {
    my ($class) = @_;

    my $currentToken = $class->getToken();
    if($currentToken->{"type"} eq "Symbol") {
        return $currentToken->{"value"};
    } else {
        $class->putToken($currentToken);
        return 0;
    }
}

sub ClassBlock {
    my ($class) = @_;
    my $classBlock = "";

    my $currentToken = $class->getToken();
    if($currentToken->{"value"} eq "{") {
        $classBlock .= "{\n";
    } else {
        $class->putToken($currentToken);
        return 0;
    }

    my $classGroups = $class->ClassGroups();
    if($classGroups) {
        $classBlock .= $classGroups;
    } else {
        return 0;
    }

    if($currentToken->{"value"} eq "}") {
        $classBlock .= "\n}";
    } else {
        $class->putToken($currentToken);
        return 0;
    }
}

sub ClassGroups {
    # check line return $classGroups; => return 0;
    my ($class) = @_;
    my $classGroups = "";

    my $group = $class->Group();
    if($group) {
        $classGroups .= $group;
    } else {
        return $classGroups;
    }

    my $rightClassGroup = $class->ClassGroups();
    if($rightClassGroup) {
        $classGroups .= $rightClassGroup;
    } else {
        return 0;
    }

    return $classGroups;
}

sub Group {
    my ($class) = @_;

    my $comment = $class->Comment();
    if ( $comment ) {
        return $comment;
    }

    my $parent = $class->Parent();
    if($parent) {
        return $parent;
    }

    my $packages = $class->Packages();
    if($packages) {
        return $packages;
    }

    my $function = $class->Function();
    if($function) {
        return $function;
    }

    my $embedBlock = $class->EmbedBlock();
    if($embedBlock) {
        return $embedBlock;
    }

    return 0;
}

sub Comment {
    my ($class) = @_;
    my $comment = "";

    my $currentToken = $class->getToken();
    if($currentToken->{"value"} eq "#") {
        $comment .= "#";
    } else {
        $class->putToken($currentToken);
        return 0;
    }

    my $lineComment = $class->LineComment();
    if($lineComment) {
        $comment .= $lineComment;
    } else {
        return 0;
    }

    $currentToken = $class->getToken();
    if($currentToken->{"value"} eq "@") {
        $comment .= "@";
    } else {
        $class->putToken($currentToken);
        return 0;
    }

    return $comment;
}

sub LineComment {
    my ($class) = @_;
    my $lineComment = "";

    my $currentToken = $class->getToken();
    if($currentToken->{"type"} eq "Comment") {
        $lineComment .= $currentToken->{"value"};
    } else {
        $class->putToken($currentToken);
        return 0;
    }

    return $lineComment;
}

sub Parent {
    my ($class) = @_;
    my $parent = "";

    my $currentToken = $class->getToken();
    if($currentToken->{"value"} eq "parent") {
        my $parent .= $currentToken->{"value"};
    } else {
        $class->putToken($currentToken);
        return 0;
    }

    $currentToken = $class->getToken();
    if($currentToken->{"value"} eq "(") {
        my $parent .= $currentToken->{"value"};
    } else {
        $class->putToken($currentToken);
        return 0;
    }

    my $classNames = $class->ClassNames();
    if($classNames) {
        $parent .= $classNames;
    } else {
        return 0;
    }

    $currentToken = $class->getToken();
    if($currentToken->{"value"} eq ")") {
        my $parent .= $currentToken->{"value"};
    } else {
        $class->putToken($currentToken);
        return 0;
    }

    $currentToken = $class->getToken();
    if($currentToken->{"value"} eq ";") {
        my $parent .= $currentToken->{"value"};
    } else {
        $class->putToken($currentToken);
        return 0;
    }

    return $currentToken;
}

sub ClassNames {
    my ($class) = @_;
    my $classNames = "";

    my $className = $class->ClassName();
    if($className) {
        $classNames .= $className;
    } else {
        return 0;
    }

    my $comma = $class->Comma();
    if($comma) {
        $classNames .= $comma;
    } else {
        return $classNames;
    }

    my $rightClassNames = $class->ClassNames();
    if($classNames) {
        $classNames .= $rightClassNames;
    } else {
        return 0;
    }

    return $classNames;
}

sub Packages {
    my ($class) = @_;
    my $packages = "";

    my $currentToken = $class->getToken();
    if($currentToken->{"value"} eq "(") {
        $packages .= $currentToken->{"value"};
    } else {
        $class->putToken($currentToken);
        return 0;
    }

    my $packageList = $class->PackageList();
    if($packageList) {
        $packages .= $packageList;
    } else {
        return 0;
    }

    $currentToken = $class->getToken();
    if($currentToken->{"value"} eq ")") {
        $packages .= $currentToken->{"value"};
    } else {
        $class->putToken($currentToken);
        return 0;
    }

    $currentToken = $class->getToken();
    if($currentToken->{"value"} eq ";") {
        $packages .= $currentToken->{"value"};
    } else {
        $class->putToken($currentToken);
        return 0;
    }

    return $packages;
}

sub PackageList {
    my ($class) = @_;
    my $packageList = "";

    my $package = $class->Package();
    if($package) {
        $packageList .= $package;
    } else {
        return 0;
    }

    my $comma = $class->Comma();
    if($package) {
        $packageList .= $comma;
    } else {
        return $packageList;
    }

    my $rightpackageList = $class->PackageList();
    if($rightpackageList) {
        $packageList .= $rightpackageList;
    } else {
        return 0;
    }

    return $rightpackageList;
}

sub Package {
    my ($class) = @_;

    my $packageWithConstructor = $class->PackageWithConstructor();
    if($packageWithConstructor) {
        return $packageWithConstructor;
    }

    my $packageWithoutConstructor = $class->PackageWithoutConstructor();
    if($packageWithoutConstructor) {
        return $packageWithoutConstructor;
    }

    return 0;
}

sub PackageWithConstructor {
    my ($class) = @_;
    my $packageWithConstructor = "";

    my $currentToken = $class->getToken();
    if($currentToken->{"type"} eq "Object") {
        $packageWithConstructor .= $currentToken->{"value"};
    } else {
		$class->putToken($currentToken);
		return 0;
	}

	$currentToken = $class->getToken();
    if($currentToken->{"value"} eq "=") {
        $packageWithConstructor .= $currentToken->{"value"};
    } else {
		$class->putToken($currentToken);
		return 0;
	}

	my $packageName = $class->PackageName();
	if($packageName) {
		$packageWithConstructor .= $packageName;
	} else {
		return 0;
	}

	$currentToken = $class->getToken();
    if($currentToken->{"value"} eq ".") {
        $packageWithConstructor .= $currentToken->{"value"};
    } else {
		$class->putToken($currentToken);
		return 0;
	}

	my $constructor = $class->Constructor();
	if($constructor) {
		$packageWithConstructor .= $packageName;
	} else {
		return 0;
	}

	$currentToken = $class->getToken();
    if($currentToken->{"value"} eq "(") {
        $packageWithConstructor .= $currentToken->{"value"};
    } else {
		$class->putToken($currentToken);
		return 0;
	}

	my $objectParameters = $class->ObjectParameters();
	if($objectParameters) {
		$packageWithConstructor .= $objectParameters;
	} else {
		return 1;
	}

	$currentToken = $class->getToken();
    if($currentToken->{"value"} eq ")") {
        $packageWithConstructor .= $currentToken->{"value"};
    } else {
		$class->putToken($currentToken);
		return 0;
	}

	return $packageWithConstructor;
}

sub ObjectParameters {
	my ($class) = @_;

	my $packageParams = $class->PackageParams();
	if($packageParams) {
		return $packageParams;
	}

	my $parameters = $class->Parameters();
	if($parameters) {
		return $parameters;
	}

	return 0;
}

sub PackageParams {
	my ($class) = @_;
	my $packageParams = "";

	my $keyValue = $class->KeyValue();
	if($keyValue) {
		$packageParams .= $keyValue;
	} else {
		return 0;
	}

	my $comma = $class->Comma();
	if($comma) {
		$packageParams .= $comma;
	} else {
		return $packageParams;
	}

	my $rightPackageParams = $class->PackageParams();
	if($rightPackageParams) {
		$packageParams .= $rightPackageParams;
	} else {
		return 0;
	}

	return $packageParams;
}

sub PackageName {
	my ($class) = @_;
	my $packageName = "";

	my $packageDir = $class->PackageDir();
	if($packageDir) {
		$packageName .= $packageDir;
	} else {
		return 0;
	}

	my $currentToken = $class->getToken();
	if($currentToken->{"type"} eq "ObjectColon") {
		$packageName .= $currentToken->{"value"};
	} else {
		return $packageName;
	}

	my $rightpackageName = $class->PackageName();
	if($rightpackageName) {
		$packageName .= $rightpackageName;
	} else {
		return 0;
	}

	return $packageName;
}

sub PackageWithConstructor {
	my ($class) = @_;
	my $packageWithConstructor = "";

	my $packageName = $class->PackageName();
	if($packageName) {
		$packageWithConstructor .= $packageName;
	} else {
		return 0;
	}

	my $qw = $class->QW();
	if($qw) {
		$packageWithConstructor .= $qw;
	} else {
		return 1;
	}

	return $packageWithConstructor;
}

sub QW {
	my ($class) = @_;
	my $qw = "";

	my $dot = $class->Dot();
	if($dot) {
		$qw .= $dot;
	} else {
		return 0;
	}

	my $currentToken = $class->getToken();
	if($currentToken->{"value"} eq "(") {
		$qw .= $currentToken->{"value"};
	} else {
		return 0;
	}

	my $functionList = $class->FunctionList();
	if($functionList) {
		$qw .= $functionList;
	} else {
		return 0;
	}

	$currentToken = $class->getToken();
	if($currentToken->{"value"} eq ")") {
		$qw .= $currentToken->{"value"};
	} else {
		return 0;
	}

	return $qw;
}

sub FunctionList {
	my ($class) = @_;
	my $functionList = "";

	my $functionName = $class->FunctionName();
	if($functionName) {
		$functionList .= $functionName;
	} else {
		return 0;
	}

	my $comma = $class->Comma();
	if($comma) {
		$functionList .= $comma;
	} else {
		return $functionList;
	}

	my $rightFunctionName = $class->FunctionList();
	if($rightFunctionName) {
		$functionList .= $rightFunctionName;
	} else {
		return 0;
	}

	return $functionList;
}

sub Constructor {
	my ($class) = @_;
	my $constructor = "";

	my $currentToken = $class->getToken();
	if($currentToken->{"type"} eq "Symbol") {
		$constructor .= $currentToken->{"value"};
	} else {
		return 0;
	}

	return $constructor;
}

sub Object {
	my ($class) = @_;
	my $object = "";

	my $currentToken = $class->getToken();
	if($currentToken->{"type"} eq "Symbol") {
		$object .= $currentToken->{"value"};
	} else {
		return 0;
	}

	return $object;
}

sub PackageDir {
	my ($class) = @_;
	my $packageDir = "";

	my $currentToken = $class->getToken();
	if($currentToken->{"type"} eq "Symbol") {
		$packageDir .= $currentToken->{"value"};
	} else {
		return 0;
	}

	return $packageDir;
}

sub Function {
	my ($class) = @_;
	my $function = "";

	my $tokenFunction = $class->TokenFunction();
	if($tokenFunction) {
		$function .= $tokenFunction;
	} else {
		return 0;
	}

	my $functionName = $class->FunctionName();
	if($functionName) {
		$function .= $functionName;
	} else {
		return 0;
	}

	my $currentToken = $class->getToken();
	if($currentToken->{"value"} eq "(") {
		$function .= $currentToken->{"value"};
	} else {
		return 0;
	}

	my $functionParamList = $class->FunctionParamList();
	if($functionParamList) {
		$function .= $functionParamList;
	} else {
		return 0;
	}

	my $currentToken = $class->getToken();
	if($currentToken->{"value"} eq ")") {
		$function .= $currentToken->{"value"};
	} else {
		return 0;
	}

	my $codeBlock = $class->CodeBlock();
	if($codeBlock) {
		$function .= $codeBlock;
	} else {
		return 0;
	}

	return $function;
}

sub FunctionName {
	my ($class) = @_;
	my $functionName = "";

	my $currentToken = $class->getToken();
	if($currentToken->{"type"} eq "Symbol") {
		$functionName .= $currentToken->{"value"};
	} else {
		return 0;
	}

	return $functionName;
}

sub FunctionParamList {
	my ($class) = @_;
	my $functionParamList = "";

	my $emptyParamList = $class->EmptyParamList();
	if($emptyParamList) {
		return $emptyParamList;
	}

	my $functionParams = $class->FunctionParams();
	if($functionParams) {
		return $functionParams;
	}

	return 0;
}

sub EmptyParamList {
	my ($class) = @_;
	return "";
}

sub FunctionParams {
	my ($class) = @_;
	my $functionParams = "";

	my $arg = $class->Arg();
	if($arg) {
		$functionParams .= $arg;
	} else {
		return 0;
	}

	my $comma = $class->Comma();
	if($arg) {
		$functionParams .= $comma;
	} else {
		return $functionParams;
	}

	my $rightFunctionParams = $class->FunctionParams();
	if($rightFunctionParams) {
		$functionParams .= $rightFunctionParams;
	} else {
		return 0;
	}

	return $rightFunctionParams;
}

sub Arg {
	my ($class) = @_;
	my $arg = "";

	my $currentToken = $class->getToken();
	if($currentToken->{"type"} eq "Symbol") {
		$arg .= $currentToken->{"value"};
	} else {
		return 0;
	}

	return $arg;
}

sub CodeBlock {
	my ($class) = @_;
	my $codeBlock = "";

	my $currentToken = $class->getToken();
	if($currentToken->{"type"} eq "{") {
		$codeBlock .= $currentToken->{"value"};
	} else {
		return 0;
	}

	my $blocks = $class->Blocks();
	if($blocks) {
		$codeBlock .= $blocks;
	} else {
		return 0;
	}

	my $currentToken = $class->getToken();
	if($currentToken->{"type"} eq "}") {
		$codeBlock .= $currentToken->{"value"};
	} else {
		return 0;
	}

	return $codeBlock;
}


1;

package Lang::HL::Syntax;

use strict;
use warnings;
use utf8;

our $VERSION = '0.03';

sub new {
    my ($class) = @_;
    return bless {}, $class;
}

sub syntax {
    my ($class, $program) = @_;

    $program =~ s/[\t\r\n\f]+//g;

    my @tokens = $class->lexer($program);
    my $code = $class->parse(\@tokens);
    return $code;
}

1;

__END__

=head1 NAME

Lang::HL::Syntax - Syntax Check for HL

=head1 SYNOPSIS

	$> hle <directory>

=head1 AUTHOR

Rajkumar Reddy

=head1 COPYRIGHT AND LICENSE

Copyright (C) 2022 by Rajkumar Reddy. All rights reserved.


=cut
