/*  You may distribute under the terms of either the GNU General Public License
 *  or the Artistic License (the same terms as Perl itself)
 *
 *  (C) Paul Evans, 2016 -- leonerd@leonerd.org.uk
 */

#include "EXTERN.h"
#include "perl.h"
#include "XSUB.h"

/*
 * Some Future class helper functions
 */

static SV *future_new_done_from_stack(pTHX_ SV **mark)
{
  dSP;

  EXTEND(SP, 1);

  ENTER;
  SAVETMPS;

  PUSHMARK(mark);
  SV **bottom = mark + 1;

  // splice the class name 'Future' in to the start of the stack

  for (SV **svp = SP; svp >= bottom; svp--) {
    *(svp+1) = *svp;
  }
  *bottom = sv_2mortal(newSVpvn("Future", 6));
  SP++;
  PUTBACK;

  call_method("done", G_SCALAR);

  SPAGAIN;

  SV *f = SvREFCNT_inc(POPs);

  FREETMPS;
  LEAVE;

  return f;
}

static SV *future_new_fail(pTHX_ SV *failure)
{
  dSP;

  ENTER;
  SAVETMPS;

  PUSHMARK(SP);
  mPUSHp("Future", 6);
  mPUSHs(newSVsv(failure));
  PUTBACK;

  call_method("fail", G_SCALAR);

  SPAGAIN;

  SV *f = SvREFCNT_inc(POPs);

  FREETMPS;
  LEAVE;

  return f;
}

static int future_is_ready(pTHX_ SV *f)
{
  dSP;

  ENTER;
  SAVETMPS;

  PUSHMARK(SP);
  XPUSHs(f);
  PUTBACK;

  call_method("is_ready", G_SCALAR);

  SPAGAIN;

  int is_ready = POPi;

  PUTBACK;
  FREETMPS;
  LEAVE;

  return is_ready;
}

static void future_get_to_stack(pTHX_ SV *f, I32 gimme)
{
  dSP;

  ENTER;

  PUSHMARK(SP);
  XPUSHs(f);
  PUTBACK;

  call_method("get", gimme);

  LEAVE;
}

/*
 * Custom ops
 */

static XOP xop_leaveasync;
static OP *pp_leaveasync(pTHX)
{
  dSP;
  dMARK;

  if(SvTRUE(ERRSV)) {
    PUSHs(future_new_fail(aTHX_ ERRSV));
  }
  else {
    PUSHs(future_new_done_from_stack(aTHX_ mark));
  }

  return PL_op->op_next;
}

static OP *newLEAVEASYNCOP(I32 flags)
{
  OP *op = newOP(OP_CUSTOM, flags);
  op->op_ppaddr = &pp_leaveasync;

  return op;
}

static XOP xop_await;
static OP *pp_await(pTHX)
{
  dSP;
  SV *f = POPs;
  PUTBACK;

  if(!sv_isobject(f))
    croak("Expected a blessed object reference to await");

  if(future_is_ready(aTHX_ f)) {
    // This might throw
    future_get_to_stack(aTHX_ f, GIMME_V);
    return PL_op->op_next;
  }

  croak("TODO: implement await on non-ready Future\n");
}

static OP *newAWAITOP(I32 flags, OP *expr)
{
  OP *op = newUNOP(OP_CUSTOM, flags, expr);
  op->op_ppaddr = &pp_await;

  return op;
}

/*
 * Lexer extensions
 */

#define lex_consume(s)  MY_lex_consume(aTHX_ s)
static int MY_lex_consume(pTHX_ char *s)
{
  // I want strprefix()
  size_t i;
  for(i = 0; s[i]; i++) {
    if(s[i] != PL_parser->bufptr[i])
      return 0;
  }

  lex_read_to(PL_parser->bufptr + i);
  return i;
}

#define sv_cat_c(sv, c)  MY_sv_cat_c(aTHX_ sv, c)
static void MY_sv_cat_c(pTHX_ SV *sv, U32 c)
{
  char ds[UTF8_MAXBYTES + 1], *d;
  d = (char *)uvchr_to_utf8((U8 *)ds, c);
  if (d - ds > 1) {
    sv_utf8_upgrade(sv);
  }
  sv_catpvn(sv, ds, d - ds);
}

#define lex_scan_ident()  MY_lex_scan_ident(aTHX)
static SV *MY_lex_scan_ident(pTHX)
{
  // Inspired by
  //   https://metacpan.org/source/MAUKE/Function-Parameters-1.0705/Parameters.xs#L265
  I32 c;
  bool at_start;
  SV *ret = newSVpvs("");
  if(lex_bufutf8())
    SvUTF8_on(ret);

  at_start = TRUE;

  c = lex_peek_unichar(0);

  while(c != -1) {
    if(at_start ? isIDFIRST_uni(c) : isALNUM_uni(c)) {
      at_start = FALSE;
      sv_cat_c(ret, lex_read_unichar(0));

      c = lex_peek_unichar(0);
    }
    else
      break;
  }

  if(SvCUR(ret))
    return ret;

  SvREFCNT_dec(ret);
  return NULL;
}

/*
 * Keyword plugins
 */

static int async_keyword_plugin(pTHX_ OP **op_ptr)
{
  lex_read_space(0);

  // At this point we want to parse the sub NAME BLOCK or sub BLOCK
  // We can't just call parse_fullstmt because that will do too much that we
  //   can't hook into. We'll have to go a longer way round.

  // async must be immediately followed by 'sub'
  if(!lex_consume("sub"))
    croak("Expected async to be followed by sub");
  lex_read_space(0);

  // Might be named or anonymous
  SV *name = lex_scan_ident();
  lex_read_space(0);

  if(lex_peek_unichar(0) != '{')
    croak("Expected async sub %sto be followed by '{'", name ? "NAME " : "");

  I32 floor_ix = start_subparse(FALSE, name ? 0 : CVf_ANON);
  SAVEFREESV(PL_compcv);

  I32 save_ix = block_start(TRUE);

  OP *body = parse_block(0);

  SvREFCNT_inc(PL_compcv);
  body = block_end(save_ix, body);

  // turn block into
  //    PUSHMARK; eval { BLOCK }; LEAVEASYNC

  OP *op = newLISTOP(OP_LINESEQ, 0, newOP(OP_PUSHMARK, 0), NULL);

  OP *try;
  op = op_append_elem(OP_LINESEQ, op, try = newUNOP(OP_ENTERTRY, 0, body));
  op_contextualize(try, G_ARRAY);

  op = op_append_elem(OP_LINESEQ, op, newLEAVEASYNCOP(OPf_WANT_SCALAR));

  CV *cv = newATTRSUB(floor_ix,
    name ? newSVOP(OP_CONST, 0, SvREFCNT_inc(name)) : NULL,
    NULL,
    NULL,
    op);

  if(name) {
    *op_ptr = newOP(OP_NULL, 0);

    SvREFCNT_dec(name);
    return KEYWORD_PLUGIN_STMT;
  }
  else {
    *op_ptr = newUNOP(OP_REFGEN, 0,
      newSVOP(OP_ANONCODE, 0, (SV *)cv));

    return KEYWORD_PLUGIN_EXPR;
  }
}

static int await_keyword_plugin(pTHX_ OP **op_ptr)
{
  // TODO: Forbid this except inside 'async sub'

  lex_read_space(0);

  // await EXPR wants a single term expression
  OP *expr = parse_termexpr(0);

  *op_ptr = newAWAITOP(0, expr);

  return KEYWORD_PLUGIN_EXPR;
}

static int (*next_keyword_plugin)(pTHX_ char *, STRLEN, OP **);

static int my_keyword_plugin(pTHX_ char *kw, STRLEN kwlen, OP **op_ptr)
{
  HV *hints = GvHV(PL_hintgv);

  if((PL_parser && PL_parser->error_count) ||
     !hints)
    return (*next_keyword_plugin)(aTHX_ kw, kwlen, op_ptr);

  if(kwlen == 5 && strEQ(kw, "async") &&
      hv_fetchs(hints, "Future::AsyncAwait/async", 0))
    return async_keyword_plugin(aTHX_ op_ptr);

  if(kwlen == 5 && strEQ(kw, "await") &&
      hv_fetchs(hints, "Future::AsyncAwait/async", 0))
    return await_keyword_plugin(aTHX_ op_ptr);

  return (*next_keyword_plugin)(aTHX_ kw, kwlen, op_ptr);
}

MODULE = Future::AsyncAwait    PACKAGE = Future::AsyncAwait

BOOT:
  XopENTRY_set(&xop_leaveasync, xop_name, "leaveasync");
  XopENTRY_set(&xop_leaveasync, xop_desc, "leaveasync()");
  XopENTRY_set(&xop_leaveasync, xop_class, OA_UNOP);
  Perl_custom_op_register(aTHX_ &pp_leaveasync, &xop_leaveasync);

  XopENTRY_set(&xop_await, xop_name, "await");
  XopENTRY_set(&xop_await, xop_desc, "await()");
  XopENTRY_set(&xop_await, xop_class, OA_UNOP);
  Perl_custom_op_register(aTHX_ &pp_await, &xop_await);

  next_keyword_plugin = PL_keyword_plugin;
  PL_keyword_plugin = &my_keyword_plugin;
