#  You may distribute under the terms of either the GNU General Public License
#  or the Artistic License (the same terms as Perl itself)
#
#  (C) Paul Evans, 2015 -- leonerd@leonerd.org.uk

package Device::Chip::Adapter;

use strict;
use warnings;

our $VERSION = '0.01';

use Carp;

=head1 NAME

C<Device::Chip::Adapter> - an abstraction of a hardware communication device

=head1 DESCRIPTION

This package describes an interfaces tha classes can use to implement a driver
to provide access to some means of connecting an electronic chip or hardware
module to a computer. An instance implementing this interface provides some
means to send electrical signals to a connected chip or module, and receive
replies back from it; this device is called the I<adapter>. This is provided
as a service to some instance of the related interface, L<Device::Chip>.

It is suggested that a driver for a particular adapter provides a concrete
class named within the C<Device::Chip::Adapter> heirarchy, adding the basic
name of the product or means of communication as a suffix; for example the
driver for communication device based on the I<FDTI> range of devices would
be called:

   package Device::Chip::Adapter::FTDI;

This package provides a base class that such a specific implementation class
could use as a superclass, but it is not required to. The important detail is
that it provides the interface described by this documentation.

=cut

=head1 METHODS

The following methods documented with a trailing call to C<< ->get >> return
L<Future> instances.

=cut

=head2 make_protocol

   $protocol = $adapter->make_protocol( $pname )->get

Returns an object that satisfies one of the interfaces documented below in
L</PROTOCOLS>, depending on the protocol name given by I<$pname>. This should
be one of the following values:

   SPI

It is unspecified what class these objects should belong to. In particular, it
is permitted that an adapter could even return itself as the protocol
implementation, provided it has the methods to satisfy the interface for that
particular protocol. This is especially convenient in the case that the
adapter is only capable of one kind of protocol.

=cut

# A default implementation that uses some reflection to simplify
# implementations
sub make_protocol
{
   my $self = shift;
   my ( $pname ) = @_;

   if( my $code = $self->can( "make_protocol_$pname" ) ) {
      return $code->( $self );
   }
   else {
      croak "Unrecognised protocol name $pname";
   }
}

=head2 shutdown

   $adapter->shutdown

Shuts down the adapter in whatever manner is appropriate at the end of the
lifetime of the containing program; or at least, at the point when the program
has finished using the connected device.

This method is allowed to block; it does not yield a L<Future>. It is suitable
to call from a C<DESTROY> method or C<END> block.

=cut

=head1 PROTOCOLS

The following methods are common to all protocol instances:

=head2 power

   $protocol->power( $on )->get

Switches on or off the power to the actual chip or module, if such ability is
provided by the adapter.

=cut

=head1 SPI PROTOCOL

=head2 configure

   $spi->configure( %args )

Sets up the configuration of the SPI communication. The following arguments
are recognised:

=over 4

=item mode => 0 | 1 | 2 | 3

The numbered SPI mode used to communicate with the chip.

=item max_bitrate => INT

The highest speed, in bits per second, that the chip can accept. The adapter
must pick a rate that is no higher than this. Note specifically that not all
adapters are able to choose a rate arbitrarily, and so the actually
communication may happen at some rate slower than this.

=back

=head2 readwrite

   $bytes_in = $spi->readwrite( $bytes_out )->get

Performs a complete SPI transaction; synchronously clocking the data given by
the plain byte string I<$bytes_out> out of the MOSI pin of the adapter, while
simultaneously capturing the data coming in to the MISO pin, which is
eventually returned as the result of the returned future.

=head2 write

   $spi->write( $bytes )->get

A variant of C<readwrite> where the caller does not intend to make use of the
data returned by the device, and so the adapter does not need to return it.
This may or may not make a material difference to the actual communication
with the adapter or device; it could be implemented simply by calling the
C<readwrite> method and ignoring the return value.

=cut

=head1 AUTHOR

Paul Evans <leonerd@leonerd.org.uk>

=cut

0x55AA;
