package BorderStyle::ASCII::SingleLineDoubleAfterHeader;

use strict;
use warnings;
use parent 'BorderStyleBase';

our $AUTHORITY = 'cpan:PERLANCAR'; # AUTHORITY
our $DATE = '2022-01-24'; # DATE
our $DIST = 'BorderStyles-Standard'; # DIST
our $VERSION = '0.008'; # VERSION

our %BORDER = (
    v => 2,
    summary => 'Just like ASCII::SingleLine but uses double line to separate header row and first data row',
    dynamic => 1,
    chars => [
        ['.','-','+','.'], # 0
        ['|','|','|'],     # 1
        ['+','=','+','+', '+','+'], # 2
        ['|','|','|'],     # 3
        ['+','-','+','+', '+','+','+','+'], # 4 - separator between data rows
        ['`','-','+',"'"], # 5

        [], # 6
        [], # 7

        ['+','=','+','+', '+','+','+','+'], # 8 - separator between header rows
    ],
    args => {
        header_row_count => {
            summary => 'Number of header rows',
            schema => 'uint*',
            default => 1,
            description => <<'_',

This argument is useful if you have multiple header rows.

_
        },
    },
);

sub get_border_char {
    my ($self, $y, $x, $n, $args) = @_;
    $args = {} unless defined $args;

    my $in_header = defined $args->{rownum} ?
        $args->{rownum} < $self->{args}{header_row_count} :
        ($y >= 0 && $y <= 2 || $y == 8);
    #use DD; dd $self->{args}; dd $args; dd {in_header=>$in_header, y=>$y, x=>$x, n=>$n};
    if ($in_header) {
        if ($y == 3) { $y = 1 }
        elsif ($y == 4) { $y = 8 }
    }
    #use DD; dd {y=>$y, x=>$x, n=>$n};

    $self->SUPER::get_border_char($y, $x, $n, $args);
}

1;
# ABSTRACT: Just like ASCII::SingleLine but uses double line to separate header row and first data row

__END__

=pod

=encoding UTF-8

=head1 NAME

BorderStyle::ASCII::SingleLineDoubleAfterHeader - Just like ASCII::SingleLine but uses double line to separate header row and first data row

=head1 VERSION

This document describes version 0.008 of BorderStyle::ASCII::SingleLineDoubleAfterHeader (from Perl distribution BorderStyles-Standard), released on 2022-01-24.

=head1 SYNOPSIS

To use with L<Text::ANSITable>:

 use Text::ANSITable;
 my $rows =
   [
     ["ColumName1", "ColumnNameB", "ColumnNameC"],
     ["row1A", "row1B", "row1C"],
     ["row2A", "row2B", "row2C"],
     ["row3A", "row3B", "row3C"],
   ];
 my $t = Text::ANSITable->new;
 $t->border_style("ASCII::SingleLineDoubleAfterHeader");
 $t->columns($rows->[0]);
 $t->add_row($rows->[$_]) for 1 .. $#{ $rows };
 print $t->draw;


Sample output:

 .------------+-------------+-------------.
 | ColumName1 | ColumnNameB | ColumnNameC |
 +============+=============+=============+
 | row1A      | row1B       | row1C       |
 | row2A      | row2B       | row2C       |
 | row3A      | row3B       | row3C       |
 `------------+-------------+-------------'

To use with L<Text::Table::More>:

 use Text::Table::More qw/generate_table/;
 my $rows =
   [
     ["ColumName1", "ColumnNameB", "ColumnNameC"],
     ["row1A", "row1B", "row1C"],
     ["row2A", "row2B", "row2C"],
     ["row3A", "row3B", "row3C"],
   ];
 generate_table(rows=>$rows, header_row=>1, separate_rows=>1, border_style=>"ASCII::SingleLineDoubleAfterHeader");


Sample output:

 .------------+-------------+-------------.
 | ColumName1 | ColumnNameB | ColumnNameC |
 +============+=============+=============+
 | row1A      | row1B       | row1C       |
 +------------+-------------+-------------+
 | row2A      | row2B       | row2C       |
 +------------+-------------+-------------+
 | row3A      | row3B       | row3C       |
 `------------+-------------+-------------'
 

To use with L<Text::Table::TinyBorderStyle>:

 use Text::Table::TinyBorderStyle qw/generate_table/;
 my $rows =
   [
     ["ColumName1", "ColumnNameB", "ColumnNameC"],
     ["row1A", "row1B", "row1C"],
     ["row2A", "row2B", "row2C"],
     ["row3A", "row3B", "row3C"],
   ];
 generate_table(rows=>$rows, header_row=>1, separate_rows=>1, border_style=>"BorderStyle::ASCII::SingleLineDoubleAfterHeader");


Sample output:

 .------------+-------------+-------------.
 | ColumName1 | ColumnNameB | ColumnNameC |
 +============+=============+=============+
 | row1A      | row1B       | row1C       |
 +------------+-------------+-------------+
 | row2A      | row2B       | row2C       |
 +------------+-------------+-------------+
 | row3A      | row3B       | row3C       |
 `------------+-------------+-------------'

=head1 HOMEPAGE

Please visit the project's homepage at L<https://metacpan.org/release/BorderStyles-Standard>.

=head1 SOURCE

Source repository is at L<https://github.com/perlancar/perl-BorderStyles-Standard>.

=head1 SEE ALSO

L<BorderStyle::Custom>

L<BorderStyle::UTF8::SingleLineDoubleAfterHeader>

L<BorderStyle::UTF8::SingleLine>

L<BorderStyle::BoxChar::SingleLine>

L<BorderStyle::UTF8::SingleLine>

=head1 AUTHOR

perlancar <perlancar@cpan.org>

=head1 CONTRIBUTING


To contribute, you can send patches by email/via RT, or send pull requests on
GitHub.

Most of the time, you don't need to build the distribution yourself. You can
simply modify the code, then test via:

 % prove -l

If you want to build the distribution (e.g. to try to install it locally on your
system), you can install L<Dist::Zilla>,
L<Dist::Zilla::PluginBundle::Author::PERLANCAR>, and sometimes one or two other
Dist::Zilla plugin and/or Pod::Weaver::Plugin. Any additional steps required
beyond that are considered a bug and can be reported to me.

=head1 COPYRIGHT AND LICENSE

This software is copyright (c) 2022, 2021, 2020 by perlancar <perlancar@cpan.org>.

This is free software; you can redistribute it and/or modify it under
the same terms as the Perl 5 programming language system itself.

=head1 BUGS

Please report any bugs or feature requests on the bugtracker website L<https://rt.cpan.org/Public/Dist/Display.html?Name=BorderStyles-Standard>

When submitting a bug or request, please include a test-file or a
patch to an existing test-file that illustrates the bug or desired
feature.

=cut
