package App::DateUtils;

our $DATE = '2016-06-19'; # DATE
our $VERSION = '0.05'; # VERSION

use 5.010001;
use strict;
use warnings;

our %SPEC;

my %time_zone_arg = (
    time_zone => {
        schema => 'str*',
        cmdline_aliases => {timezone=>{}},
    },
);

my %dates_arg = (
    dates => {
        schema => ['array*', of=>'str*', min_len=>1],
        'x.name.is_plural' => 1,
        req => 1,
        pos => 0,
        greedy => 1,
    },
);

my %durations_arg = (
    durations => {
        schema => ['array*', of=>'str*', min_len=>1],
        'x.name.is_plural' => 1,
        req => 1,
        pos => 0,
        greedy => 1,
    },
);

$SPEC{parse_date} = {
    v => 1.1,
    summary => 'Parse date string(s) using one of several modules',
    args => {
        module => {
            schema  => ['str*', in=>[
                'DateTime::Format::Alami::EN',
                'DateTime::Format::Alami::ID',
                'DateTime::Format::Natural',
            ]],
            default => 'DateTime::Format::Natural',
            cmdline_aliases => {m=>{}},
        },
        %time_zone_arg,
        %dates_arg,
    },
    examples => [
        {
            argv => ['23 sep 2015','tomorrow','foo'],
        },
    ],
};
sub parse_date {
    my %args = @_;

    my $mod = $args{module};

    my $parser;
    if ($mod eq 'DateTime::Format::Alami::EN') {
        require DateTime::Format::Alami::EN;
        $parser = DateTime::Format::Alami::EN->new(
            ( time_zone => $args{time_zone} ) x !!(defined($args{time_zone})),
        );
    } elsif ($mod eq 'DateTime::Format::Alami::ID') {
        require DateTime::Format::Alami::ID;
        $parser = DateTime::Format::Alami::ID->new(
            ( time_zone => $args{time_zone} ) x !!(defined($args{time_zone})),
        );
    } elsif ($mod eq 'DateTime::Format::Natural') {
        require DateTime::Format::Natural;
        $parser = DateTime::Format::Natural->new(
            ( time_zone => $args{time_zone} ) x !!(defined($args{time_zone})),
        );
    } else {
        return [400, "Unknown module '$mod'"];
    }

    my @res;
    for my $date (@{ $args{dates} }) {
        my $rec = { original => $date };
        if ($mod =~ /^DateTime::Format::(Alami|Natural)/) {
            my $dt = $parser->parse_datetime($date);
            my $success = $mod =~ /Alami/ ? $dt : $parser->success;

            if ($success) {
                $rec->{is_parseable} = 1;
                $rec->{as_epoch} = $dt->epoch;
                $rec->{as_datetime_obj} = "$dt";
            } else {
                $rec->{is_parseable} = 0;
            }
        }
        push @res, $rec;
    }
    [200, "OK", \@res];
}

$SPEC{parse_date_using_df_natural} = {
    v => 1.1,
    summary => 'Parse date string(s) using DateTime::Format::Natural',
    args => {
        %time_zone_arg,
        %dates_arg,
    },
};
sub parse_date_using_df_natural {
    my %args = @_;
    parse_date(module=>'DateTime::Format::Natural', %args);
}

$SPEC{parse_date_using_df_alami_en} = {
    v => 1.1,
    summary => 'Parse date string(s) using DateTime::Format::Alami::EN',
    args => {
        %time_zone_arg,
        %dates_arg,
    },
};
sub parse_date_using_df_alami_en {
    my %args = @_;
    parse_date(module=>'DateTime::Format::Alami::EN', %args);
}

$SPEC{parse_date_using_df_alami_id} = {
    v => 1.1,
    summary => 'Parse date string(s) using DateTime::Format::Alami::ID',
    args => {
        %time_zone_arg,
        %dates_arg,
    },
};
sub parse_date_using_df_alami_id {
    my %args = @_;
    parse_date(module=>'DateTime::Format::Alami::ID', %args);
}

$SPEC{parse_duration} = {
    v => 1.1,
    summary => 'Parse duration string(s) using one of several modules',
    args => {
        module => {
            schema  => ['str*', in=>[
                'DateTime::Format::Duration',
                'Time::Duration::Parse',
            ]],
            default => 'Time::Duration::Parse',
            cmdline_aliases => {m=>{}},
        },
        %durations_arg,
    },
};
sub parse_duration {
    my %args = @_;

    my $mod = $args{module};

    my $parser;
    if ($mod eq 'DateTime::Format::Duration') {
        require DateTime::Format::Duration;
        $parser = DateTime::Format::Duration->new(
        );
    } elsif ($mod eq 'Time::Duration::Parse') {
        require Time::Duration::Parse;
    }

    my @res;
    for my $dur (@{ $args{durations} }) {
        my $rec = { original => $dur };
        if ($mod eq 'DateTime::Format::Duration') {
            my $dtdur = $parser->parse_duration($dur);
            if ($dtdur) {
                $rec->{is_parseable} = 1;
                $rec->{as_dtdur_obj} = "$dtdur";
                $rec->{as_secs} = $dtdur->in_units('seconds');
            } else {
                $rec->{is_parseable} = 0;
            }
        } elsif ($mod eq 'Time::Duration::Parse') {
            my $secs;
            eval { $secs = Time::Duration::Parse::parse_duration($dur) };
            if ($@) {
                $rec->{is_parseable} = 0;
                $rec->{error_msg} = $@;
                $rec->{error_msg} =~ s/\n+/ /g;
            } else {
                $rec->{is_parseable} = 1;
                $rec->{as_secs} = $secs;
            }
        }
        push @res, $rec;
    }
    [200, "OK", \@res];
}

$SPEC{parse_duration_using_df_duration} = {
    v => 1.1,
    summary => 'Parse date string(s) using DateTime::Format::Duration',
    args => {
        %durations_arg,
    },
};
sub parse_duration_using_df_duration {
    my %args = @_;
    parse_duration(module=>'DateTime::Format::Duration', %args);
}

$SPEC{dateconv} = {
    v => 1.1,
    summary => 'Convert date to another format',
    args => {
        date => {
            schema => ['date*', {
                'x.perl.coerce_to' => 'DateTime',
                'x.perl.coerce_rules' => ['str_alami'],
            }],
            req => 1,
            pos => 0,
        },
        to => {
            schema => ['str*', in=>[qw/epoch ymd/]], # XXX: iso8601, ...
            default => 'epoch',
        },
    },
    result_naked => 1,
    examples => [
        {
            summary => 'Convert "today" to epoch',
            args => {date => 'today'},
            test => 0,
        },
        {
            summary => 'Convert epoch to ymd',
            args => {date => '1463702400', to=>'ymd'},
            result => '2016-05-20',
        },
    ],
};
sub dateconv {
    my %args = @_;
    my $date = $args{date};
    my $to   = $args{to};

    if ($to eq 'epoch') {
        return $date->epoch;
    } elsif ($to eq 'ymd') {
        return $date->ymd;
    } else {
        die "Unknown format '$to'";
    }
}

$SPEC{durconv} = {
    v => 1.1,
    summary => 'Convert duration to another format',
    args => {
        duration => {
            schema => ['duration*', {
                'x.perl.coerce_to' => 'DateTime::Duration',
            }],
            req => 1,
            pos => 0,
        },
        to => {
            schema => ['str*', in=>[qw/secs hash/]], # XXX: iso8601, ...
            default => 'secs',
        },
    },
    result_naked => 1,
    examples => [
        {
            summary => 'Convert "3h2m" to number of seconds',
            args => {duration => '3h2m'},
            result => 10920,
        },
    ],
};
sub durconv {
    my %args = @_;
    my $dur = $args{duration};
    my $to  = $args{to};

    if ($to eq 'secs') {
        # approximation
        return (
            $dur->years       * 365*86400 +
            $dur->months      *  30*86400 +
            $dur->weeks       *   7*86400 +
            $dur->days        *     86400 +
            $dur->hours       *      3600 +
            $dur->minutes     *        60 +
            $dur->seconds     *         1 +
            $dur->nanoseconds *      1e-9
        );
    } elsif ($to eq 'hash') {
        my $h = {
            years => $dur->years,
            months => $dur->months,
            weeks => $dur->weeks,
            days => $dur->days,
            hours => $dur->hours,
            minutes => $dur->minutes,
            seconds => $dur->seconds,
            nanoseconds => $dur->nanoseconds,
        };
        for (keys %$h) {
            delete $h->{$_} if $h->{$_} == 0;
        }
        return $h;
    } else {
        die "Unknown format '$to'";
    }
}

1;
# ABSTRACT: An assortment of date-/time-related CLI utilities

__END__

=pod

=encoding UTF-8

=head1 NAME

App::DateUtils - An assortment of date-/time-related CLI utilities

=head1 VERSION

This document describes version 0.05 of App::DateUtils (from Perl distribution App-DateUtils), released on 2016-06-19.

=head1 SYNOPSIS

This distribution provides the following command-line utilities related to
date/time:

=over

=item * L<dateconv>

=item * L<durconv>

=item * L<parse-date>

=item * L<parse-date-using-df-alami-en>

=item * L<parse-date-using-df-alami-id>

=item * L<parse-date-using-df-natural>

=item * L<parse-duration>

=item * L<parse-duration-using-df-duration>

=back

=head1 FUNCTIONS


=head2 dateconv(%args) -> any

Convert date to another format.

Examples:

=over

=item * Convert "today" to epoch:

 dateconv(date => "today"); # -> 1466294400

=item * Convert epoch to ymd:

 dateconv(date => 1463702400, to => "ymd"); # -> "2016-05-20"

=back

This function is not exported.

Arguments ('*' denotes required arguments):

=over 4

=item * B<date>* => I<date>

=item * B<to> => I<str> (default: "epoch")

=back

Return value:  (any)


=head2 durconv(%args) -> any

Convert duration to another format.

Examples:

=over

=item * Convert "3h2m" to number of seconds:

 durconv(duration => "3h2m"); # -> 10920

=back

This function is not exported.

Arguments ('*' denotes required arguments):

=over 4

=item * B<duration>* => I<duration>

=item * B<to> => I<str> (default: "secs")

=back

Return value:  (any)


=head2 parse_date(%args) -> [status, msg, result, meta]

Parse date string(s) using one of several modules.

Examples:

=over

=item * Example #1:

 parse_date( dates => ["23 sep 2015", "tomorrow", "foo"]);

Result:

 [
   200,
   "OK",
   [
     {
       as_datetime_obj => "2015-09-23T00:00:00",
       as_epoch        => 1442966400,
       is_parseable    => 1,
       original        => "23 sep 2015",
     },
     {
       as_datetime_obj => "2016-06-20T00:00:00",
       as_epoch        => 1466380800,
       is_parseable    => 1,
       original        => "tomorrow",
     },
     { is_parseable => 0, original => "foo" },
   ],
   {},
 ]

=back

This function is not exported.

Arguments ('*' denotes required arguments):

=over 4

=item * B<dates>* => I<array[str]>

=item * B<module> => I<str> (default: "DateTime::Format::Natural")

=item * B<time_zone> => I<str>

=back

Returns an enveloped result (an array).

First element (status) is an integer containing HTTP status code
(200 means OK, 4xx caller error, 5xx function error). Second element
(msg) is a string containing error message, or 'OK' if status is
200. Third element (result) is optional, the actual result. Fourth
element (meta) is called result metadata and is optional, a hash
that contains extra information.

Return value:  (any)


=head2 parse_date_using_df_alami_en(%args) -> [status, msg, result, meta]

Parse date string(s) using DateTime::Format::Alami::EN.

This function is not exported.

Arguments ('*' denotes required arguments):

=over 4

=item * B<dates>* => I<array[str]>

=item * B<time_zone> => I<str>

=back

Returns an enveloped result (an array).

First element (status) is an integer containing HTTP status code
(200 means OK, 4xx caller error, 5xx function error). Second element
(msg) is a string containing error message, or 'OK' if status is
200. Third element (result) is optional, the actual result. Fourth
element (meta) is called result metadata and is optional, a hash
that contains extra information.

Return value:  (any)


=head2 parse_date_using_df_alami_id(%args) -> [status, msg, result, meta]

Parse date string(s) using DateTime::Format::Alami::ID.

This function is not exported.

Arguments ('*' denotes required arguments):

=over 4

=item * B<dates>* => I<array[str]>

=item * B<time_zone> => I<str>

=back

Returns an enveloped result (an array).

First element (status) is an integer containing HTTP status code
(200 means OK, 4xx caller error, 5xx function error). Second element
(msg) is a string containing error message, or 'OK' if status is
200. Third element (result) is optional, the actual result. Fourth
element (meta) is called result metadata and is optional, a hash
that contains extra information.

Return value:  (any)


=head2 parse_date_using_df_natural(%args) -> [status, msg, result, meta]

Parse date string(s) using DateTime::Format::Natural.

This function is not exported.

Arguments ('*' denotes required arguments):

=over 4

=item * B<dates>* => I<array[str]>

=item * B<time_zone> => I<str>

=back

Returns an enveloped result (an array).

First element (status) is an integer containing HTTP status code
(200 means OK, 4xx caller error, 5xx function error). Second element
(msg) is a string containing error message, or 'OK' if status is
200. Third element (result) is optional, the actual result. Fourth
element (meta) is called result metadata and is optional, a hash
that contains extra information.

Return value:  (any)


=head2 parse_duration(%args) -> [status, msg, result, meta]

Parse duration string(s) using one of several modules.

This function is not exported.

Arguments ('*' denotes required arguments):

=over 4

=item * B<durations>* => I<array[str]>

=item * B<module> => I<str> (default: "Time::Duration::Parse")

=back

Returns an enveloped result (an array).

First element (status) is an integer containing HTTP status code
(200 means OK, 4xx caller error, 5xx function error). Second element
(msg) is a string containing error message, or 'OK' if status is
200. Third element (result) is optional, the actual result. Fourth
element (meta) is called result metadata and is optional, a hash
that contains extra information.

Return value:  (any)


=head2 parse_duration_using_df_duration(%args) -> [status, msg, result, meta]

Parse date string(s) using DateTime::Format::Duration.

This function is not exported.

Arguments ('*' denotes required arguments):

=over 4

=item * B<durations>* => I<array[str]>

=back

Returns an enveloped result (an array).

First element (status) is an integer containing HTTP status code
(200 means OK, 4xx caller error, 5xx function error). Second element
(msg) is a string containing error message, or 'OK' if status is
200. Third element (result) is optional, the actual result. Fourth
element (meta) is called result metadata and is optional, a hash
that contains extra information.

Return value:  (any)

=head1 HOMEPAGE

Please visit the project's homepage at L<https://metacpan.org/release/App-DateUtils>.

=head1 SOURCE

Source repository is at L<https://github.com/perlancar/perl-App-DateUtils>.

=head1 BUGS

Please report any bugs or feature requests on the bugtracker website L<https://rt.cpan.org/Public/Dist/Display.html?Name=App-DateUtils>

When submitting a bug or request, please include a test-file or a
patch to an existing test-file that illustrates the bug or desired
feature.

=head1 SEE ALSO

L<App::datecalc>

=head1 AUTHOR

perlancar <perlancar@cpan.org>

=head1 COPYRIGHT AND LICENSE

This software is copyright (c) 2016 by perlancar@cpan.org.

This is free software; you can redistribute it and/or modify it under
the same terms as the Perl 5 programming language system itself.

=cut
