package App::CreateAcmeCPANListsImportModules;

our $DATE = '2016-02-21'; # DATE
our $VERSION = '0.01'; # VERSION

use 5.010001;
use strict;
use warnings;

use Log::Any::IfLOG '$log';

our %SPEC;

$SPEC{create_acme_cpanlists_import_modules} = {
    v => 1.1,
    summary => 'Create Acme::CPANLists::Import::* modules',
    description => <<'_',

An `Acme::CPANLists::Import::*` module contains a module list where its entries
(modules) are extracted from a web page. The purpose of creating such module is
to have a POD mentioning the modules, thus adding/improving to the POD "mentions
cloud" on CPAN.

_
    args => {
        modules => {
            schema => ['array*', of=>'hash*'],
            req => 1,
        },
        namespace => {
            schema => 'str*',
            req => 1,
        },
        cache => {
            schema => 'bool',
            default => 1,
        },
        user_agent => {
            summary => 'Set HTTP User-Agent',
            schema => 'str*',
        },
        dist_dir => {
            schema => 'str*',
        },
    },
};
sub create_acme_cpanlists_import_modules {
    require Data::Dmp;
    require File::Slurper;
    require HTML::Extract::CPANModules;
    require LWP::UserAgent;
    require POSIX;

    my %args = @_;

    my $modules = $args{modules};
    my $namespace = $args{namespace};
    my $dist_dir = $args{dist_dir} // do { require Cwd; Cwd::get_cwd() };
    my $cache = $args{cache} // 1;

    my $namespace_pm = $namespace; $namespace_pm =~ s!::!/!g;

    my $ua = LWP::UserAgent->new;
    $ua->agent($args{user_agent}) if $args{user_agent};

    my $now = time();

    for my $mod (@$modules) {
        $log->infof("Processing %s ...", $mod->{name});

        my $cache_path = "$dist_dir/devdata/$mod->{name}.html";
        my @st_cache = stat $cache_path;
        my $content;
        my $date;
        if (!$cache || !@st_cache || $st_cache[9] < $now-30*86400) {
            $log->infof("Retrieving %s ...", $mod->{url});
            my $resp = $ua->get($mod->{url});
            $resp->is_success or return [500, "Can't get $mod->{url}: ".$resp->status_line];
            $content = $resp->content;
            File::Slurper::write_text($cache_path, $content);
            $date = POSIX::strftime("%Y-%m-%d", localtime $now);
        } else {
            $log->infof("Using cache file %s", $cache_path);
            $content = File::Slurper::read_text($cache_path);
            $date = POSIX::strftime("%Y-%m-%d", localtime($st_cache[9]));
        }

        my $mods = HTML::Extract::CPANModules::extract_cpan_modules_from_html(
            html => $content, %{ $mod->{extract_opts} // {}});

        my $module_path = "$dist_dir/lib/$namespace_pm/$mod->{name}.pm";

        my $mod_list = {
            summary => $mod->{summary},
            description => "This list is generated by extracting module names mentioned in the article [$mod->{url}] (retrieved on $date). For the full article, visit the URL.",
            entries => [map {+{module=>$_}} @$mods],
        };

        my @pm_content = (
            "package $namespace\::$mod->{name};\n",
            "\n",
            "# DATE\n",
            "# VERSION\n",
            "\n",
            "our \@Module_Lists = (", Data::Dmp::dmp($mod_list), ");\n",
            "\n",
            "1;\n",
            "# ABSTRACT: $mod->{summary}\n",
            "\n",
            "=head1 DESCRIPTION\n",
            "\n",
            "This module is generated by extracting module names mentioned in the article L<$mod->{url}> (retrieved on $date). For the full article, visit the URL.\n",
            "\n",
        );

        $log->infof("Writing module %s ...", $module_path);
        File::Slurper::write_text($module_path, join("", @pm_content));
    }

    [200];
}

1;
# ABSTRACT: Create Acme::CPANLists::Import::* modules

__END__

=pod

=encoding UTF-8

=head1 NAME

App::CreateAcmeCPANListsImportModules - Create Acme::CPANLists::Import::* modules

=head1 VERSION

This document describes version 0.01 of App::CreateAcmeCPANListsImportModules (from Perl distribution App-CreateAcmeCPANListsImportModules), released on 2016-02-21.

=head1 FUNCTIONS


=head2 create_acme_cpanlists_import_modules(%args) -> [status, msg, result, meta]

Create Acme::CPANLists::Import::* modules.

An C<Acme::CPANLists::Import::*> module contains a module list where its entries
(modules) are extracted from a web page. The purpose of creating such module is
to have a POD mentioning the modules, thus adding/improving to the POD "mentions
cloud" on CPAN.

This function is not exportable.

Arguments ('*' denotes required arguments):

=over 4

=item * B<cache> => I<bool> (default: 1)

=item * B<dist_dir> => I<str>

=item * B<modules>* => I<array[hash]>

=item * B<namespace>* => I<str>

=item * B<user_agent> => I<str>

Set HTTP User-Agent.

=back

Returns an enveloped result (an array).

First element (status) is an integer containing HTTP status code
(200 means OK, 4xx caller error, 5xx function error). Second element
(msg) is a string containing error message, or 'OK' if status is
200. Third element (result) is optional, the actual result. Fourth
element (meta) is called result metadata and is optional, a hash
that contains extra information.

Return value:  (any)

=head1 HOMEPAGE

Please visit the project's homepage at L<https://metacpan.org/release/App-CreateAcmeCPANListsImportModules>.

=head1 SOURCE

Source repository is at L<https://github.com/perlancar/perl-App-CreateAcmeCPANListsImportModules>.

=head1 BUGS

Please report any bugs or feature requests on the bugtracker website L<https://rt.cpan.org/Public/Dist/Display.html?Name=App-CreateAcmeCPANListsImportModules>

When submitting a bug or request, please include a test-file or a
patch to an existing test-file that illustrates the bug or desired
feature.

=head1 SEE ALSO

L<Acme::CPANLists>

Some C<Acme::CPANLists::Import::*> modules which utilize this during building:
L<Acme::CPANLists::Import::NEILB>, L<Acme::CPANLists::Import::SHARYANTO>,
L<Acme::CPANLists::Import::RSAVAGE>, L<Acme::CPANLists::Import>, and so on.

L<App::lcpan>, L<lcpan>, especially the B<related-mods> subcommand.

=head1 AUTHOR

perlancar <perlancar@cpan.org>

=head1 COPYRIGHT AND LICENSE

This software is copyright (c) 2016 by perlancar@cpan.org.

This is free software; you can redistribute it and/or modify it under
the same terms as the Perl 5 programming language system itself.

=cut
