package Net::DNS::SEC::DSA;

#
# $Id: DSA.pm 1669 2018-04-27 10:17:13Z willem $
#
our $VERSION = (qw$LastChangedRevision: 1669 $)[1];


=head1 NAME

Net::DNS::SEC::DSA - DNSSEC DSA digital signature algorithm


=head1 SYNOPSIS

    require Net::DNS::SEC::DSA;

    $signature = Net::DNS::SEC::DSA->sign( $sigdata, $private );

    $validated = Net::DNS::SEC::DSA->verify( $sigdata, $keyrr, $sigbin );


=head1 DESCRIPTION

Implementation of DSA digital signature
generation and verification procedures.

=head2 sign

    $signature = Net::DNS::SEC::DSA->sign( $sigdata, $private );

Generates the wire-format signature from the sigdata octet string
and the appropriate private key object.

=head2 verify

    $validated = Net::DNS::SEC::DSA->verify( $sigdata, $keyrr, $sigbin );

Verifies the signature over the sigdata octet string using the specified
public key resource record.

=cut

use strict;
use integer;
use warnings;
use MIME::Base64;

my %DSA = (
	3 => [sub { Net::DNS::SEC::libcrypto::EVP_sha1() }],
	6 => [sub { Net::DNS::SEC::libcrypto::EVP_sha1() }],
	);


sub sign {
	my ( $class, $sigdata, $private ) = @_;

	my $algorithm = $private->algorithm;
	my ($evpmd) = @{$DSA{$algorithm} || []};
	die 'private key not DSA' unless $evpmd;

	my ( $p, $q, $g, $x, $y ) = map decode_base64( $private->$_ ),
			qw(prime subprime base private_value public_value);
	my $t = ( length($g) - 64 ) / 8;

	my $dsa = Net::DNS::SEC::libcrypto::DSA_new();
	Net::DNS::SEC::libcrypto::DSA_set0_pqg( $dsa, $p, $q, $g );
	Net::DNS::SEC::libcrypto::DSA_set0_key( $dsa, $y, $x );

	my $evpkey = Net::DNS::SEC::libcrypto::EVP_PKEY_new();
	Net::DNS::SEC::libcrypto::EVP_PKEY_assign_DSA( $evpkey, $dsa );

	my $ASN1 = Net::DNS::SEC::libcrypto::EVP_sign( $sigdata, $evpkey, &$evpmd );
	my $l = 2 + unpack 'x3 C', $ASN1;
	pack 'C a* a*', $t, map _BERunwrap($_), unpack "x2 a$l a*", $ASN1;
}


sub verify {
	my ( $class, $sigdata, $keyrr, $sigbin ) = @_;

	my $algorithm = $keyrr->algorithm;
	my ($evpmd) = @{$DSA{$algorithm} || []};
	die 'public key not DSA' unless $evpmd;

	return unless $sigbin;

	my $key = $keyrr->keybin;				# public key
	my $len = 64 + 8 * unpack( 'C', $key );			# RFC2536, section 2
	my ( $q, $p, $g, $y ) = unpack "x a20 a$len a$len a$len", $key;

	my $dsa = Net::DNS::SEC::libcrypto::DSA_new();
	Net::DNS::SEC::libcrypto::DSA_set0_pqg( $dsa, $p, $q, $g );
	Net::DNS::SEC::libcrypto::DSA_set0_key( $dsa, $y, '' );

	my $evpkey = Net::DNS::SEC::libcrypto::EVP_PKEY_new();
	Net::DNS::SEC::libcrypto::EVP_PKEY_assign_DSA( $evpkey, $dsa );

	my ( $r, $s ) = map _BERwrap($_), unpack 'x a20 a20', $sigbin;
	my $ASN1 = pack 'n a* a*', 0x3000 + length($r) + length($s), $r, $s;

	Net::DNS::SEC::libcrypto::EVP_verify( $sigdata, $ASN1, $evpkey, &$evpmd );
}


sub _BERwrap {
	my $s = shift;
	$s =~ s/^\000+(?=.)//;
	$s =~ s/^(?=[\200-\377])/\000/;
	pack 'C2 a*', 2, length($s), $s;
}

sub _BERunwrap {
	my ( $l, $s ) = unpack 'x C a*', shift;
	substr pack( "x20 a$l", $s ), -20;
}


1;

__END__

########################################

=head1 ACKNOWLEDGMENT

Thanks are due to Eric Young and the many developers and 
contributors to the OpenSSL cryptographic library.


=head1 COPYRIGHT

Copyright (c)2014,2018 Dick Franks.

All rights reserved.


=head1 LICENSE

Permission to use, copy, modify, and distribute this software and its
documentation for any purpose and without fee is hereby granted, provided
that the above copyright notice appear in all copies and that both that
copyright notice and this permission notice appear in supporting
documentation, and that the name of the author not be used in advertising
or publicity pertaining to distribution of the software without specific
prior written permission.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL
THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
DEALINGS IN THE SOFTWARE.


=head1 SEE ALSO

L<Net::DNS>, L<Net::DNS::SEC>,
RFC2536,
L<OpenSSL|http://www.openssl.org/docs>

=cut

