#-*- Mode: CPerl -*-

## File: DDC::Client::Distributed.pm
## Author: Bryan Jurish <moocow@cpan.org>
## Description:
##  + DDC Query utilities: clients for distributed server
##======================================================================

package DDC::Client::Distributed;
use DDC::Client;
use Carp;
use Encode qw(encode decode);
use strict;

##======================================================================
## Globals
our @ISA = qw(DDC::Client);

##======================================================================
## Constructors etc

## $dc = $CLASS_OR_OBJ->new(%args)
##  + %args:
##    (
##     start=>$hit_start_index,     ##-- default=0,
##     limit=>$hit_limit_index,     ##-- default=10,
##     timeout=>$hit_timeout_index, ##-- default=60
##
##     connect=>\%connectArgs,      ##-- see DDC::Client->new(), IO::Socket::INET->new()
##    )
sub new {
  my $that = shift;
  return $that->SUPER::new(
			   ##-- defaults
			   start=>0,
			   limit=>10,
			   timeout=>60,

			   ##-- user args
			   @_,
			  );
}

##======================================================================
## Querying

## $buf = $dc->queryRaw($query_string)
## $buf = $dc->queryRaw(\@raw_strings)
sub queryRaw {
  my $dc = shift;
  my $buf = $dc->queryRawNC(@_);
  $dc->close(); ##-- this apparently has to happen: bummer
  return $buf;
}

## $buf = $dc->queryRawNC($query_string)
## $buf = $dc->queryRawNC(\@raw_strings)
##  + guts for queryRaw() without implicit close()
sub queryRawNC {
  my ($dc,$query) = @_;
  if (UNIVERSAL::isa($query,'ARRAY')) {
    ##-- raw array: send raw data to DDC
    $dc->send(join("\001",@$query));
  }
  elsif ($dc->{mode} =~ /^(?:raw|req)/i) {
    ##-- "raw" or "request" mode: send raw request to DDC
    $dc->send($query);
  }
  else {
    ##-- query string: send 'run-query Distributed'
    $dc->send(join("\001",
		   "run_query Distributed",
		   $query,
		   $dc->{mode},
		   "$dc->{start} $dc->{limit} $dc->{timeout}"));
  }
  ##-- get output buffer
  return $dc->readData();
}

## @bufs = $dc->queryMulti($queryString1, $queryString2, ...)
## @bufs = $dc->queryMulti(\@queryStrings1, \@queryStrings2, ...)
sub queryMulti {
  my $dc   = shift;
  my @bufs = map {$dc->queryRawNC($_)} @_;
  $dc->close(); ##-- this apparently has to happen: bummer
  return @bufs;
}

## $obj = $dc->queryJson($query_string)
## $obj = $dc->queryJson(\@raw_strings)
sub queryJson {
  my ($dc,$query) = @_;
  return $dc->decodeJson($dc->queryRaw($query));
}


## $hits = $dc->query($query_string)
sub query {
  my ($dc,$query) = @_;
  return $dc->parseData($dc->queryRaw($query));
}


1; ##-- be happy

__END__

##========================================================================
## POD DOCUMENTATION, auto-generated by podextract.perl

##========================================================================
## NAME
=pod

=head1 NAME

DDC::Client::Distributed - DDC Query utilities: client for distributed server

=cut

##========================================================================
## SYNOPSIS
=pod

=head1 SYNOPSIS

 ##========================================================================
 ## PRELIMINARIES

 use DDC::Client;
 use DDC::Client::Distributed;

 ##========================================================================
 ## Constructors etc

 $dc = $CLASS_OR_OBJ->new(%args);

 ##========================================================================
 ## Querying

 $buf  = $dc->queryRaw($query_string);
 $buf  = $dc->queryRawNC($query_string);
 @bufs = $dc->queryMulti(@query_strings);
 $hits = $dc->query($query_string);

 ## ... or any DDC::Client method ...

=cut

##========================================================================
## DESCRIPTION
=pod

=head1 DESCRIPTION

=cut

##----------------------------------------------------------------
## DESCRIPTION: DDC::Client::Distributed: Globals
=pod

=head2 Globals

=over 4

=item Variable: @ISA

DDC::Client::Distributed inherits from L<DDC::Client|DDC::Client>7.

=back

=cut

##----------------------------------------------------------------
## DESCRIPTION: DDC::Client::Distributed: Constructors etc
=pod

=head2 Constructors etc

=over 4

=item new

 $dc = $CLASS_OR_OBJ->new(%args);

=over 4

=item %args:

   (
    start=>$hit_start_index,     ##-- default=0,
    limit=>$hit_limit_index,     ##-- default=10,
    timeout=>$hit_timeout_index, ##-- default=60
    connect=>\%connectArgs,      ##-- see DDC::Client->new(), IO::Socket::INET->new()
   )

=back

=back

=cut

##----------------------------------------------------------------
## DESCRIPTION: DDC::Client::Distributed: Querying
=pod

=head2 Querying

=over 4

=item queryRaw

 $buf = $dc->queryRaw($query_string);

Send a query string to the selected server and returns the raw result buffer.
Implicitly close()s the connection.

=item queryRawNC

 $buf = $dc->queryRawNC($query_string);

Send a query string to the selected server and returns the raw result buffer.
No implicit close().

=item queryMulti

 @bufs = $dc->queryMulti(@query_strings);

Sends a series of query strings or requests to the server, and returns a list of raw result buffers.
Implicitly close()s the client after all requests have been sent,
but not between individual requests.

=item query

 $hits = $dc->query($query_string);

Send a query string to the selected server and parses the result into a list of hits.

=back

=cut

##========================================================================
## END POD DOCUMENTATION, auto-generated by podextract.perl

##======================================================================
## Footer
##======================================================================

=pod

=head1 AUTHOR

Bryan Jurish E<lt>moocow@cpan.orgE<gt>

=head1 COPYRIGHT AND LICENSE

Copyright (C) 2006-2016 by Bryan Jurish

This package is free software; you can redistribute it and/or modify
it under the same terms as Perl itself, either Perl version 5.14.2 or,
at your option, any later version of Perl 5 you may have available.

=cut
