use 5.008001;
use strict;
use warnings;

package BSON::Double;
# ABSTRACT: BSON type wrapper for Double

use version;
our $VERSION = 'v0.999.1'; # TRIAL

use Carp;

#pod =attr value
#pod
#pod A numeric scalar (or the special strings "Inf", "-Inf" or "NaN").  This
#pod will be coerced to Perl's numeric type.  The default is 0.0.
#pod
#pod =cut

use Moo;

has 'value' => (
    is => 'ro'
);

use namespace::clean -except => 'meta';

sub BUILD {
    my $self = shift;
    # coerce to NV internally
    $self->{value} = defined( $self->{value} ) ? $self->{value} / 1.0 : 0.0;
}

#pod =method TO_JSON
#pod
#pod Returns a double, unless the value is 'Inf', '-Inf' or 'NaN'
#pod (which are illegal in JSON), in which case an exception is thrown.
#pod
#pod =cut

sub TO_JSON {
    my $copy = "$_[0]->{value}"; # avoid changing value to PVNV
    return $_[0]->{value}/1.0 unless $copy =~ /^(?:Inf|-Inf|NaN)/i;

    croak( "The value '$copy' is illegal in JSON" );
}

use overload (
    q{0+}    => sub { $_[0]->{value} },
    fallback => 1,
);

1;

=pod

=encoding UTF-8

=head1 NAME

BSON::Double - BSON type wrapper for Double

=head1 VERSION

version v0.999.1

=head1 SYNOPSIS

    use BSON::Types ':all';

    my $bytes = bson_double( $number );

=head1 DESCRIPTION

This module provides a BSON type wrapper for a numeric value that
would be represented in BSON as a double.

=head1 ATTRIBUTES

=head2 value

A numeric scalar (or the special strings "Inf", "-Inf" or "NaN").  This
will be coerced to Perl's numeric type.  The default is 0.0.

=head1 METHODS

=head2 TO_JSON

Returns a double, unless the value is 'Inf', '-Inf' or 'NaN'
(which are illegal in JSON), in which case an exception is thrown.

=for Pod::Coverage BUILD

=head1 OVERLOADING

The numification operator, C<0+> is overloaded to return the C<value>
and fallback overloading is enabled.

=head1 AUTHORS

=over 4

=item *

David Golden <david@mongodb.com>

=item *

Stefan G. <minimalist@lavabit.com>

=back

=head1 COPYRIGHT AND LICENSE

This software is Copyright (c) 2016 by Stefan G. and MongoDB, Inc.

This is free software, licensed under:

  The Apache License, Version 2.0, January 2004

=cut

__END__


# vim: set ts=4 sts=4 sw=4 et tw=75:
