package Test::Pod::Content;
use strict;
use warnings;
use base qw(Pod::Simple Test::More);
use Exporter;
use version; our $VERSION = qv(0.0.2);

our @EXPORT = qw(pod_section_is pod_section_like);

# Globals for running a simple state machine
my $_state = q{};
my $_section = q{};
my $_content = q{};
my $_test_content_sub;

# cleanup everything once we've run our test
sub _reset { 
    my $parser = shift;
    $_state = q{};
    $_section = q{};
    $_content = q{};

    # source_dead is not reliable - just die to force terminating the 
    # parser run
    $parser->source_dead(1);
    die "OK\n";
}

sub pod_section_is {
    my ($name, $section, $content, $comment ) = @_;

    my $found = 0;

    $_test_content_sub = sub {
        my ($parser, $section_name, $test_content) = @_;
        if ($section_name eq $section) {
            $found++;
            Test::More::is($test_content, $content, $comment);
            _reset($parser);
        }
    };

    eval { Test::Pod::Content->filter( _find_file($name) ) };
    if ($@) { die $@ if ($@ !~m{^OK\n$}) };

    Test::More::fail $comment if not $found;
}

sub pod_section_like {
    my ($name, $section, $regex, $comment ) = @_;

    my $found = 0;

    $_test_content_sub = sub {
        my ($parser, $section_name, $test_content) = @_;
        if ($section_name eq $section) {
            $found++;
            Test::More::like($test_content, $regex, $comment);
            _reset($parser);
        }
    };

    eval { Test::Pod::Content->filter( _find_file($name) ) };
    if ($@) { die $@ if ($@ !~m{^OK\n$}) };

    Test::More::fail $comment if not $found;
}

sub _find_file {
    my $name = shift;
    return $name if (-e $name);
    for my $path (@INC) {
        return $path . '/' . $name if -e $path . '/' . $name;
    }
    $name =~s{::}{/}g;
    $name .= '.pm';
    for my $path (@INC) {
        return $path . '/' . $name if -e $path . '/' . $name;
    }
}

sub _handle_element_start {
    my($parser, $element_name, $attr_hash_r) = @_;
    # print "START $element_name\n";
    if ($element_name =~m{^head\d$}) {
        # Test last section's content on every new section
        $_test_content_sub->($parser, $_section, $_content);
        $_state = 'section';
        $_content = '';
    }
}

sub _handle_element_end {
    my($parser, $element_name) = @_;
    # print "END $element_name\n";
    if ($element_name =~m{^Document$}) {
        $_test_content_sub->($parser, $_section, $_content);
    }
}

sub _handle_text {
    my($parser, $text) = @_;
    # print "TEXT $text\n";
    if ($_state eq 'section') {
        $_section = $text;
        $_state = 'section_content_start';
        return;
    }
    if ($_state eq 'section_content_start') {
        $_content .= $text;
    }
}


1;

__END__

=pod

=head1 NAME

Test::Pod::Content - Test a Pod's content

=head1 SYNOPSIS

 use Test::Pod::Content tests => 3;
 pod_section_is 'Test::Pod::Content' , 'NAME', "Test::Pod::Content - Test a Pod's content", 'NAME section';
 pod_section_like 'Test/Pod/Content.pm', 'SYNOPSIS', qr{ use \s Test::Pod::Content; }xm, 'SYNOPSIS section';
 pod_section_like 'Test/Pod/Content.pm', 'DESCRIPTION', qr{ Test::Pod::Content \s provides \s the }xm, 'DESCRIPTION section';

=head1 DESCRIPTION

This is a very simple module for testing a Pod's content. It is mainly 
intended for testing the content of generated Pod - that s, the Pod included 
in perl modules generated by some mechanism.

Another usage example is to test whether all files contain the same copyright
notice:

 plan tests => scalar @filelist;
 
 for my $file (sort @filelist) {
    pod_section_like( $file, 'LICENSE AND COPYRIGHT', qr{ 
        This \s library \s is \s free \s software\. \s
        You \s may \s distribute/modify \s it \s under \s
        the \s same \s terms \s as \s perl \s itself
    }xms, "$file License notice");
 }

See the files in the t/ directory for live examples.

Test::Pod::Content has a very simple concept of Pods: To Test::Pod::Content, a
Pod is separated into section. Each section starts with a =head(1|2|3|4)
directive, and ends with the next =head, or with the end of the document
(=cut).

This is a very drastic simplification of Pod's document object model, and only
allows for coarse-grained tests.

Test::Pod::Content provides the following subroutines for testing a Pod's content:

=head2 Subroutines

=head3 pod_section_is

  pod_section_is $file, $section, $content, $comment;

Tests whether a Pod section contains exactly the text given. Most useful 
for testing the NAME section. You probably want to use pod_section_like 
for all other sections.

$file may either be a filename (including path) or a module name. 
Test::Pod::Content will search in @INC for the file/module given.

=head3 pod_section_like

 pod_section_like $file, $section, qr{ use \s Test::Pod::Content\s }xm, $comment;

Tests whether the text in a Pod section matches the given regex. Be sure to 
include the m / s regex qualifier if you expect your Pod section to span
multiple lines.

$file may either be a filename (including path) or a module name. 
Test::Pod::Content will search in @INC for the file/module given.

=head1 BUGS AND LIMITATIONS

=over

=item * Performance

Every call to a pod_section_* method searches for the file in question in 
@INC and parses it from its start. This means that every test requires 
a Pod parser run, which is quite inefficient if you conduct a big number of
tests.

=head1 LICENSE AND COPYRIGHT

Copyright 2007 Martin Kutter.

This library is free software. You may distribute/modify it under
the same terms as perl itself

=head1 AUTHOR

Martin Kutter E<lt>martin.kutter fen-net.deE<gt>

=cut