package WebService::NetSuite::SuiteTalk::Role::Country;

use Modern::Perl '2011';    ## no critic (Modules::ProhibitUseQuotedVersion)

our $VERSION = '0.006';     # VERSION
use utf8;
use Const::Fast;
use Locale::Codes::Country 3.20;
use Moo::Role;

const my %COUNTRY_FOR_CODE => (
    bo => '_bolivia',
    cd => '_congoDemocraticPeoplesRepublic',
    hr => '_croatiaHrvatska',
    tl => '_eastTimor',
    fk => '_falklandIslands',
    hm => '_heardAndMcDonaldIslands',
    va => '_holySeeCityVaticanState',
    kp => '_koreaDemocraticPeoplesRepublic',
    ly => '_libyanArabJamahiriya',
    mo => '_macau',
    mk => '_macedonia',
    fm => '_micronesiaFederalStateOf',
    ps => '_palestinianTerritories',
    pn => '_pitcairnIsland',
    re => '_reunionIsland',
    sh => '_saintHelena',
    mf => '_saintMartin',
    sx => '_sintMaarten',
    sk => '_slovakRepublic',
    gs => '_southGeorgia',
    pm => '_stPierreAndMiquelon',
    sj => '_svalbardAndJanMayenIslands',
    tw => '_taiwan',
    tz => '_tanzania',
    um => '_uSMinorOutlyingIslands',
    gb => '_unitedKingdomGB',
    ve => '_venezuela',
    vn => '_vietnam',
    vi => '_virginIslandsUSA',
    wf => '_wallisAndFutunaIslands',
);

## no critic (ProhibitCallsToUndeclaredSubs,ProhibitCallsToUnexportedSubs)
for my $code ( all_country_codes() ) {
    my $name = q{_} . code2country($code);
    $name =~ s/ [ ] (\w) /\u$1/xmsg;
    $name =~ s/ \W //xmsg;
    $name =~ s/\A _ ([[:upper:]]) /_\l$1/xms;
    Locale::Codes::Country::rename_country( $code, $name );
}
while ( my ( $code, $name ) = each %COUNTRY_FOR_CODE ) {
    Locale::Codes::Country::rename_country( $code, $name );
}
Locale::Codes::Country::add_country_code_alias( gb => 'uk' );

1;

# ABSTRACT: Role renaming Locale::Code::Country names for NetSuite web services

__END__

=pod

=encoding UTF-8

=for :stopwords Mark Gardner ZipRecruiter cpan testmatrix url annocpan anno bugtracker rt
cpants kwalitee diff irc mailto metadata placeholders metacpan

=head1 NAME

WebService::NetSuite::SuiteTalk::Role::Country - Role renaming Locale::Code::Country names for NetSuite web services

=head1 VERSION

version 0.006

=head1 SYNOPSIS

    use Moo;
    with 'WebServices::NetSuite::SuiteTalk::Role::Country';

    my $country_name = code2country('us'); # _unitedStates

=head1 DESCRIPTION

This role renames the countries in Locale::Code::Country's database at runtime
to better match the country names enumerated in NetSuite's SuiteTalk platform
common types schema.

=head1 SEE ALSO

=over

=item L<https://webservices.netsuite.com/xsd/platform/v2013_1_0/commonTypes.xsd>

Version 2013.1 of the NetSuite platfrom common types schema document, upon which
this module is based.

=back

=head1 SUPPORT

=head2 Perldoc

You can find documentation for this module with the perldoc command.

  perldoc WebService::NetSuite::SuiteTalk

=head2 Websites

The following websites have more information about this module, and may be of help to you. As always,
in addition to those websites please use your favorite search engine to discover more resources.

=over 4

=item *

MetaCPAN

A modern, open-source CPAN search engine, useful to view POD in HTML format.

L<http://metacpan.org/release/WebService-NetSuite-SuiteTalk>

=item *

Search CPAN

The default CPAN search engine, useful to view POD in HTML format.

L<http://search.cpan.org/dist/WebService-NetSuite-SuiteTalk>

=item *

AnnoCPAN

The AnnoCPAN is a website that allows community annotations of Perl module documentation.

L<http://annocpan.org/dist/WebService-NetSuite-SuiteTalk>

=item *

CPAN Ratings

The CPAN Ratings is a website that allows community ratings and reviews of Perl modules.

L<http://cpanratings.perl.org/d/WebService-NetSuite-SuiteTalk>

=item *

CPAN Forum

The CPAN Forum is a web forum for discussing Perl modules.

L<http://cpanforum.com/dist/WebService-NetSuite-SuiteTalk>

=item *

CPANTS

The CPANTS is a website that analyzes the Kwalitee ( code metrics ) of a distribution.

L<http://cpants.cpanauthors.org/dist/WebService-NetSuite-SuiteTalk>

=item *

CPAN Testers

The CPAN Testers is a network of smokers who run automated tests on uploaded CPAN distributions.

L<http://www.cpantesters.org/distro/W/WebService-NetSuite-SuiteTalk>

=item *

CPAN Testers Matrix

The CPAN Testers Matrix is a website that provides a visual overview of the test results for a distribution on various Perls/platforms.

L<http://matrix.cpantesters.org/?dist=WebService-NetSuite-SuiteTalk>

=item *

CPAN Testers Dependencies

The CPAN Testers Dependencies is a website that shows a chart of the test results of all dependencies for a distribution.

L<http://deps.cpantesters.org/?module=WebService::NetSuite::SuiteTalk>

=back

=head2 Bugs / Feature Requests

Please report any bugs or feature requests through the web
interface at
L<https://github.com/mjgardner/webservice-netsuite-suitetalk/issues>.
You will be automatically notified of any progress on the
request by the system.

=head2 Source Code

The code is open to the world, and available for you to hack on. Please feel free to browse it and play
with it, or whatever. If you want to contribute patches, please send me a diff or prod me to pull
from your repository :)

L<https://github.com/mjgardner/webservice-netsuite-suitetalk>

  git clone git://github.com/mjgardner/webservice-netsuite-suitetalk.git

=head1 AUTHOR

Mark Gardner <mjgardner@cpan.org>

=head1 COPYRIGHT AND LICENSE

This software is copyright (c) 2015 by ZipRecruiter.

This is free software; you can redistribute it and/or modify it under
the same terms as the Perl 5 programming language system itself.

=cut
