#!/usr/local/bin/perl
#
#	@(#)dbschema.pl	1.13	5/7/95
#
#
# dbschema.pl	A script to extract a database structure from
#		a Sybase database
#
# Written by:	Michael Peppler (mpeppler@itf.ch)
# Last Mods:    23 Jan 1995
#
# Usage:	dbschema.pl -d database -o script.name -t pattern -s server -v
#		    where   database is self-explanatory (default: master)
#                           script.name is the output file (default: script.isql)
#                           pattern is the pattern of object names (in sysobjects)
#                           that we will look at (default: %), and server is
#			    the server to connect to (default, the value of $ENV{DSQUERY}).
#
#		    -v turns on a verbose switch.
#
#    Changes:   11/18/93 - bpapp - Put in interactive SA password prompt
#               11/18/93 - bpapp - Get protection information for views and
#                                  stored procedures.
#		02/22/94 - mpeppler - Merge bpapp's changes with itf version
#		09/15/94 - mpeppler - Minor changes for use with Sybperl2
#				      alpha1
#		01/23/95 - mpeppler - Re-write for use with Sybase::DBlib module.
#
#------------------------------------------------------------------------------

use Sybase::DBlib;

require 'sybutil.pl';
require 'getopts.pl';
require 'ctime.pl';

@nul = ('not null','null');

select(STDOUT); $| = 1;		# make unbuffered

&Getopts('d:t:o:s:v');

$opt_d = 'master' unless $opt_d;
$opt_o = 'script.isql' unless $opt_o;
$opt_t = '%' unless $opt_t;
$opt_s = $ENV{DSQUERY} unless $opt_s;

open(SCRIPT, "> $opt_o") || die "Can't open $opt_o: $!\n";
open(LOG, "> $opt_o.log") || die "Can't open $opt_o.log: $!\n";

#
# Log us in to Sybase as 'sa' and prompt for admin password.
#
print "\nAdministrative account password: ";
system("stty -echo");
chop($sapw = <>);
system("stty echo");

($dbproc = new Sybase::DBlib "sa", $sapw, $opt_s) or
    die "Can't connect to the $opt_s Sybase server.\n";
$dbproc->dbuse($opt_d);

chop($date = &ctime(time));

print "dbschema.pl on Database $opt_d\n";

print LOG "Error log from dbschema.pl on Database $opt_d on $date\n\n";
print LOG "The following objects cannot be reliably created from the script in $opt_o.
Please correct the script to remove any inconsistencies.\n\n";

print SCRIPT
    "/* This Isql script was generated by dbschema.pl on $date.
** The indexes need to be checked: column names & index names
** might be truncated!
*/\n";

print SCRIPT "\nuse $opt_d\ngo\n"; # Change to the appropriate database


# first, Add the appropriate user data types:
#

print "Add user-defined data types...";
print SCRIPT
    "/* Add user-defined data types: */\n\n";

$dbproc->dbcmd("select s.length, s.name, st.name,\n");
$dbproc->dbcmd("       object_name(s.tdefault),\n");
$dbproc->dbcmd("       object_name(s.domain)\n");
$dbproc->dbcmd("from   $opt_d.dbo.systypes s, $opt_d.dbo.systypes st\n");
$dbproc->dbcmd("where  st.type = s.type\n");
$dbproc->dbcmd("and s.usertype > 100 and st.usertype < 100 and st.usertype != 18\n");

$dbproc->dbsucceed(TRUE);	# passing TRUE will cause dbsucceed to abort
			 	# in case of failure.

while(@dat = $dbproc->dbnextrow())
{
    print SCRIPT "sp_addtype $dat[1],";
    if ($dat[2] =~ /char|binary/)
    {
        print SCRIPT "'$dat[2]($dat[0])'";
    }
    else
    {
        print SCRIPT "$dat[2]";
    }
    print SCRIPT "\ngo\n";
				# Now remeber the default & rule for later.
    $urule{$dat[1]} = $dat[4] if defined($dat[4]);
    $udflt{$dat[1]} = $dat[3] if defined($dat[3]);
}

print "Done\n";

print "Create rules...";
print SCRIPT
    "\n/* Now we add the rules... */\n\n";

&getObj('Rule', 'R');
print "Done\n";

print "Create defaults...";
print SCRIPT
    "\n/* Now we add the defaults... */\n\n";

&getObj('Default', 'D');
print "Done\n";

print "Bind rules & defaults to user data types...";
print SCRIPT "/* Bind rules & defaults to user data types... */\n\n";

while(($dat, $dflt)=each(%udflt))
{
    print SCRIPT "sp_bindefault $dflt, $dat\ngo\n";
}
while(($dat, $rule) = each(%urule))
{
    print SCRIPT "sp_bindrule $rule, $dat\ngo\n";
}
print "Done\n";

print "Create Tables & Indices...";
print "\n" if $opt_v;

$dbproc->dbcmd("select o.name,u.name, o.id\n");
$dbproc->dbcmd("from $opt_d.dbo.sysobjects o, $opt_d.dbo.sysusers u\n");
$dbproc->dbcmd("where o.type = 'U' and o.name like '$opt_t' and u.uid = o.uid\n");
$dbproc->dbcmd("order by o.name\n");

$dbproc->dbsucceed(1);

while(@dat = $dbproc->dbnextrow())
{
    $_ = join('@', @dat);	# join the data together on a line
    push(@tables,$_);		# and save it in a list
}


foreach (@tables)		# For each line in the list
{
    @tab = split(/@/, $_);

    print "Creating table $tab[0], owner $tab[1]\n" if $opt_v;

    print SCRIPT "/* Start of description of table $tab[1].$tab[0] */\n\n";

    $dbproc->dbcmd("select Column_name = c.name, \n");
    $dbproc->dbcmd("       Type = t.name, \n");
    $dbproc->dbcmd("       Length = c.length, \n");
    $dbproc->dbcmd("       Nulls = convert(bit, (c.status & 8)),\n");
    $dbproc->dbcmd("       Default_name = object_name(c.cdefault),\n");
    $dbproc->dbcmd("       Rule_name = object_name(c.domain)\n");
    $dbproc->dbcmd("from   $opt_d.dbo.syscolumns c, $opt_d.dbo.systypes t\n");
    $dbproc->dbcmd("where  c.id = $tab[2]\n");
    $dbproc->dbcmd("and    c.usertype *= t.usertype\n");

    $dbproc->dbsucceed(TRUE);

    undef(%rule);
    undef(%dflt);

    print SCRIPT "\n\nCREATE TABLE $opt_d.$tab[1].$tab[0]\n ("; 
    $first = 1;
    while(@field = $dbproc->dbnextrow())
    {
        print SCRIPT ",\n" if !$first;		# add a , and a \n if not first field in table
        
        print SCRIPT "\t$field[0] \t$field[1]";
        print SCRIPT "($field[2])" if $field[1] =~ /char|bin/;
	print SCRIPT "($field[3], $field[4])" if $field[1] =~ /num|dec/;
#	print SCRIPT "($field[3])" if $field[1] =~ /float/;
        print SCRIPT " $nul[$field[3]]";
	
	$rule{"$tab[0].$field[0]"} = $field[5] if (defined($field[5]) && $urule{$field[1]} ne $field[5]);
	$dflt{"$tab[0].$field[0]"} = $field[4] if (defined($field[4]) && $udflt{$field[1]} ne $field[4]);;
        $first = 0 if $first;
        
    }
    print SCRIPT " )\n";

# now get the indexes...
#

    print "Indexes for table $tab[1].$tab[0]\n" if $opt_v;
    
    $dbproc->dbcmd("sp_helpindex '$tab[1].$tab[0]'\n");

    $dbproc->dbsucceed(TRUE);

    while(@field = $dbproc->dbnextrow())
    {
        print SCRIPT "\nCREATE ";
        print SCRIPT "unique " if $field[1] =~ /unique/;
        print SCRIPT "clustered " if $field[1] =~ /^clust/;
        print SCRIPT "index $field[0]\n";
        @col = split(/,/,$field[2]);
        print SCRIPT "on $opt_d.$tab[1].$tab[0] (";
        $first = 1;
        foreach (@col)
        {
            print SCRIPT ", " if !$first;
            $first = 0;
            print SCRIPT "$_";
        }
        print SCRIPT ")\n";
    }

    &getPerms("$tab[1].$tab[0]");

    print SCRIPT "go\n";

    print "Bind rules & defaults to columns...\n" if $opt_v;
    print SCRIPT "/* Bind rules & defaults to columns... */\n\n";

    if($tab[1] ne 'dbo' && (keys(%dflt) || keys(%rule)))
    {
	print SCRIPT "/* The owner of the table is $tab[1].
** I can't bind the rules/defaults to a table of which I am not the owner.
** The procedures below will have to be run manualy by user $tab[1].
*/";
	print LOG "Defaults/Rules for $tab[1].$tab[0] could not be bound\n";
    }

    while(($dat, $dflt)=each(%dflt))
    {
	print SCRIPT "/* " if $tab[1] ne 'dbo';
	print SCRIPT "sp_bindefault $dflt, '$dat'";
	if($tab[1] ne 'dbo')
	{
	    print SCRIPT " */\n";
	}
	else
	{
	    print SCRIPT "\ngo\n";
	}
    }
    while(($dat, $rule) = each(%rule))
    {
	print SCRIPT "/* " if $tab[1] ne 'dbo';
	print SCRIPT "sp_bindrule $rule, '$dat'";
	if($tab[1] ne 'dbo')
	{
	    print SCRIPT " */\n";
	}
	else
	{
	    print SCRIPT "\ngo\n";
	}
    }
    print SCRIPT "\n/* End of description of table $tab[1].$tab[0] */\n";

}

print "Done\n";


#
# Now create any views that might exist
#

print "Create views...";
print SCRIPT
    "\n/* Now we add the views... */\n\n";

&getObj('View', 'V');

print "Done\n";

#
# Now create any stored procs that might exist
#

print "Create stored procs...";
print SCRIPT
    "\n/* Now we add the stored procedures... */\n\n";
&getObj('Stored Proc', 'P');

print "Done\n";

#
# Now create the triggers
#

print "Create triggers...";
print SCRIPT
    "\n/* Now we add the triggers... */\n\n";

&getObj('Trigger', 'TR');


print "Done\n";

print "\nLooks like I'm all done!\n";
close(SCRIPT);
close(LOG);


sub getPerms
{
    my($obj) = $_[0];
    my($ret, @dat, $act, $cnt);

    $dbproc->dbcmd("sp_helprotect '$obj'\n");
    $dbproc->dbsqlexec();

    $cnt = 0;
    while(($ret = $dbproc->dbresults()) != NO_MORE_RESULTS)
    {
	if($ret == FAIL)
	{
	    warn("dbresults() failed in getPerms()");
	    next;
	}
	while(@dat = $dbproc->dbnextrow())
	{
	    $act = 'to';
	    $act = 'from' if $dat[0] =~ /Revoke/;
	    print SCRIPT "$dat[0] $dat[1] on $obj $act $dat[2]\n";
	    ++$cnt;
	}
    }
    $cnt;
}

sub getObj
{
    my($objname, $obj) = @_;
    my(@dat, @items, @vi, $found);
    
    $dbproc->dbcmd("select o.name, u.name, o.id\n");
    $dbproc->dbcmd("from $opt_d.dbo.sysobjects o, $opt_d.dbo.sysusers u\n");
    $dbproc->dbcmd("where o.type = '$obj' and o.name like '$opt_t' and u.uid = o.uid\n");
    $dbproc->dbcmd("order by o.name\n");

    $dbproc->dbsucceed(TRUE);

    while(@dat = $dbproc->dbnextrow())
    {
	$_ = join('@', @dat);	# join the data together on a line
	push(@items, $_);	# and save it in a list
    }

    foreach (@items)
    {
	@vi = split(/@/, $_);
	$found = 0;

	$dbproc->dbcmd("select text from syscomments where id = $vi[2]");
	$dbproc->dbsucceed(TRUE);
	
	print SCRIPT
	    "/* $objname $vi[0], owner $vi[1] */\n";

	while(($text) = $dbproc->dbnextrow())
	{
	    if(!$found && $vi[1] ne 'dbo')
	    {
		++$found if($text =~ /$vi[1]/);
	    }
	    print SCRIPT $text;
	}
	print SCRIPT "\ngo\n";
	if(!$found && $vi[1] ne 'dbo')
	{
	    print "**Warning**\n$objname $vi[0] has owner $vi[1]\nbut this is not mentioned in the CREATE PROC statement!!\n";
	    print LOG "$objname $vi[0] (owner $vi[1])\n";
	}
	if ($obj eq 'V' || $obj eq 'P')
	{
	   &getPerms("$vi[0]") && print SCRIPT "go\n";
	}

    }
}

