# NAME

symbol - Symbol manipulation methods for packages.

# DESCRIPTION

This package introduces several subs that can be called as methods on packages.
These subs allow you to modify symbol tables. This module does not do anything
that can't be done with [Package::Stash](https://metacpan.org/pod/Package::Stash), or other tools. What this module does
give you is a package method interface.

# SYNOPSYS

    use symbol;

    # Move a symbol, the old name will be removed
    Foo::Bar->symbol::move('&foo' => '&bar');

    # Alias a symbol, both names will work
    Foo::Bar->symbol::alias('&foo' => '&bar');

    # Get a reference to the symbol
    my $ref = Foo::Bar->symbol::fetch('%foo');

    # Delete a symbol (and return the reference that was removed)
    my $ref = Foo::Bar->symbol::delete('&foo');

    # Check if a symbol exists.
    if(Foo::Bar->symbol::exists('&foo')) {
        ...
    }

# METHODS

These methods all exist in the `symbol::` namespace. These can always be
called as methods on any package thanks to the way perl resolves methods.

- $PACKAGE->symbol::move($SYMBOL, $NEW\_NAME)
- $PACKAGE->symbol::alias($SYMBOL, $NEW\_NAME)

    These will grab the symbol specified by `$SYMBOL` and make it available under
    the name in `$NEW_NAME`. `alias()` will leave the symbol available under both
    names, `move()` will remove it from the original name.

    `$SYMBOL` must be a string identifying the symbol. The symbol string must
    include the sigil unless it is a subroutine. You can provide a fully qualified
    symbol name, or it will be assumed the symbol is in `$PACKAGE`.

    `$NEW_NAME` must be a string identifying the symbol. The string may include a
    symbol, or the sigil from the `$SYMBOL` string will be used. The string can be
    a fully qualified symbol name, or it will be assumed that the new name is in
    `$PACKAGE`.

- $ref = $PACKAGE->symbol::fetch($SYMBOL)
- $ref = $PACKAGE->symbol::delete($SYMBOL)

    These will both find the specified symbol and return a reference to it.
    `fetch()` will simply return the reference, `delete()` will remove the symbol
    before returning the reference.

    `$SYMBOL` must be a string identifying the symbol. The symbol string must
    include the sigil unless it is a subroutine. You can provide a fully qualified
    symbol name, or it will be assumed the symbol is in `$PACKAGE`.

- $bool = $PACKAGE->symbol::exists($SYMBOL)

    This will check if the specified symbol exists. If the symbol exists a true
    value is returned. If the symbol does not exist a false value is returned.

    `$SYMBOL` must be a string identifying the symbol. The symbol string must
    include the sigil unless it is a subroutine. You can provide a fully qualified
    symbol name, or it will be assumed the symbol is in `$PACKAGE`.

# SEE ALSO

- Symbol::Alias

    [Symbol::Alias](https://metacpan.org/pod/Symbol::Alias) Allows you to set up aliases within a package at compile-time.

- Symbol::Delete

    [Symbol::Delete](https://metacpan.org/pod/Symbol::Delete) Allows you to remove symbols from a package at compile time.

- Symbol::Extract

    [Symbol::Extract](https://metacpan.org/pod/Symbol::Extract) Allows you to extract symbols from packages and into
    variables at compile time.

- Symbol::Move

    [Symbol::Move](https://metacpan.org/pod/Symbol::Move) allows you to rename or relocate symbols at compile time.

# SOURCE

The source code repository for symbol can be found at
`http://github.com/exodist/symbol`.

# MAINTAINERS

- Chad Granum &lt;exodist@cpan.org>

# AUTHORS

- Chad Granum &lt;exodist@cpan.org>

# COPYRIGHT

Copyright 2015 Chad Granum &lt;exodist7@gmail.com>.

This program is free software; you can redistribute it and/or
modify it under the same terms as Perl itself.

See `http://dev.perl.org/licenses/`
