package Astro::Constants;
# ABSTRACT: this library provides physical constants for use in Astronomy

'They are not constant but are changing still. - Cymbeline, Act II, Scene 5';

__END__

=pod

=encoding UTF-8

=head1 NAME

Astro::Constants - this library provides physical constants for use in Astronomy

=head1 VERSION

version 0.09_90

=head1 SYNOPSIS

	use strict;		# important!
    use Astro::Constants::MKS qw/:long/;

	# to calculate the gravitational force of the Sun on the Earth in Newtons, use GMm/r^2
	my $force_sun_earth = GRAVITATIONAL * MASS_SOLAR * MASS_EARTH / ASTRONOMICAL_UNIT**2;

=head1 DESCRIPTION

This module provides physical and mathematical constants for use
in Astronomy and Astrophysics.  The two metric systems of units,
MKS and CGS, are kept in two separate modules and are called by
name explicitly.
It allows you to choose between constants in units of
centimetres /grams /seconds
with B<Astro::Constants::CGS> and metres /kilograms /seconds with
B<Astro::Constants::MKS>.

Short forms of the constant names are included to provide backwards
compatibility with older versions based on Jeremy Bailin's Astroconst
library and are available through the import tag C<:short>.

The values are stored in F<Physical_Constants.xml> in the B<data> directory
and are mostly based on the 2014 CODATA values from NIST.

The problem with long constants is that they are not interpolated
in double quotish situations because they are really inlined functions.
The problem with short name constants is that they are not read-only
and can be assigned to which will mess up your program.

=head1 METHODS

=head2 LIGHT_SPEED

    2.99792458e8	MKS
    2.99792458e10	CGS

speed of light in a vacuum

This constant can also be accessed through the short name $A_c (deprecated)

=head2 BOLTZMANN

    1.38064852e-23	MKS
    1.38064852e-16	CGS

Boltzmann's constant

This constant can also be accessed through the short name $A_k (deprecated)

=head2 GRAVITATIONAL

    6.67408e-11	MKS
    6.67408e-8	CGS

universal gravitational constant

This constant can also be accessed through the short name $A_G (deprecated)

=head2 ELECTRON_VOLT

    1.6021766208e-19	MKS
    1.6021766208e-12	CGS

electron volt

This constant can also be accessed through the short name $A_eV (deprecated)

=head2 PLANCK

    6.626070040e-34	MKS
    6.626070040e-27	CGS

Planck constant

This constant can also be accessed through the short name $A_h (deprecated)

=head2 HBAR

    1.054571800e-34	MKS
    1.054571800e-27	CGS

Planck's constant /2pi

This constant can also be accessed through the short name $A_hbar (deprecated)

=head2 ELECTRON_CHARGE

    1.6021766208e-19	MKS
    4.8032046729e-10	CGS

electron charge (defined positive)

This constant can also be accessed through the short name $A_e (deprecated)

=head2 STEFAN_BOLTZMANN

    5.670367e-8	MKS
    5.670367e-5	CGS

Stefan-Boltzmann constant

This constant can also be accessed through the short name $A_sigma (deprecated)

=head2 A_RAD

    7.565723e-16	MKS
    7.565723e-15	CGS

radiation density constant, 4 * sigma / c

This constant can also be accessed through the short name $A_arad (deprecated)

=head2 WIEN

    2.8977729e-3	MKS
    2.8977729e-1	CGS

Wien wavelength displacement law constant

This constant can also be accessed through the short name $A_Wien (deprecated)

=head2 ALPHA

    7.2973525664e-3	MKS
    7.2973525664e-3	CGS

fine structure constant

This constant can also be accessed through the short name $A_alpha (deprecated)

=head2 VACUUM_IMPEDANCE

    376.730313461

characteristic impedance of vacuum

This constant can also be accessed through the short name $A_Z0 (deprecated)

=head2 PERMITIV_FREE_SPACE

    8.854187817e-12	MKS
    1	CGS

permittivity of free space, epsilon_0, the electric constant

This constant can also be accessed through the short name $A_eps0 (deprecated)

=head2 PERMEABL_FREE_SPACE

    1.2566370614e-6	MKS
    1	CGS

permeability of free space, mu_0, the magnetic constant

This constant can also be accessed through the short name $A_mu0 (deprecated)

=head2 PI

    3.14159265358979324

trig constant pi

This constant can also be accessed through the short name $A_pi (deprecated)

=head2 EXP

    2.71828182846

base of natural logarithm

This constant can also be accessed through the short name $A_exp (deprecated)

=head2 ATOMIC_MASS_UNIT

    1.660539040e-27	MKS
    1.660539040e-24	CGS

atomic mass unit

This constant can also be accessed through the short name $A_amu (deprecated)

=head2 PARSEC

    3.08567758149e16	MKS
    3.08567758149e18	CGS

parsec

This constant can also be accessed through the short name $A_pc (deprecated)

=head2 ASTRONOMICAL_UNIT

    149_597_870_700	MKS
    1.496e13	CGS

astronomical unit

This constant can also be accessed through the short name $A_AU (deprecated)

=head2 LIGHT_YEAR

    9_460_730_472_580_800	MKS
    9.4607304725808e17	CGS

the distance that light travels in vacuum in one Julian year

This constant can also be accessed through the short name $A_ly (deprecated)

=head2 ANGSTROM

    1e-10	MKS
    1e-8	CGS

Angstrom

This constant can also be accessed through the short name $A_AA (deprecated)

=head2 JANSKY

    1e-26	MKS
    1e-23	CGS

Jansky

This constant can also be accessed through the short name $A_Jy (deprecated)

=head2 AVOGADRO

    6.022140857e23

Avogadro's number

This constant can also be accessed through the short name $A_NA (deprecated)

=head2 YEAR

    31_557_600

defined as exactly 365.25 days of 86400 SI seconds

This constant can also be accessed through the short name $A_yr (deprecated)

=head2 YEAR_TROPICAL

    31_556_925.1

the period of time for the ecliptic longitude of the Sun to increase 360 degrees, approximated by the Gregorian calendar

=head2 YEAR_SIDEREAL

    31_558_149.8

the period of revolution of the Earth around the Sun in a fixed reference frame

=head2 YEAR_ANOMALISTIC

    31_558_432.6

the period between successive passages of the Earth through perihelion

=head2 YEAR_ECLIPSE

    29_947_974.3

the period between successive passages of the Sun (as seen from the geocenter) through the same lunar node

=head2 SOLAR_MASS

    1.9885e30	MKS
    1.9885e33	CGS

solar mass

This constant can also be accessed through the short name $A_msun (deprecated)

=head2 SOLAR_LUMINOSITY

    3.846e26	MKS
    3.846e33	CGS

solar luminosity

This constant can also be accessed through the short name $A_Lsun (deprecated)

=head2 RHO_C

    1.8791e-26	MKS
    1.8791e-29	CGS

critical density /h^2

This constant can also be accessed through the short name $A_rhoc (deprecated)

=head2 HUBBLE_TIME

    3.0853056e17

Hubble time *h, the inverse of Hubble's constant valued at 100 km/s/Mpc (DEPRECATED - see ChangeLog)

This constant can also be accessed through the short name $A_tH (deprecated)

=head2 CMB_TEMPERATURE

    2.725

cosmic microwave background temperature

This constant can also be accessed through the short name $A_TCMB (deprecated)

=head2 SOLAR_V_MAG

    -26.74

solar V magnitude

This constant can also be accessed through the short name $A_Vsun (deprecated)

=head2 SOLAR_V_ABS_MAG

    4.83

solar absolute V magnitude

This constant can also be accessed through the short name $A_MVsun (deprecated)

=head2 SOLAR_RADIUS

    6.96e8	MKS
    6.96e10	CGS

solar radius

This constant can also be accessed through the short name $A_rsun (deprecated)

=head2 EARTH_MASS

    5.9726e24	MKS
    5.9726e27	CGS

mass of Earth

This constant can also be accessed through the short name $A_mearth (deprecated)

=head2 EARTH_RADIUS

    6.371e6	MKS
    6.371e8	CGS

radius of Earth

This constant can also be accessed through the short name $A_rearth (deprecated)

=head2 SOLAR_TEMPERATURE

    5778

surface temperature of sun

This constant can also be accessed through the short name $A_Tsun (deprecated)

=head2 SOLAR_DENSITY

    1408	MKS
    1.408	CGS

mean solar density

This constant can also be accessed through the short name $A_dsun (deprecated)

=head2 EARTH_DENSITY

    5514	MKS
    5.514	CGS

mean Earth density

This constant can also be accessed through the short name $A_dearth (deprecated)

=head2 SOLAR_GRAVITY

    274.0	MKS
    27400	CGS

solar surface gravity

This constant can also be accessed through the short name $A_gsun (deprecated)

=head2 EARTH_GRAVITY

    9.78	MKS
    978	CGS

Earth surface gravity

This constant can also be accessed through the short name $A_gearth (deprecated)

=head2 LUNAR_RADIUS

    1.7381e6	MKS
    1.7381e8	CGS

lunar radius

This constant can also be accessed through the short name $A_rmoon (deprecated)

=head2 LUNAR_MASS

    7.342e22	MKS
    7.342e25	CGS

lunar mass

This constant can also be accessed through the short name $A_mmoon (deprecated)

=head2 LUNAR_SM_AXIS

    3.844e8	MKS
    3.844e10	CGS

lunar orbital semi-major axis

This constant can also be accessed through the short name $A_amoon (deprecated)

=head2 LUNAR_ECCENTRICITY

    0.0549

lunar orbital eccentricity

This constant can also be accessed through the short name $A_emoon (deprecated)

=head2 THOMSON_XSECTION

    6.6524587158e-29	MKS
    6.6524587158e-25	CGS

Thomson cross-section

This constant can also be accessed through the short name $A_sigmaT (deprecated)

=head2 ELECTRON_MASS

    9.10938356e-31	MKS
    9.10938356e-28	CGS

mass of electron

This constant can also be accessed through the short name $A_me (deprecated)

=head2 PROTON_MASS

    1.672621898e-27	MKS
    1.672621898e-24	CGS

mass of proton

This constant can also be accessed through the short name $A_mp (deprecated)

=head2 NEUTRON_MASS

    1.674927471e-27	MKS
    1.674927471e-24	CGS

neutron mass

This constant can also be accessed through the short name $A_mn (deprecated)

=head2 HYDROGEN_MASS

    1.67372e-24

mass of Hydrogen atom

This constant can also be accessed through the short name $A_mH (deprecated)

=head2 ELECTRON_RADIUS

    2.8179403227e-15	MKS
    2.8179403227e-13	CGS

classical electron radius

This constant can also be accessed through the short name $A_re (deprecated)

=head2 BOHR_RADIUS

    5.2917721067e-11	MKS
    5.2917721067e-9	CGS

Bohr radius

This constant can also be accessed through the short name $A_a0 (deprecated)

=head1 EXPORT

Nothing is exported by default.  Select from the following tags:

=over 4

=item *

long		(use this one to get the most constants)

=item *

short

=item *

fundamental

=item *

conversion

=item *

mathematics

=item *

cosmology

=item *

planetary

=item *

electromagnetic

=item *

nuclear

=back

=head1 SEE ALSO

=over 4

=item *

L<Astro::Cosmology>

=item *

L<PDL|Perl Data Language>

=item *

L<http://physics.nist.gov/|NIST>

=item *

L<http://neilb.org/reviews/constants.html|Neil Bower's review on providing read-only values>

=back

=head1 ISSUES

File issues at the Github repository L<https://github.com/duffee/Astro-Constants/>

Using C<strict> is a must with this code.  Any constants you forgot to import will
evaluate to 0 and silently introduce errors in your code.

=head2 Extending the data set

If you want to add in your own constants or override the factory defaults,
run make, edit the F<PhysicalConstants.xml> file and then run C<dzil build> again.
If you have a pre-existing F<PhysicalConstants.xml> file, drop it in place
before running C<dzil build>.

=head2 Availability

the original astroconst sites have disappeared

=head1 ROADMAP

I plan to deprecate the short names and change the order in which
long names are constructed, moving to a I<noun_adjective> format.
LIGHT_SPEED and SOLAR_MASS become SPEED_LIGHT and MASS_SOLAR.
This principle should make the code easier to read with the most
important information coming at the beginning of the name.

=head1 ASTROCONST  X<ASTROCONST>

(Gleaned from the Astroconst home page -
L<http://web.astroconst.org|http://web.astroconst.org> )

Astroconst is a set of header files in various languages (currently C,
Fortran, Perl, Java, IDL and Gnuplot) that provide a variety of useful
astrophysical constants without constantly needing to look them up.

The generation of the header files from one data file is automated, so you
can add new constants to the data file and generate new header files in all
the appropriate languages without needing to fiddle with each header file
individually.

This package was created and is maintained by Jeremy Bailin.  It's license
states that it I<is completely free, both as in speech and as in beer>.

=head1 DISCLAIMER

No warranty expressed or implied.  This is free software.  If you
want someone to assume the risk of an incorrect value, you better
be paying them.

(What would you want me to test for you to depend on this module?)

I<from Jeremy Bailin's astroconst header files>

The Astroconst values have been gleaned from a variety of sources,
and have quite different precisions depending both on the known
precision of the value in question, and in some cases on the
precision of the source I found it from. These values are not
guaranteed to be correct. Astroconst is not certified for any use
whatsoever. If your rocket crashes because the precision of the
lunar orbital eccentricity isn't high enough, that's too bad.

=head1 ACKNOWLEDGMENTS

Jeremy Balin, for writing the astroconst package and helping
test and develop this module.

Doug Burke, for giving me the idea to write this module in the
first place, tidying up Makefile.PL, testing and improving the
documentation.

=head1 AUTHOR

Boyd Duffee <duffee at cpan.org>

=head1 COPYRIGHT AND LICENSE

This software is copyright (c) 2015 by Boyd Duffee.

This is free software; you can redistribute it and/or modify it under
the same terms as the Perl 5 programming language system itself.

=cut
