#!/usr/bin/env perl
# PODNAME: list2mapper.pl
# ABSTRACT: Build final id mapper from id list using the NCBI Taxonomy database

use Modern::Perl '2011';
use autodie;

use Getopt::Euclid qw(:vars);
use Smart::Comments;

use Bio::MUST::Core;
use Bio::MUST::Core::Utils qw(change_suffix);
use aliased 'Bio::MUST::Core::IdList';
use aliased 'Bio::MUST::Core::Taxonomy';


# build taxonomy object
my $tax = Taxonomy->new_from_cache( tax_dir => $ARGV_taxdir );

my $method = $ARGV_from_must ? 'load_lis' : 'load';

for my $infile (@ARGV_infiles) {

    ### Processing: $infile
    my $list = IdList->$method($infile);
	my $mapper = $tax->tax_mapper(
		$list, { append_acc => $ARGV_append_acc }
	);
	my $outfile = change_suffix($infile, '.final-idm');
	$mapper->store($outfile);
}

__END__

=pod

=head1 NAME

list2mapper.pl - Build final id mapper from id list using the NCBI Taxonomy database

=head1 VERSION

version 0.210230

=head1 USAGE

    list2mapper.pl --taxdir=<dir> <infiles> [optional arguments]

=head1 REQUIRED ARGUMENTS

=over

=item <infiles>

Path to input IDL files [repeatable argument].

=for Euclid: infiles.type: readable
    repeatable

=item --taxdir=<dir>

Path to local mirror of the NCBI Taxonomy database.

=for Euclid: dir.type: string

=back

=head1 OPTIONAL ARGUMENTS

=over

=item --from-must

Consider the input file as generated by ed/treeplot [default: no]. Currently,
this switches to the legacy .lis format (instead of the modern .idl format).

=item --append-acc

Append accession (or gi) number to organism name [default: no].

=item --version

=item --usage

=item --help

=item --man

Print the usual program information

=back

=head1 AUTHOR

Denis BAURAIN <denis.baurain@uliege.be>

=head1 COPYRIGHT AND LICENSE

This software is copyright (c) 2013 by University of Liege / Unit of Eukaryotic Phylogenomics / Denis BAURAIN.

This is free software; you can redistribute it and/or modify it under
the same terms as the Perl 5 programming language system itself.

=cut
