package OpenInteract::Handler::NewUser;

# $Id: NewUser.pm,v 1.4 2001/09/14 20:52:31 lachoy Exp $

use strict;
use Email::Valid;
use Data::Dumper  qw( Dumper );
use SPOPS::Secure qw( :level :scope );

@OpenInteract::Handler::NewUser::ISA     = qw( OpenInteract::Handler::GenericDispatcher );
$OpenInteract::Handler::NewUser::VERSION = sprintf("%d.%02d", q$Revision: 1.4 $ =~ /(\d+)\.(\d+)/);

$OpenInteract::Handler::NewUser::author            = 'chris@cwinters.com';
$OpenInteract::Handler::NewUser::default_method    = 'show';
@OpenInteract::Handler::NewUser::forbidden_methods = ();

my $MAIN_SCRIPT  = '/NewUser';
my $REMOVAL_TIME = 60 * 60 * 24; # 1 day

sub show {
    my ( $class, $p ) = @_;
    my $R = OpenInteract::Request->instance;
    $p->{main_script} = $MAIN_SCRIPT;
    $R->{page}{title} = 'Create a new User Account!';
    return $R->template->handler( {}, $p,
                                  { db      => 'new_user_form',
                                    package => 'base_user' } );
}

sub edit {
    my ( $class, $p ) = @_;
    my $R = OpenInteract::Request->instance;
    my $login = $R->apache->param( 'requested_login' );
    my $email = $R->apache->param( 'working_email' );

    # First ensure the email is good

    unless ( $email and Email::Valid->address( $email ) ) {
        $R->throw({ code       => 421, 
                    type       => 'email',
                    system_msg => "Invalid email address: $email",
                    extra      => { bad_address => $email } } );
        my $error_msg = 'The email address you entered is invalid! Please try again.';
        return $class->show({ error_msg       => $error_msg,
                              requested_login => $login });
    }

    # Next, ensure they typed in a login

    unless( $login ) {
        my $error_msg = 'You cannot create an account without entering a login!';
        return $class->show( { error_msg => $error_msg, working_email => $email } );
    }

    # Next, see if the username exists

    my $user = eval { $R->user->fetch_by_login_name( $login, 
                                                     { skip_security => 1,
                                                       return_single => 1 } ) };
    if ( $user or $@ ) {
        $R->throw({ code       => 422,
                    type       => 'user',
                    system_msg => "Tried to create a new user with an existing name: $login",
                    extra      => { bad_login => $login } } );
        my $error_msg = 'The username you requested is already in use. Please try again.';
        return $class->show({ error_msg     => $error_msg,
                              working_email => $email } );
    }

    # Now, create an entry in the user table; note that we set the removal
    # date to now plus whatever $REMOVAL_TIME is set to

    my $new_user = $R->user->new;
    $new_user->{login_name} = $login;
    $new_user->{email}      = $email;
    $new_user->{theme_id}   = $R->CONFIG->{default_objects}{theme}; 
    my $password            = $R->user->generate_random_code( 12, 'mixed' );
    if ( $R->CONFIG->{login}{crypt_password} ) {
        $new_user->{password}   = $R->user->crypt_it( $password );
    }
    $new_user->{removal_date} = $R->user->now( { time => $R->{time} + $REMOVAL_TIME } );
    eval { $new_user->save };

    # If there was an error, log it and send to the admin (error handler)
    if ( $@ ) {
        OpenInteract::Error->set( SPOPS::Error->get );
        $OpenInteract::Error::extra->{login_name} = $login;
        $OpenInteract::Error::extra->{password}   = $password;
        $OpenInteract::Error::extra->{email}      = $email;
        $R->throw({ code => 321 });
        my $error_msg = 'Could not create user in database. Error has been logged and administrator contacted.';
        return $class->show({ error_msg => $error_msg });
    }

    # Ensure that the user can read/write his/her own record!

    eval { $R->user->set_item_security({ class     => $R->user, 
                                         object_id => $new_user->{user_id},
                                         scope     => SEC_SCOPE_USER, 
                                         scope_id  => $new_user->{user_id},
                                         level     => SEC_LEVEL_WRITE }) };

    # Log the failed security set, if it happens...

    if ( $@ ) {
        OpenInteract::Error->set( SPOPS::Error->get );
        $R->throw( { code => 406 } );
    }

    # ...otherwise, mark the user as the creator of his/her own record

    else {
        $new_user->log_action_enter( 'create', $new_user->{user_id}, $new_user->{user_id} );
    }

    # If that worked ok, send the user an email with the password created

    my $msg = <<NOTIFY;
Someone requested a new user account on: $R->{server_name} and gave
this email address as their contact. If this was you, you can login
using the username and password below. Once you are properly logged
in, you can modify the (admittedly ugly) system-generated password by
clicking 'Edit my info' in the box labeled 'User Info'.

Account information, for your records:
Username: $login
Password: $password

I hope you enjoy $R->{server_name}!

The Management

PS: Note that if you do not login within 24 hours your account will 
be deleted. So hop to it!

NOTIFY
  eval { OpenInteract::Utility->send_email({ message => $msg,
                                             to      => $email,
                                             subject => "Account information for $R->{server_name}" }) };
    if ( $@ ) {
        $R->scrib( 0, "Cannot send email! ", Dumper( OpenInteract::Error->get ) );
        $R->throw({ code => 203 });
        return $class->show({ error_msg => "Could not send email to you with assigned password." });
    }
    $R->{page}{title} = 'Account Created! Now login.';
    $R->{page}{return_url} = '/NewUser/';
    return <<OK;
<h2 align="center">E-Mail Sent</h2>
<p>Your account information has been sent to <tt>$email</tt>. Follow
the instructions to login and activate your account.
OK
}

__END__

=pod

=head1 NAME

OpenInteract::Handler::NewUser - Display form for and process new user requests

=head1 SYNOPSIS

 # Display form for users to create a new account
 my $html = OpenInteract::Handler::NewUser->show;

 # Process the form
 my $html = OpenInteract::Handler::NewUser->edit;

=head1 DESCRIPTION

This handler takes care of creating a new user record on request,
creating a temporary password for the new user and notifying the user
on how to login. It does some preliminary checks on the email address
to ensure it is at least valid. We also set a date on the temporary
account creation so a simple cron job can cleanup abandoned attempts.

=head1 METHODS

=over 4

=item B<show>

Displays the form for creating a new account, plus any error messsages
that might occur when processing the request (in I<edit()>).

=item B<edit>

Creates the user account and notifies the user with the temporary
password as well as the fact that the account will be removed in 24
hours if he/she does not login.

=back

=head1 NOTES

=head1 TO DO

=head1 BUGS

=head1 COPYRIGHT

Copyright (c) 2001 intes.net, inc.. All rights reserved.

This library is free software; you can redistribute it and/or modify
it under the same terms as Perl itself.

=head1 AUTHORS

Chris Winters <chris@cwinters.com>

=cut
