package PDK::Device::Paloalto;

use 5.030;
use strict;
use warnings;

use Moose;
use Expect qw'exp_continue';
use Carp   qw'croak';
with 'PDK::Device::Base';
use namespace::autoclean;

# 定义提示符的正则表达式
has prompt => (
  is       => 'ro',
  required => 1,
  default  => '^.*?\((?:active|passive)\)[>#]\s*$',    # 默认提示符格式
);

#------------------------------------------------------------------------------
# errCodes: 返回可能的错误模式数组
# 描述：这些错误模式可用于检测在命令执行时是否发生错误。
#------------------------------------------------------------------------------
sub errCodes {
  my $self = shift;                                    # 获取对象实例

  return [
    qr/(Unknown command|Invalid syntax)/i,             # 错误提示模式
    qr/^Error:/mi,                                     # 其他常见错误
  ];
}

#------------------------------------------------------------------------------
# waitfor: 期望匹配特定的提示符，自动交互式执行脚本
# 描述：此方法将等待设备输出，直到达到指定的提示符。
#------------------------------------------------------------------------------
sub waitfor {
  my ($self, $prompt) = @_;

  my $buff = "";                # 存储命令输出
  $prompt //= $self->prompt;    # 使用提供的提示符或默认提示符

  my $exp = $self->exp;         # 获取 Expect 对象实例

  # 处理期望输出
  my @ret = $exp->expect(
    10,
    [
      qr/^lines\s*\d+-\d+\s*$/mi => sub {
        $exp->send(" ");            # 发送空格继续输出
        $buff .= $exp->before();    # 追加输出
        exp_continue;               # 继续监听
      }
    ],
    [
      qr/are you sure\?/mi => sub {
        $exp->send("y\r");          # 发送确认选择
        $buff .= $exp->before() . $exp->match();
        exp_continue;
      }
    ],
    [
      qr/$prompt/mi => sub {
        $buff .= $exp->before() . $exp->match();    # 处理预期提示符，保存输出
      }
    ],
  );

  # 检查期望结果，处理错误
  if (defined $ret[1]) {
    croak($ret[3]);    # 抛出异常
  }

  # 修正回显指令，清除多余的回车符
  $buff =~ s/ \cH//g;                     # 捕捉数字之间空白符
  $buff =~ s/(\c[\S+)+\cM(\c[\[K)?//g;    # 捕捉字符信息
  $buff =~ s/\cM(\c[\S+)+\c[>//g;         # 登陆成功提示符

  return $buff;                           # 返回捕获的输出
}

#------------------------------------------------------------------------------
# getConfig: 获取设备的当前配置
# 返回：包含成功标志和配置内容的哈希引用
#------------------------------------------------------------------------------
sub getConfig {
  my $self = shift;                       # 获取对象实例

  # 定义要执行的命令
  my $commands = [
    " set cli pager off",                  # 设置屏幕宽度限制，确保完整输出
    "set cli config-output-format set",    # 禁用屏幕长度限制，确保完整输出
    "show",                                # 显示当前配置
    "exit",                                # 强制保存配置
  ];

  # 执行命令并获取结果
  my $config = $self->execCommands($commands);

  # 返回错误信息（如果有）
  if ($config->{success} == 0) {
    return $config;                        # 若执行失败，直接返回
  }
  else {
    my $lines = $config->{result};         # 获取命令执行结果

    return {success => 1, config => $lines};    # 返回成功及配置内容
  }
}

#------------------------------------------------------------------------------
# ftpConfig: 将设备配置通过 FTP 备份
# 参数：$server - FTP 服务器地址, $hostname - 可选的主机名,
#       $username - FTP 用户名, $password - FTP 密码
# 返回：执行命令的结果
#------------------------------------------------------------------------------
# sub ftpConfig {
#   my ($self, $server, $hostname, $username, $password) = @_;
#
#   # 生成 FTP 脚本
#   my $date = `date +%Y-%m-%d`;  # 获取当前日期
#   chomp($date);  # 去除换行符
#   my $host = $self->host;  # 获取主机名
#   my $command = "put startup.cfg $date/";  # 构建初始命令
#
#   # 根据是否提供 hostname 生成文件名
#   if ($hostname) {
#     $command .= $hostname . '_' . $host . '.cfg';  # 使用 hostname 和 host 作为文件名
#   } else {
#     $command .= $host . '.cfg';  # 仅使用 host 作为文件名
#   }
#
#   # 检查是否已经登录设备
#   if (!$self->exp) {
#     my $login = $self->login();
#     if ($login->{success} != 1) {
#       croak $login->{reason};  # 登录失败抛出异常
#     }
#   }
#
#   # 下发脚本并执行
#   my $exp = $self->exp;
#   my $result = $exp->match() // '';  # 初始化结果
#
#   $exp->send("ftp $server\n");  # 连接到 FTP 服务器
#   my @ret = $exp->expect(15,
#     [ qr/User \(/i => sub {
#       $result .= $exp->before() . $exp->match();  # 捕获输出
#       $exp->send("$username\n");  # 发送用户名
#       exp_continue;
#     }],
#     [ qr/assword:/i => sub {
#       $result .= $exp->before() . $exp->match();  # 捕获输出
#       $exp->send("$password\n");  # 输入密码
#     }],
#   );
#
#   # 检查期望结果，处理错误
#   if (defined $ret[1]) {
#     croak $ret[3];  # 登录失败抛出异常
#   }
#
#   # 检查是否正常登录
#   @ret = $exp->expect(10,
#     [ qr/ftp: Login failed./i => sub {
#       croak("FTP 连接失败: username or password is wrong!");
#     }],
#     [ qr/User logged in/ => sub {
#       $result .= $exp->before() . $exp->match();
#       say "成功连接 FTP 服务器($server)" if $self->debug;  # 连接成功，调试信息
#     }],
#   );
#
#   $exp->send("$command\n");  # 发送 FTP 上传命令
#   @ret = $exp->expect(15,
#     [ qr/(No such file or directory|The system cannot)/i => sub {
#       croak "执行脚本 $command 异常，上传失败!";
#     }],
#     [ qr/Transfer complete.*ftp>/ms => sub {
#       $result .= $exp->before() . $exp->match() . $exp->after();
#       say "脚本 $command 已执行完毕, 文件上传成功!" if $self->debug;  # 上传成功
#     }],
#   );
#
#   # 检查期望结果，处理错误
#   if (defined $ret[1]) {
#     croak $ret[3];  # 抛出异常
#   }
#
#   return { success => 1, config => $result };  # 返回成功及结果
# }

# 标记类为不可变，以提高性能
__PACKAGE__->meta->make_immutable;
1;    # 返回真值以表示模块加载成功
