package PDK::Device::Huawei::Usg;

use 5.030;
use strict;
use warnings;

use Moose;
use Expect qw'exp_continue';
use Carp   qw'croak';
extends 'PDK::Device::Huawei';
use namespace::autoclean;

#------------------------------------------------------------------------------
# getConfig: 获取设备的当前配置
# 返回：包含成功标志和配置内容的哈希引用
#------------------------------------------------------------------------------
sub getConfig {
  my $self = shift;    # 获取对象实例

  # 定义要执行的命令
  my $commands = [

    # "screen-length disable temporary", # 禁用屏幕长度限制，确保完整输出
    "screen-width 512",               # 禁用屏幕长度限制，确保完整输出
    "screen-length 512 temporary",    # 禁用屏幕长度限制，确保完整输出
    "dis current-configuration",      # 显示当前配置
    "save"                            # 强制保存配置
  ];

  # 执行命令并获取结果
  my $config = $self->execCommands($commands);

  # 返回错误信息（如果有）
  if ($config->{success} == 0) {
    return $config;                   # 若执行失败，直接返回
  }
  else {
    my $lines = $config->{result};    # 获取命令执行结果

    # 特殊字符串处理
    # $lines =~ s/.*?(\[\#*\s*\]\s*\d{1,3}%)[^[]*$/$1/;

    return {success => 1, config => $lines};    # 返回成功及配置内容
  }
}

#------------------------------------------------------------------------------
# ftpConfig: 将设备配置通过 FTP 备份
# 描述：此方法连接到 FTP 服务器并上传配置文件。
# 参数：
#   可选参数：$hostname - 可选的主机名, $server - FTP 服务器地址, $username - FTP 用户名, $password - FTP 密码
# 优化：增加从环境变量加载 FTP用户名和密码
# 返回：包含成功状态和上传结果的哈希引用
#------------------------------------------------------------------------------
sub ftpConfig {
  my ($self, $hostname, $server, $username, $password) = @_;

  # 从环境变量加载用FTP服务器、户名和密码（如果未提供）
  $server   //= $ENV{PDK_FTP_SERVER};
  $username //= $ENV{PDK_FTP_USERNAME};
  $password //= $ENV{PDK_FTP_PASSWORD};

  # 检查用户名和密码是否有效
  croak "请正确提供 FTP 服务器地址、账户和密码!" unless $username and $password and $server;

  # 生成 FTP 脚本
  my $host    = $self->{host};                                     # 获取主机名
  my $command = "put config.cfg $self->{month}/$self->{date}/";    # 构建初始命令
      # my $command = "put vrpcfg.zip $self->{month}/$self->{date}/";  # 构建初始命令

  # 根据是否提供 hostname 生成文件名
  if ($hostname) {
    $command .= $hostname . '_' . $host . '.cfg';    # 使用 hostname 和 host 作为文件名
  }
  else {
    $command .= $host . '.cfg';                      # 仅使用 host 作为文件名
  }

  # 检查是否已经登录设备
  if (!$self->{exp}) {
    my $login = $self->login();
    croak $login->{reason} if $login->{success} == 0;    # 登录失败抛出异常
  }

  # 获取初始匹配结果
  my $exp    = $self->{exp};
  my $result = $exp ? $exp->match() : "";

  # 下发脚本并执行
  my $ftp_cmd = "ftp $server vpn-instance default";
  say "[debug] 生成 FTP 备份指令：$ftp_cmd" if $self->{debug};

  $exp->send("$ftp_cmd\n");    # 连接到 FTP 服务器
  my @ret = $exp->expect(
    15,
    [
      qr/User\s*\(/i => sub {
        $result .= $exp->before() . $exp->match();    # 捕获输出
        $exp->send("$username\n");                    # 发送用户名
        exp_continue;
      }
    ],
    [
      qr/assword:/i => sub {
        $result .= $exp->before() . $exp->match();    # 捕获输出
        $exp->send("$password\n");                    # 输入密码
      }
    ],
    [
      eof => sub {
        croak("执行[ftpConfig/登录FTP服务器]，与设备 $self->{host} 会话丢失，连接被意外关闭！具体原因：\n" . $exp->before());
      }
    ],
    [
      timeout => sub {
        croak("执行[ftpConfig/登录FTP服务器]，与设备 $self->{host} 会话超时，请检查网络连接或服务器状态！具体原因：\n" . $exp->before());
      }
    ],
  );

  # 检查期望结果，处理错误
  croak($ret[3]) if defined $ret[1];    # 抛出异常

  # 检查是否正常登录
  @ret = $exp->expect(
    10,
    [
      qr/(ftp: Login failed.|Username)/i => sub {
        croak("FTP 会话丢失: username or password is wrong!");
      }
    ],
    [
      qr/User logged in/i => sub {
        $result .= $exp->before() . $exp->match();
        say "[debug] 成功连接 FTP 服务器($server)" if $self->{debug};    # 连接成功，调试信息
      }
    ],
    [
      eof => sub {
        croak("执行[ftpConfig/检查是否成功登录FTP]，与设备 $self->{host} 会话丢失，连接被意外关闭！具体原因：\n" . $exp->before());
      }
    ],
    [
      timeout => sub {
        croak("执行[ftpConfig/检查是否成功登录FTP]，与设备 $self->{host} 会话超时，请检查网络连接或服务器状态！具体原因：\n" . $exp->before());
      }
    ],
  );

  # 发送 FTP 上传命令
  $exp->send("$command\n");
  @ret = $exp->expect(
    15,
    [
      qr/(No such file or directory|The system cannot)/i => sub {
        croak "执行脚本 $command 异常，上传失败!";
      }
    ],
    [
      qr/Transfer complete.*ftp[>\]]/ms => sub {
        $result .= $exp->before() . $exp->match() . $exp->after();
        say "\n[debug] 脚本 $command 已执行完毕, 文件上传成功!" if $self->{debug};    # 上传成功
      }
    ],
    [
      eof => sub {
        croak("执行[ftpConfig/检查备份任务是否成功]，与设备 $self->{host} 会话丢失，连接被意外关闭！具体原因：\n" . $exp->before());
      }
    ],
    [
      timeout => sub {
        croak("执行[ftpConfig/检查备份任务是否成功]，与设备 $self->{host} 会话超时，请检查网络连接或服务器状态！具体原因：\n" . $exp->before());
      }
    ],
  );

  # 检查期望结果，处理错误
  croak($ret[3]) if defined $ret[1];    # 抛出异常
  $exp->send("quit\n");

  return {success => 1, config => $result};    # 返回成功及结果
}

# 标记类为不可变，以提高性能
__PACKAGE__->meta->make_immutable;
1;                                             # 返回真值以表示模块加载成功
