package PDK::Device::Cisco::Nxos;

use 5.030;
use strict;
use warnings;

use Moose;
use Expect qw'exp_continue';
use Carp   qw'croak';
extends 'PDK::Device::Cisco';
use namespace::autoclean;

#------------------------------------------------------------------------------
# ftpConfig: 将设备配置通过 FTP 备份
# 参数：$hostname - 可选的主机名, $server FTP - 服务器地址, $username - FTP 用户名, $password - FTP 密码
# 硬编码 vrf: 默认为 default, 实际 IDC 环境有带外管理，需要用到 management 平面
# 优化：增加从环境变量加载 FTP用户名和密码
# 返回：执行命令的结果
#------------------------------------------------------------------------------
sub ftpConfig {
  my ($self, $hostname, $server, $username, $password) = @_;

  # 从环境变量加载用FTP服务器、户名和密码（如果未提供）
  $server   //= $ENV{PDK_FTP_SERVER};
  $username //= $ENV{PDK_FTP_USERNAME};
  $password //= $ENV{PDK_FTP_PASSWORD};

  # 检查用户名和密码是否有效
  croak "请正确提供 FTP 服务器地址、账户和密码!" unless $username and $password and $server;

  # 获取主机名
  my $host = $self->{host};

  # 构建初始命令
  my $command = "copy running-config ftp://$username" . '@' . "$server/$self->{month}/$self->{date}/";

  # 根据是否提供 hostname 生成文件名
  if ($hostname) {
    $command .= $hostname . '_' . $host . '.cfg';    # 使用 hostname 和 host 作为文件名
  }
  else {
    $command .= $host . '.cfg';                      # 仅使用 host 作为文件名
  }

  # 检查是否已经登录设备
  if (!$self->{exp}) {
    my $login = $self->login();
    if ($login->{success} != 1) {
      croak $login->{reason};    # 登录失败抛出异常
    }
  }

  # 下发脚本并执行
  my $exp    = $self->{exp};
  my $result = $exp->match() // '';    # 初始化结果

  # FTP 备份使用的网络屏幕
  # my $vrf = 'management';  # FTP 建联虚拟路由器
  my $vrf = 'default';    # FTP 建联虚拟路由器

  $exp->send("$command\n");    # 连接到 FTP 服务器
  my @ret = $exp->expect(
    15,
    [
      qr/Enter vrf/mi => sub {
        $result .= $exp->before() . $exp->match();    # 捕获输出
        $exp->send("$vrf\n");                         # 发送用户名
        exp_continue;
      }
    ],
    [
      qr/assword:/mi => sub {
        $result .= $exp->before() . $exp->match();    # 捕获输出
        $exp->send("$password\n");                    # 输入密码
      }
    ],
    [
      eof => sub {
        croak("执行[$command/尝试FTP备份配置]，与设备 $self->{host} 会话丢失，连接被意外关闭！具体原因：\n" . $exp->before());
      }
    ],
    [
      timeout => sub {
        croak("执行[$command/尝试FTP备份配置]，与设备 $self->{host} 会话超时，请检查网络连接或服务器状态！具体原因：\n" . $exp->before());
      }
    ],
  );

  # 检查期望结果，处理错误
  if (defined $ret[1]) {
    croak $ret[3];    # 登录失败抛出异常
  }

  # 检查是否正常登录
  @ret = $exp->expect(
    10,
    [
      qr/Transfer of file aborted \*/mi => sub {
        croak "执行脚本 $command 异常，上传失败!";
      }
    ],
    [
      qr/Copy complete\./mi => sub {
        $result .= $exp->before() . $exp->match();
        say "[debug] 脚本 $command 已执行完毕, 文件上传成功!" if $self->{debug};    # 上传成功
        exp_continue;
      }
    ],
    [
      qr/$self->{prompt}/mi => sub {
        $result .= $exp->before() . $exp->match();
      }
    ],
    [
      eof => sub {
        croak("执行[$command/检查备份任务是否完成]，与设备 $self->{host} 会话丢失，连接被意外关闭！具体原因：\n" . $exp->before());
      }
    ],
    [
      timeout => sub {
        croak("执行[$command/检查备份任务是否完成]，与设备 $self->{host} 会话超时，请检查网络连接或服务器状态！具体原因：\n" . $exp->before());
      }
    ],
  );

  # 检查期望结果，处理错误
  if (defined $ret[1]) {
    croak $ret[3];    # 抛出异常
  }

  return {success => 1, config => $result};    # 返回成功及结果
}

# 标记类为不可变，以提高性能
__PACKAGE__->meta->make_immutable;
1;                                             # 返回真值以表示模块加载成功
