package Netstack::Utils::Set;

#------------------------------------------------------------------------------
# 加载扩展模块功能
#------------------------------------------------------------------------------
use 5.016;
use Moose;
use namespace::autoclean;
use POSIX;
use experimental 'smartmatch';

#------------------------------------------------------------------------------
# 定义 Netstack::Utils::Set 方法属性
#------------------------------------------------------------------------------
has mins => (
  is      => 'rw',
  isa     => 'ArrayRef[Int]',
  default => sub {[]},
  traits  => ['Array'],
  handles  => {
    add_mins => 'push',
  }
);

has maxs => (
  is      => 'rw',
  isa     => 'ArrayRef[Int]',
  default => sub {[]},
  traits  => ['Array'],
  handles  => {
    add_maxs => 'push'
  }
);

#------------------------------------------------------------------------------
# length 集合对象长度
#------------------------------------------------------------------------------
sub length {
  my $self = shift;
  # 标量上下文
  my $lengthOfMin = $self->mins->@*;
  my $lengthOfMax = $self->maxs->@*;
  # 异常拦截
  confess "ERROR: Attribute (mins) 's length($lengthOfMin) not equal (maxs) 's length($lengthOfMax)"
    if $lengthOfMin != $lengthOfMax;
  # 返回计算结果
  return $lengthOfMin;
}

#------------------------------------------------------------------------------
# min 集合对象最小值
#------------------------------------------------------------------------------
sub min {
  my $self = shift;
  return ($self->length > 0 ? $self->mins->[0] : undef);
}

#------------------------------------------------------------------------------
# max 集合对象最大值
#------------------------------------------------------------------------------
sub max {
  my $self = shift;
  return ($self->length > 0 ? $self->maxs->[-1] : undef);
}

#------------------------------------------------------------------------------
# dump 打印结合对象数据结构
#------------------------------------------------------------------------------
sub dump {
  my $self = shift;
  my $length = $self->length;
  for (my $i = 0; $i < $length; $i++) {
    say $self->mins->[$i] . "  " . $self->maxs->[$i];
  }
}

#------------------------------------------------------------------------------
# 不需要检查重复，需要检查排序，所以用这个的时候要特别慎重
# 只有在确定输入与set不重复的情况下才可使用，否则会有问题
# addToSet 添加元素到集合对象
#------------------------------------------------------------------------------
sub addToSet {
  my ($self, $MIN, $MAX) = @_;
  # 判断并修正 数字 大小排序
  ($MIN, $MAX) = $MIN > $MAX ? ($MAX, $MIN) : ($MIN, $MAX);
  my $length = $self->length;
  # 空集合对象
  if ($length == 0) {
    $self->add_mins($MIN);
    $self->add_maxs($MAX);
    return;
  }
  # 非空集合对象
  my $minArray = $self->mins;
  my $maxArray = $self->maxs;
  # 遍历集合对象，本身数据就有顺序
  my $index;
  for (my $i = 0; $i < $length; $i++) {
    if ($MIN < $minArray->[$i]) {
      $index = $i;
      last;
    }
  }

  # $MIN 大于 $minArray 的所有元素，则需要在数组新增该$MIN(特例)
  $index = $length if not defined $index;
  # 初始化对象
  my (@min, @max);
  push @min, @{$minArray}[ 0 .. $index - 1 ];
  push @max, @{$maxArray}[ 0 .. $index - 1 ];
  push @min, $MIN;
  push @max, $MAX;
  push @min, @{$minArray}[ $index .. $length - 1 ];
  push @max, @{$maxArray}[ $index .. $length - 1 ];

  $self->add_mins(\@min);
  $self->add_maxs(\@max);
}

#------------------------------------------------------------------------------
# mergeToSet 添加元素到集合对象
#------------------------------------------------------------------------------
sub mergeToSet {
  my $self = shift;
  if (@_ == 1 and ref($_[0]) eq __PACKAGE__) {
    my $setObj = $_[0];
    my $length = $setObj->length;
    for (my $i = 0; $i < $length; $i++) {
      $self->_mergeToSet($setObj->mins->[$i], $setObj->maxs->[$i]);
    }
  }
  else {
    $self->_mergeToSet(@_);
  }
}

#------------------------------------------------------------------------------
# _mergeToSet 将数据加入集合对象 | 需要检查重复、同时检查排序
#------------------------------------------------------------------------------
sub _mergeToSet {
  my ($self, $MIN, $MAX) = @_;
  # 判断数据排序是否正确
  ($MIN, $MAX) = $MIN > $MAX ? ($MAX, $MIN) : ($MIN, $MAX);
  my $length = $self->length;

  # 情况1：空集合对象
  if ($length == 0) {
    $self->add_mins($MIN);
    $self->add_maxs($MAX);
    return;
  }

  # 情况2：非空集合对象，需要确定插入点
  my $minArray = $self->mins;
  my $maxArray = $self->maxs;
  # 设置缺省值，POSIX::ceil/floor 函数向上或向下取整，实际数组长度为 (0, $length - 1)
  my ($minIndex, $maxIndex) = (-1, $length);

  # 最小数从左往右找
  MIN: {
    for (my $i = 0; $i < $length; $i++) {
      # 命中 ($minArray[$i], $maxArray[$i]) 区间值
      if ($MIN >= $minArray->[$i] and $MIN <= $maxArray->[$i] + 1) {
        $minIndex = $i;
        last MIN;
      }
      # 命中 ($minArray[$i], $maxArray[$i]) 左侧，比区间最小值还小
      elsif ($MIN < $minArray->[$i]) {
        $minIndex += 0.5;
        last MIN;
      }
      # 命中 ($minArray[$i], $maxArray[$i]) 右侧，比区间最大值还大
      else {
        $minIndex++;
      }
    }
    $minIndex += 0.5;
  }
  # 最大数从右往左找
  MAX: {
    for (my $j = $length - 1; $j >= $minIndex; $j--) {
      # 命中 ($minArray[$i], $maxArray[$i]) 区间值
      if ($MAX >= $minArray->[$j] - 1 and $MAX <= $maxArray->[$j]) {
        $maxIndex = $j;
        last MAX;
      }
      # 命中 ($minArray[$i], $maxArray[$i]) 右侧，比最大值还大
      elsif ($MAX > $maxArray->[$j]) {
        $maxIndex -= 0.5;
        last MAX;
      }
      # 命中 ($minArray[$i], $maxArray[$i]) 左侧，比最小值还小
      else {
        $maxIndex--;
      }
    }
    $maxIndex -= 0.5;
  }

  # min使用向上取整，POSIX::ceil(0.5) = 1
  # max使用向下取整，POSIX::floor(-0.5) = -1
  my $minIndexInt = POSIX::ceil($minIndex);
  my $maxIndexInt = POSIX::floor($maxIndex);
  my $isMinIndexInSet = $minIndex == $minIndexInt ? 1 : 0;
  my $isMaxIndexInSet = $maxIndex == $maxIndexInt ? 1 : 0;
  # 初始化变量
  my (@min, @max);
  push @min, $minArray->@*->[ 0 .. $minIndexInt - 1 ];
  push @max, $maxArray->@*->[ 0 .. $minIndexInt - 1 ];
  # 数据插入点
  push @min, $isMinIndexInSet ? $minArray->[$minIndexInt] : $MIN;
  push @max, $isMaxIndexInSet ? $maxArray->[$maxIndexInt] : $MAX;
  push @min, $minArray->@*->[ $maxIndexInt + 1 .. $length - 1 ];
  push @max, $maxArray->@*->[ $maxIndexInt + 1 .. $length - 1 ];
  # 将展开数据插入到mins maxs
  $self->add_mins(\@min);
  $self->add_maxs(\@max);
}

#------------------------------------------------------------------------------
# compare 集合对象比较逻辑：相等、包含且不相等、部分属于但不完全包含、不相干关系
#------------------------------------------------------------------------------
sub compare {
  my ($self, $setObj) = @_;
  if ($self->isEqual($setObj)) {
    return 'equal';
  }
  elsif ($self->_isContain($setObj)) {
    return 'containButNotEqual';
  }
  elsif ($self->_isBelong($setObj)) {
    return 'belongButNotEqual';
  }
  else {
    return 'other';
  }
}

#------------------------------------------------------------------------------
# isEqual 集合对象相等性判断
#------------------------------------------------------------------------------
sub isEqual {
  my ($self, $setObj) = @_;
  return ($self->mins->@* ~~ $setObj->mins->@* and $self->maxs->@* ~~ $setObj->maxs->@*);
}
#------------------------------------------------------------------------------
# notEqual 集合对象不相等性判断
#------------------------------------------------------------------------------
sub notEqual {
  my ($self, $setObj) = @_;
  return !($self->mins->@* ~~ $setObj->mins->@* and $self->maxs->@* ~~ $setObj->maxs->@*);
}
#------------------------------------------------------------------------------
# isContain 集合对象包含另外一个集合对象
#------------------------------------------------------------------------------
sub isContain {
  my ($self, $setObj) = @_;
  if ($self->isEqual($setObj)) {
    return 1;
  }
  else {
    return $self->_isContain($setObj);
  }
}
#------------------------------------------------------------------------------
# compare 集合对象比较逻辑：相等、包含且不相等、部分属于但不完全包含、不相干关系
#------------------------------------------------------------------------------
sub _isContain {
  my ($self, $setObj) = @_;
  # 新实例化集合对象，并将待比较的集合对象 merge 到 该对象
  my $copyOfSelf = Netstack::Utils::Set->new($self);
  $copyOfSelf->mergeToSet($setObj);
  # 进行相等性判断
  return $self->isEqual($copyOfSelf);
}
#------------------------------------------------------------------------------
# isContainButNotEqual 包含待比较对象，但不完全相等 | 包含住了待比较集合
#------------------------------------------------------------------------------
sub isContainButNotEqual {
  my ($self, $setObj) = @_;
  # 确保集合对象不相等
  if ($self->isEqual($setObj)) {
    return;
  }
  else {
    return $self->_isContain($setObj);
  }
}
#------------------------------------------------------------------------------
# isBelong 集合对象属于另外一个集合对象
#------------------------------------------------------------------------------
sub isBelong {
  my ($self, $setObj) = @_;
  # 集合对象相等，模糊上看也是包含了这个集合
  if ($self->isEqual($setObj)) {
    return 1;
  }
  else {
    return $self->_isBelong($setObj);
  }
}
#------------------------------------------------------------------------------
# _isBelong 实例化新的
#------------------------------------------------------------------------------
sub _isBelong {
  my ($self, $setObj) = @_;
  my $copyOfSetObj = Netstack::Utils::Set->new($setObj);
  $copyOfSetObj->mergeToSet($self);
  return $setObj->isEqual($copyOfSetObj);
}
#------------------------------------------------------------------------------
# isBelongButNotEqual
#------------------------------------------------------------------------------
sub isBelongButNotEqual {
  my ($self, $setObj) = @_;
  if ($self->isEqual($setObj)) {
    return;
  }
  else {
    return $self->_isBelong($setObj);
  }
}
#------------------------------------------------------------------------------
# interSet
#------------------------------------------------------------------------------
sub interSet {
  my ($self, $setObj) = @_;
  # 实例化集合对象
  my $result = Netstack::Utils::Set->new;
  # 边界条件检查
  if ($self->length == 0) {
    return $self;
  }
  if ($setObj->length == 0) {
    return $setObj;
  }

  # 初始化变量
  my $i = 0;
  my $j = 0;
  while ($i < $self->length and $j < $setObj->length) {
    my @rangeSet1 = ($self->mins->[$i], $self->maxs->[$i]);
    my @rangeSet2 = ($setObj->mins->[$j], $setObj->maxs->[$j]);
    my ($min, $max) = $self->interRange(\@rangeSet1, \@rangeSet2);
    # 判断是否定义 min max
    $result->_mergeToSet($min, $max) if defined $min;

    if ($setObj->maxs->[$j] > $self->maxs->[$i]) {
      $i++;
    }
    elsif ($setObj->maxs->[$j] == $self->maxs->[$i]) {
      $i++;
      $j++;
    }
    else {
      $j++;
    }
  }
  # 返回计算结果
  return $result;
}

#------------------------------------------------------------------------------
# interRange 返回真
#------------------------------------------------------------------------------
sub interRange {
  my ($self, $rangeSet1, $rangeSet2) = @_;
  # 初始化变量
  my $min
    = $rangeSet1->[0] < $rangeSet2->[0]
    ? $rangeSet1->[0]
    : $rangeSet2->[0];
  my $max
    = $rangeSet1->[1] > $rangeSet2->[1]
    ? $rangeSet1->[1]
    : $rangeSet2->[1];
  # 返回计算结果
  return $min > $max
    ? undef
    : ($min, $max);
}

#------------------------------------------------------------------------------
# addToSet _mergeToSet 钩子函数：入参检查，必须传递两个数字对象
#------------------------------------------------------------------------------
for my $func (qw/addToSet _mergeToSet/) {
  before $func => sub {
    my $self = shift;
    unless (@_ == 2 and $_[0] =~ /^\d+$/o and $_[1] =~ /^\d+$/o) {
      confess "ERROR: function $func can only has two numeric argument";
    }
  }
}
#------------------------------------------------------------------------------
# 集合关系比对函数需要确保为集合对象
#------------------------------------------------------------------------------
for my $func (qw/compare isEqual isContain _isContain isContainButNotEqual isBelong _isBelong isBelongButNotEqual/) {
  before $func => sub {
    my $self = shift;
    # 约束性检查
    confess "ERROR: the first param of function($func) is not a Netstack::Utils::Set"
      if ref($_[0]) ne 'Netstack::Utils::Set';
  }
}

#------------------------------------------------------------------------------
# compare 集合对象比较逻辑：相等、包含且不相等、部分属于但不完全包含、不相干关系
#------------------------------------------------------------------------------
around BUILDARGS => sub {
  my $orig = shift;
  my $class = shift;
  if (@_ == 0) {
    return $class->$orig();
  }
  elsif (@_ == 1 && ref($_[0]) eq __PACKAGE__) {
    my $setObj = $_[0];
    return $class->$orig(
      mins => [$setObj->mins->@*],
      maxs => [$setObj->maxs->@*]
    );
  }
  elsif (@_ == 2
         && defined $_[0]
         && defined $_[1]
         && $_[0] =~ /^\d+$/o
         && $_[1] =~ /^\d+$/o) {
    my ($MIN, $MAX) = $_[0] < $_[1] ? ($_[0], $_[1]) : ($_[1], $_[0]);
    return $class->$orig(
      mins => [ $MIN ],
      maxs => [ $MAX ]
    );
  }
  else {
    return $class->$orig(@_);
  }
};

sub BUILD {
  my $self = shift;
  my @ERROR;
  my $lengthOfMin = @{$self->mins};
  my $lengthOfMax = @{$self->maxs};
  if ($lengthOfMin != $lengthOfMax) {
    push @ERROR, 'Attribute (mins) and (maxs) must has same length at constructor ' . __PACKAGE__;
  }
  for (my $i = 0; $i < $lengthOfMin; $i++) {
    if ($self->mins->[$i] > $self->maxs->[$i]) {
      push @ERROR, 'Attribute (mins) must not bigger than (maxs) in the same index at constructor ' . __PACKAGE__;
      last;
    }
  }
  if (@ERROR > 0) {
    confess join(', ', @ERROR);
  }
}

__PACKAGE__->meta->make_immutable;
1;
