package Firewall::Policy::Designer::Huawei;

#------------------------------------------------------------------------------
# 引用基础模块
#------------------------------------------------------------------------------
use Moose;
use namespace::autoclean;

#------------------------------------------------------------------------------
# 引用项目模块
#------------------------------------------------------------------------------
use Firewall::Utils::Ip;
use Firewall::Utils::Set;

#------------------------------------------------------------------------------
# 继承 Firewall::Policy::Designer::Role 通用属性
#------------------------------------------------------------------------------
with 'Firewall::Policy::Designer::Role';

#------------------------------------------------------------------------------
# 新建防火墙策略
#------------------------------------------------------------------------------
sub createRule {
  my $self = shift;
  my @commands;

  # 判断是否需要 nat
  my $action = $self->searcherReportFwInfo->action->{new};
  my $natSrc = $action->{natSrc} if defined $action->{natSrc};
  my $natDst = $action->{natDst} if defined $action->{natDst};
  my $natInfo;

  if (defined $natSrc) {
    foreach (values %{$natSrc}) {
      $natInfo->{natSrc} = $_;
      $self->checkAndCreateNat($natInfo);
    }
  }
  if (defined $natDst) {
    foreach (values %{$natDst}) {
      $natInfo->{natDst} = $_;
      $self->checkAndCreateNat($natInfo);
    }
  }

  # 初始化访问控制 5 元组 -> 源目地址、区域、服务端口
  my $fromZone = $self->{searcherReportFwInfo}{fromZone};
  my $toZone   = $self->{searcherReportFwInfo}{toZone};
  my $schedule = $self->{searcherReportFwInfo}->{schedule};
  my @srcs     = keys %{$self->{searcherReportFwInfo}{action}{new}{src}};
  my @dsts     = keys %{$self->{searcherReportFwInfo}{action}{new}{dst}};
  my @srvs     = keys %{$self->{searcherReportFwInfo}{action}{new}{srv}};
  my $sch      = $self->createSchedule($schedule) if $schedule->{enddate} ne 'always';

  # 动态启用地址组
  my $randNum  = $$ || sprintf('%05d', int(rand(99999)));
  my $ruleName = 'policy_' . Firewall::Utils::Date->getFormatedDate('yyyymmdd_hhmiss') . "_$randNum";
  push @commands, "security-policy";
  push @commands, "rule name $ruleName";
  push @commands, "source-zone $fromZone";
  push @commands, "destination-zone $toZone";

  # 源地址成员大于3，优先使用动态地址组
  my ($srcGroup, $dstGroup);
  if (@srcs > 3) {
    $srcGroup = $self->createAddrGroup(\@srcs);
    push @commands, "source-address address-set $srcGroup";
  }
  else {
    my @srcAddrs = $self->createAddress(\@srcs, "src");
    push @commands, @srcAddrs;
  }

  if (@dsts > 3) {
    $dstGroup = $self->createAddrGroup(\@dsts);
    push @commands, "destination-address address-set $dstGroup";
  }
  else {
    my @dstAddrs = $self->createAddress(\@dsts, "dst");
    push @commands, @dstAddrs;
  }

  # 查询或创建服务地址对象
  my @srvObjs;
  for my $srv (@srvs) {
    my $srvName = $self->createService($srv);
    push @srvObjs, $srvName;
  }

  # 推送访问控制服务端口命令行
  for my $srvObj (@srvObjs) {
    push @commands, "service $srvObj";
  }

  # 推送访问控制时间策略
  if (defined $sch) {
    push @commands, "time-range $sch";
  }

  # 推送访问控制动作
  push @commands, "action permit";
  push @commands, "quit";

  # 返回计算结果
  $self->addCommands(@commands);
}

#------------------------------------------------------------------------------
# 修改策略对象
# FIXME 暂不支持华为防火墙策略修改
#------------------------------------------------------------------------------
sub modifyRule {
  my $self    = shift;
  my $nameMap = $self->checkAndCreateAddrOrSrvOrNat($self->searcherReportFwInfo->action->{add});

  # if (my $param = $self->searcherReportFwInfo->action->{new}) {
  #   for my $type (keys %{$param}) {
  #     if ($type eq 'natDst' or $type eq 'natSrc') {
  #       $self->createNat($param->{$type}, $type);
  #     }
  #   }
  # }

  # 初始化防火墙策略数组
  my $policyId = $self->{searcherReportFwInfo}{ruleObj}{policyId};
  my @commands;
  push @commands, "security-policy";
  push @commands, "rule name $policyId";
  for my $type (keys %{$nameMap}) {
    if ($type eq 'src') {
      for my $host (@{$nameMap->{$type}}) {
        push @commands, "source-address $host";
      }
    }
    elsif ($type eq 'dst') {
      for my $host (@{$nameMap->{$type}}) {
        push @commands, "destination-address $host";
      }
    }
    elsif ($type eq 'srv') {
      for my $srv (@{$nameMap->{$type}}) {
        push @commands, "service $srv";
      }
    }
  }
  push @commands, "exit";
  $self->addCommands(@commands);
}

#------------------------------------------------------------------------------
# createAddress 创建地址对象
#------------------------------------------------------------------------------
sub createAddress {
  my ($self, $addrMap, $srcOrDst) = @_;

  # 初始化 cmdStr
  my ($config, @commands);
  my $direction = ($srcOrDst eq "src") ? "source-address" : ($srcOrDst eq "dst") ? "destination-address" : undef;

  # 遍历 addrMap
  foreach my $addr (@{$addrMap}) {
    my ($ip, $mask) = split('/', $addr);
    if (defined $mask and $mask == 0) {
      $config = "$direction any";
      push @commands, $config;
    }
    elsif (defined $mask and $mask) {
      $mask   = Firewall::Utils::Ip->changeMaskToIpForm($mask);
      $config = "$direction $ip mask $mask" if defined $direction;
      push @commands, $config;
    }
    else {
      if ($ip =~ /^(2[0-4]\d|25[0-5]|1?\d\d?\.){3}(2[0-4]\d|25[0-5]|1?\d\d?)-(2[0-4]\d|25[0-5]|1?\d\d?)$/) {
        my ($ipMin, $ipMax) = split("-", $ip);
        $ipMax  = $1 . $2 . $3 . $5;
        $config = "$direction range $ipMin $ipMax" if defined $direction;
        push @commands, $config;
      }
    }
  }

  # 返回计算结果
  return @commands;
}

#------------------------------------------------------------------------------
# createService 新增服务端口命令行配置命令行
#------------------------------------------------------------------------------
sub createService {
  my ($self, $srv) = @_;
  my ($serviceName, $dstPort, @commands);

  # 拆解 protocol port
  my ($protocol, $port) = split('/', $srv);
  $protocol = lc $protocol;

  # 仅支持 TCP/UDP 需要指定端口
  return if $protocol ne 'tcp' and $protocol ne 'udp';

  if ($port =~ /^(?<portMin>\d+)\-(?<portMax>\d+)$/o) {
    $serviceName = $protocol . "_" . $+{portMin} . "-" . $+{portMax};
    $dstPort     = $+{portMin} . " to " . $+{portMax};
  }
  elsif ($port =~ /^\d+$/o) {
    $serviceName = $protocol . "_" . $port;
    $dstPort     = $port;
  }
  else {
    confess "ERROR: $port is not a port";
  }

  # 推送配置
  push @commands, "ip service-set $serviceName type object";
  push @commands, "service protocol source-port 0 to 65535 destination-port $dstPort";
  push @commands, "quit";

  # 返回计算结果
  $self->addCommands(@commands);
  return $serviceName;
}

#------------------------------------------------------------------------------
# createSchedule 新增时间策略命令行
#------------------------------------------------------------------------------
sub createSchedule {
  my ($self, $schedule) = @_;

  # 初始化 commands
  my @commands;

  # 获取起止时间属性
  my $startDate = $schedule->{startdate};
  my $endDate   = $schedule->{enddate};

  # 如果未定义起止时间则跳过
  return if not defined $startDate;

  # 分割时间
  my ($syear, $smon, $sday, $shh, $smm) = split('[ :-]', $startDate);
  my ($year,  $mon,  $day,  $hh,  $mm)  = split('[ :-]', $endDate);
  my $schName = "sch_$year-$mon-$day";

  # 推送配置
  push @commands, "time-range $schName";
  push @commands, "absolute-range $shh:$smm:00 $syear/$smon/$sday end $hh:$mm:00 $year/$mon/$day";
  push @commands, "quit";

  # 返回计算属性
  $self->addCommands(@commands);
  return $schName;
}

#------------------------------------------------------------------------------
# createAddrGroup 新增地址组命令行
#------------------------------------------------------------------------------
sub createAddrGroup {
  my ($self, $addrMap) = @_;

  # 初始化 addrs commands
  my (@addrs, @commands);

  # 初始化防火墙解析对象
  my $parser = $self->{searcherReportFwInfo}{parser};

  # 创建随机地址组
  my $groupId   = sprintf('%d', 100 + int(rand(10000)));
  my $groupName = "addrSet_" . $groupId;

  # 生成唯一的地址组
  while (defined $parser->getAddress($groupName)) {
    $groupName = "addrSet_" . sprintf('%d', 100 + int(rand(10000)));
  }

  push @commands, "ip address-set $groupName type object";

  # 转换 addrMap 为集合对象
  my $addrSet = Firewall::Utils::Set->new;
  for my $addr (@{$addrMap}) {
    my ($ip, $mask) = split('/', $addr);
    $addrSet->mergeToSet(Firewall::Utils::Ip->getRangeFromIpMask($ip, $mask));
  }

  # 遍历区间集合，重组 $addrs (有可能连续的地址可缩写)
  for (my $i = 0; $i < $addrSet->length; $i++) {
    my $min = $addrSet->{mins}[$i];
    my $max = $addrSet->{maxs}[$i];
    push @addrs, Firewall::Utils::Ip->getIpMaskFromRange($min, $max);
  }

  # 遍历 addrs
  for my $addr (@addrs) {
    if ($addr =~ /\-/) {
      my ($ipMin, $ipMax) = split('-', $addr);
      push @commands, "address range $ipMin $ipMax";
    }
    else {
      my ($ip, $mask) = split('/', $addr);
      push @commands, "address $ip mask $mask";
    }
  }
  push @commands, "quit";
  $self->addCommands(@commands);

  # 返回计算结果
  return $groupName;
}

#------------------------------------------------------------------------------
# isNewVer 检测是否为新版本防火墙
#------------------------------------------------------------------------------
sub isNewVer {
  my $self = shift;

  # 获取防火墙版本信息
  my $version = $self->{searcherReportFwInfo}{parser}{version};

  # 判断防火墙版本
  if (defined $version) {
    if ($version =~ /V(?<mainVer>\d+)R\d+/i) {
      my $mainVer = $+{"mainVer"};
      return ($mainVer >= 500) ? 1 : 0;
    }
  }
}

#------------------------------------------------------------------------------
# 查询或创建地址转换
#------------------------------------------------------------------------------
sub checkAndCreateNat {
  my ($self, $param) = @_;
  return unless $self->isNewVer;

  my @commands;
  my $ipRegex = qr/(2[0-4]\d|25[0-5]|1?\d\d?\.){3}(2[0-4]\d|25[0-5]|1?\d\d?)/;
  push @commands, "nat-policy";
  my $randNum  = sprintf('%05d', int(rand(99999)));
  my $timeNow  = Firewall::Utils::Date->getFormatedDate('yyyymmdd_hhmiss') . "_$randNum";
  my $ruleName = "policy_nat_$timeNow";
  my $fromZone = $self->{searcherReportFwInfo}{fromZone};
  my $toZone   = $self->{searcherReportFwInfo}{toZone};
  my $srvMap   = $self->{searcherReportFwInfo}{srvMap};
  push @commands, "rule name $ruleName";
  push @commands, "source-zone $fromZone";
  push @commands, "destination-zone $toZone";

  for my $type (keys %{$param}) {
    my $natIp   = $param->{$type}{natInfo}{natIp};
    my $natType = $param->{$type}{natInfo}{natType};
    my $realIp  = $param->{$type}->{realIp};
    my ($ip, $mask) = split('/', $realIp);
    $mask = 32 if not defined $mask;
    my ($mip, $mmask) = split('/', $natIp);
    $mmask = 32 if not defined $mmask;

    if ($type eq 'natSrc') {
      if (not defined $natIp) {
        my ($ip, $mask) = split('/', $realIp);
        $mask = 32 if not defined $mask;
        push @commands, "source-address $ip $mask";
        push @commands, "action source-nat easy-ip";
        push @commands, "quit";
      }
      else {
        my ($ip, $mask) = split('/', $realIp);
        $mask = 32 if not defined $mask;
        my $poolName = $self->createPool($natIp, $type);
        push @commands, "source-address $ip $mask";
        push @commands, "action source-nat address-group $poolName";
        push @commands, "quit";
      }
    }
    elsif ($type eq 'natDst') {
      if ($natIp =~ /$ipRegex\/\d+/) {
        my ($ip, $mask) = split('/', $natIp);
        push @commands, "destination-address $ip $mask";
      }
      elsif ($natIp =~ /$ipRegex-$ipRegex/) {
        my ($ipMin, $ipMax) = split('-', $natIp);
        push @commands, "destination-address range $ipMin $ipMax";
      }
      my $natPort = "";
      for my $srv (keys %{$srvMap}) {
        if (defined $srvMap->{$srv}{natPort}) {
          my ($proto, $port) = split('/', $srv);
          push @commands, "service protocol $proto destination-port $port";
          $natPort = $srvMap->{$srv}{natPort};
        }
      }
      my $poolName = $self->createPool($natIp, $type);
      push @commands, "action destination-nat address-group $poolName $natPort";
      push @commands, "quit";
    }
  }
  $self->addCommands(@commands);
}

#------------------------------------------------------------------------------
# 查询或创建地址转换
#------------------------------------------------------------------------------
sub getPoolName {
  my ($self, $toZone, $natIp) = @_;
  my $poolName;

  # 初始化 parser
  my $parser = $self->{searcherReportFwInfo}{parser};

  # 转换 natIp 为集合对象
  my ($ip, $mask) = split('/', $natIp);
  $mask = 32 if not defined $mask;
  my $natIpSet = Firewall::Utils::Ip->getRangeFromIpMask($ip, $mask);

  # 遍历已有 natPool
  for my $natPool (values %{$parser->{elements}{natPool}}) {
    if ($natPool->{zone} eq $toZone and $natPool->{poolRange}->isContain($natIpSet)) {
      $poolName = $natPool->{poolName};
      return $poolName;
    }
  }
}

#------------------------------------------------------------------------------
# createPool 新增 natPool
#------------------------------------------------------------------------------
sub createPool {
  my ($self, $natIp, $type) = @_;

  # 初始化 commands
  my @commands;

  #  初始化 poolName
  my $ipRange   = qr/(2[0-4]\d|25[0-5]|1?\d\d?\.){3}(2[0-4]\d|25[0-5]|1?\d\d?)-(2[0-4]\d|25[0-5]|1?\d\d?)/;
  my $ipAndMask = qr/(2[0-4]\d|25[0-5]|1?\d\d?\.){3}(2[0-4]\d|25[0-5]|1?\d\d?)/;
  my $poolName  = "natPool_$natIp";

  if ($type eq 'natSrc') {
    push @commands, "nat address-group $poolName";
    push @commands, "mode pat";
  }
  elsif ($type eq 'natDst') {
    push @commands, "destination-nat address-group $poolName";
  }

  if ($natIp =~ $ipRange) {
    my ($ipMin, $ipMax) = split('-', $natIp);
    $ipMax = $1 . $2 . $3 . $5;
    push @commands, "section $ipMin $ipMax";
  }
  elsif ($natIp =~ $ipAndMask) {
    my ($ip, $mask) = split('/', $natIp);
    $mask = 32 if not defined $mask;
    my ($min, $max) = Firewall::Utils::Ip->getRangeFromIpMask($ip, $mask);
    my $ipMin = Firewall::Utils::Ip->changeIntToIp($min);
    my $ipMax = Firewall::Utils::Ip->changeIntToIp($max);
    push @commands, "section $ipMin $ipMax";
  }
  push @commands, "quit";
  $self->addCommands(@commands);

  # 返回计算结果
  return $poolName;
}

#------------------------------------------------------------------------------
# 查询或创建地址转换
#------------------------------------------------------------------------------
sub createDyNat { }

#------------------------------------------------------------------------------
# 查询或创建地址转换
#------------------------------------------------------------------------------
sub createStaticNat { }

#------------------------------------------------------------------------------
# 创建原地址对象
#------------------------------------------------------------------------------
sub createSrc { }

#------------------------------------------------------------------------------
# 创建目的地址对象
#------------------------------------------------------------------------------
sub createDst { }

#------------------------------------------------------------------------------
# 创建服务端口对象
#------------------------------------------------------------------------------
sub createSrv { }

__PACKAGE__->meta->make_immutable;
1;
