#
#
#  Copyright (c) 2003 Andrew W. Speer <andrew.speer@isolutions.com.au>. All rights 
#  reserved.
#
#  This file is part of WebMod::Proto.
#
#  WebMod::Proto is free software; you can redistribute it and/or modify
#  it under the terms of the GNU General Public License as published by
#  the Free Software Foundation; either version 2 of the License, or
#  (at your option) any later version.
#
#  This program is distributed in the hope that it will be useful,
#  but WITHOUT ANY WARRANTY; without even the implied warranty of
#  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#  GNU General Public License for more details.
#
#  You should have received a copy of the GNU General Public License
#  along with this program; if not, write to the Free Software
#  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
#
#
#  $Id: Proto.pm,v 1.4 2003/11/03 02:45:39 aspeer Exp $

#
#  Assist in prototyping subroutine calls
#
package WebMod::Proto;


#  Compiler pragma
#
use strict 	qw(vars);
use vars 	qw($VERSION $REVISION @ISA %EXPORT_TAGS $TRUE);


#  External Modules. We 'require' rather than use the Err module
#  as we do not want to run the import() routine in this case
#
require WebMod::Err;


#  Require the Exporter
#
require Exporter;


#  Inheritance
#
@ISA=qw(Exporter);


#  Version Info, must be all one line for MakeMaker, CPAN.
#
$VERSION = eval { require WebMod::Proto::VERSION; do $INC{'WebMod/Proto/VERSION.pm'}};


#  Release info
#
$REVISION = (qw$Revision: 1.4 $)[1];


#  Export the proto function
#
%EXPORT_TAGS=(
	all	=>	['proto']
);
&Exporter::export_ok_tags('all');


#  Var to hold class cache
#
my $class;


#  Temp debugging
#
#use IO::File;
#my $fh=IO::File->new('proto.log', O_WRONLY|O_CREAT|O_TRUNC) || die $!;


#  All done, return OK
#
return OK;


#============================================================================


sub proto {


    #  Revised prototyping function. Works something like this (synopsis)
    #
    #  $param=class->proto(['hydro=$', 'heli:$', 'beri=$*', 'boro=!', 'nitr=@',
    #		{ validate=>1 }..], \@_) || die;
    #
    #  Where param is returned as a hash ref where params are available as
    #
    #  $param->{'hydro'}, $param->{'heli'} etc.
    #
    #  Where the format is ->proto( param template array ref, param ref )
    #
    #  And the param template array ref contains string rules regarding the
    #  paramater list, listed below. All values are assigned to the 'name' param
    #
    #  name=$		Mandatory scalar value
    #  name:$		Optional scalar value
    #  name=$$		Mandatory scalar ref
    #  name:$$		Optional scalar ref, all templates below also have optional (:) modifier
    #  name=@		Array ref
    #  name=%		Hash ref
    #  name=.		Any value type, must be non-null,
    #  name:.		Makes no sense (any value, optional)
    #  name=!		Boolean param, where /yes|on|ok|1+/=1, /no|off|undef|0/=undef
    #  name..*      Any of above terminated with * makes that var the default.
    #  name+{$$|@|%} Mandatory non-empty scalar/array/hash ref
    #
    #  {}			A hash ref in any position within the stream of the prototyping
    # 				modifies the behaviour of the parser, the commands being:
    #
    #  validate=>[0|1]	Where 1 means that no invalid paramaters are allowed, and will
    # 					cause an error.
    #
    #  xlate=>{old=>new} Changes param old into param new, does prototyping against
    # 					 new
    #
    #  struct=>name		Name of a class that contains the param data
    #
    #  override=>param	Name of a param that is a hash ref that will override any
    #					params already supplied
    #
    #print $fh sprintf("caller %s, line %s\n", (caller(1))[3,2]);


    #  Prototype template and param array ref are the arguments for routine
    #
    my ($proto_ar, $param_ar)=@_;


    #  Debug
    #
    #debug('proto_ar %s, param_ar %s', Dumper($proto_ar), Dumper($param_ar));


    #  Get the caller info first
    #
    my $caller_ar=[caller(0)];


    #  Debug
    #
    #debug("caller $caller_ar");


    #  Some basic usage checks, check for two defined args exactly
    #
    unless( $proto_ar && $param_ar && (@_ == 2)) {


	#  Something not right, return usage instructions
	#
	return err('usage proto([template], \@_)');


    }


    #  Check both args are ARRY refs or fail, check that prototyping
    #  template is not empty. NOTE no longer do this to allow params
    #  to be formatted, even if no template supplied
    #
    unless ( scalar(grep { ref($_) eq 'ARRAY'} @_)== 2 ) {
	return err('non array ref (%s) template supplied to prototyping template/param ref ',
		   ref($proto_ar) ? ref($proto_ar) : ref($param_ar) || 'undefined')
    }



    #  Parse the param template, storing attribute params into hashes etc.
    #
    my @template=grep { !ref($_) } @{$proto_ar};


    #  Get calling method
    #
    my $caller_method=(caller(1))[3];


    #  Look for cached results
    #
    unless ($class->{'_cache'}{$caller_method}) {



	#  Vars to hold results from parsing routine below
	#
	my %template; my %mandatory; my %noempty; my %protoclass; my %parser;


	#  Read in the prototype parsing modifiers (ie any hash ref's in the
	#  prototype template), then store them in a hash for later reference.
	#  Only the first match is looked at, all other HASH ref's are
	#  ignored.
	#
	my $parser_hr=(grep { ref($_) eq 'HASH'} @{$proto_ar})[0];
	map { $parser{$_}=$parser_hr->{$_}} keys %{$parser_hr};
	#debug('parser %s', Dumper($parser_hr));


	#  Optimisation to stop stale self scalar ref var from being kept
	#
 	if ($parser{'self'}) {
 	    for (0..$#{$proto_ar}) {
 		if ($proto_ar->[$_] eq $parser_hr) { $parser{'self'}=$_; last }
 	    }
 	}


	#  Debug
	#
	#debug('parser self %s %s', $parser{'self'}, Dumper($proto_ar));


	#  Go through template
	#
	foreach my $template (@template) {


	    #  Parse the template item
	    #
	    if ($template=~/(\w+)(\=|\:|\+)([\*|\$|\@|\%|\.|\!|\&])(.*?)([\*|\$]?)$/o) {


		#  Successfully matched, set temp vars to results
		#
		my ($name, $mandatory, $type, $class, $default)=($1, $2, $3, $4, $5);


		#  Now run through a series of tests, setting various parser
		#  flags and hash's as needed
		#
		if ($default eq '*') {
		    #$parser{'default'} ||=[];
		    push @{$parser{'default'}}, $name;
		}


		if ($default eq '$') {
		    $template{$name}='$$';
		} else {
		    $template{$name}=$type;
		}

		if ($type eq '*') {
		    $protoclass{$name}=$class
		}

		if ($mandatory eq '=') {
		    $mandatory{$name}++;
		}
		if ($mandatory eq '+') {
		    $mandatory{$name}++; $noempty{$name}++
		}


	    } else {

		#  We could not parse this particular template
		#
		return err("unable to parse prototype template '$template'")

	    }
	}


	#  Store into cache
	#
	@{$class->{'_cache'}{$caller_method}}
	    {qw(template_hr mandatory_hr noempty_hr protoclass_hr parser_hr)}=
		(\%template, \%mandatory, \%noempty, \%protoclass, \%parser);

    }



    #  Retrieve from cache
    #
    my ($template_hr, $mandatory_hr, $noempty_hr, $protoclass_hr, $parser_hr)=
	@{$class->{'_cache'}{$caller_method}}
	    {qw(template_hr mandatory_hr noempty_hr protoclass_hr parser_hr)};
    #delete $class->{'_cache'};


    #  Does the user want a self ref returned as well ?
    #
    if (my $self_sr=$proto_ar->[$parser_hr->{'self'}]->{'self'}) {


	#debug("self_sr $self_sr");


	#  Check that the self param has been passed a scalar ref, otherwise
	#  bitch about it
	#
	unless (ref($self_sr) eq 'SCALAR') {

	    #  Chuck a wobbly
	    #
	    return err("self ref param must point to a scalar ref, not $self_sr");


	}


	#  Yes the user does
	#
	my $self_caller_ar=[caller(1)];
	${$self_sr}=&self_or_class($self_caller_ar->[0], $caller_ar, $param_ar);


    }


    #  At this point we have read and parsed the template, but done nothing with
    #  the paramater ref supplied. In most cases the paramater ref will be a hash
    #  ref contiaming the param key/value pairs the function wants. In other cases,
    #  the param ref may be a plain scalar, or list of scalars (or something else),
    #  the intention of which is to supply a default value. Eg
    #
    #  Usage 1. $self->foobar({firstname='Andrew', lastname=>'Speer'})
    #
    #  Usage 2. $self->foobar('Andrew', 'Speer');
    #
    #  Attempt to cater for this type of usage here
    #
    #
    #my $param_hr=(ref($param_ar->[0]) eq 'HASH') && shift(@{$param_ar}) || {};
    my %param=%{(ref($param_ar->[0]) eq 'HASH') &&  shift(@{$param_ar})};


    #  Look for any default values
    #
    foreach my $default (@{$parser_hr->{'default'}}) {


	#  If the default param we are looking at already
	#  has a value, then we will skip it
	#
	if (exists $param{$default}) {
	    next;
	}


	if (grep {$template_hr->{$default} eq $_} qw($ .)) {


	    #  Suck in one scalar and loop, but only if there is something
	    #  to pull in
	    #
	    if (my $value=shift(@{$param_ar})) {
		$param{$default}=$value
	    }
	    next;


	}


	if ($template_hr->{$default} eq '@') {


	    #  Suck in the rest of the params as an array
	    #
	    $param{$default}=(ref($param_ar->[0]) eq 'ARRAY')
		?	shift(@{$param_ar})
		    :	[];
	    for (0 .. $#{$param_ar}) {
		push @{$param{$default}}, $param_ar->[$_];
	    }


	    #  Done all we need now, as all params have been sucked
	    #  in
	    #
	    last;

	}


	if ($template_hr->{$default} eq '%') {


	    #  Special case. If the hash ref is both default *and* mandatory,
	    #  then it is difficult to distinguish from a param ref, eg
	    #
	    #  $self->foo({'xlate'=>{a=>b}}); #  CASE 1 - Looks similar to
	    #  $self->foo({a=>b}); # CASE 2 - where foo has a proto of
	    #
	    #  ->proto(['xlate=%*'])
	    #
	    #  So, what to do - we have already assumed the return hash ref
	    #  is the first param, ie for CASE 2, return is now {a=>b} !
	    #  Not what we want. So, now try to rectify that case.
	    #
	    if ($mandatory_hr->{$default}) {


		#  OK, tricky stuff. The hash is mandatory. We will check for
		#  a hash ref as the next param. If it is not present, *AND*
		#  the return hash does not already contain this mandatory
		#  param, we will assume that the return hash is actually the
		#  default we are after. Confused yet ?
		#
		#
		if ( !scalar(@{$param_ar}) && !exists($param{$default})) {
		    my $data_hr=\%param; undef %param; %param=($default=>$data_hr);
		}

	    }


	    #  Suck in the rest of the params as an hash, unless the next is
	    #  already a hash ref.
	    #
	    $param{$default} ||= (ref($param_ar->[0]) eq 'HASH')
		?	shift(@{$param_ar})
		    :	{};
	    while (shift(@{$param_ar})) {
		$param{$default}{$_}=shift(@{$param_ar});
	    }
	    last;

	}

    }



    #  Take care of any paramater translation at this point, if the user has
    #  supplied an xlation hash ref
    #
    if (my $xlate=$parser_hr->{'xlate'}) {


	#  Check that the xlate param is a hash ref
	#
	if (ref($xlate) eq 'HASH') {

	    #  Now run through and do xlation of paramaters
	    #
	    while (my($old,$new)=each(%{$xlate})) {
		if (exists $param{$old}) {
		    $param{$new}=delete $param{$old};
		}
	    }
	} else {

	    #  Must be present, but not a hash ref so fail
	    #
	    return err('xlate parser param must be a HASH ref, was %s',
		       ref($xlate) || 'plain scalar')

	}

    }


    #  Now validate if necessary
    #
    if ($parser_hr->{'validate'}) {


	#  Yes, user wants validation. There can be no other paramaters besides those
	#  named in the prototype
	#
	my %nonvalid;
	foreach my $param (keys %param) {

	    #  Check that the param is in the
	    #
	    $template_hr->{$param} || $nonvalid{$param}++


	}


	#  Check if we had any culprits
	#
	if (my $count=scalar(grep {$_} keys %nonvalid)) {
	    return err('%s paramater%s %s invalid for this method, caller %s',
		       join(',', map {"'$_'"} keys %nonvalid),
		       ($count>1) ? ('s','are') : ('','is'));
	}
    }


    #  Set up a hash ref to rules which will be applied to all paramters. Because this
    #  is a lot of sub ref's, we do not want to set it up every iteration. Therefore
    #  the code ref's are cached in the class ref for future use
    #
    my $proto_cr_hr=($class->{'_proto_cr_hr'} ||= {


	'$'	=> sub {

	    #  Check param is a plain scalar
	    #
	    if (my $ref=ref($_[0]->{$_[1]})) {

		#  No, is a ref. Fail
		#
		return err({callstack=>2}, "param '$_[1]' is $ref %s ref, should be plain scalar")

	    }
	    if ($noempty_hr->{$_[1]} && !defined($_[0]->{$_[1]})) {

		#  Is empty. Fail
		#
		return err({callstack=>2}, "param '$_[1]' cannot be undefined");

	    }
	    return 1;
	},


	'$$'=> sub {

	    #  Check param is a scalar ref
	    #
	    unless ((my $ref=ref($_[0]->{$_[1]})) eq 'SCALAR') {

		#  Not quite a fail yet. If not a ref we will convert to a
		#  scalar ref
		#
		unless ($ref) {

		    #  OK, is not a ref, so must be a plain scalar. Convert. !! DANGER !!
		    #  self referencing ref, no garbage collection ?
		    #
		    #my $var=$param{$_[1]};
		    #$param{$_[1]}=\$var;
		} else {
		    return err({callstack=>2}, "param '$_[1]' is $ref ref, should be SCALAR ref")
		}
	    }
	    if ($noempty_hr->{$_[1]} && !defined(${$_[0]->{$_[1]}})) {

		#  Empty scalar ref, fail
		#
		return err({callstack=>2}, "param '$_[1]' cannot be an undefined SCALAR ref")

	    }
	    return 1;
	},


	'@' => sub {
	    unless ((my $ref=ref($_[0]->{$_[1]})) eq 'ARRAY') {

		#  Not quite a fail yet. If a plain scalar we will convert to an
		#  array ref of one element automatically
		#
		unless ($ref) {

		    #  OK, is not a ref, so must be plain scalar. Convert
		    #
		    $_[0]->{$_[1]}=[$_[0]->{$_[1]}]

		} else {

		    #  Fail
		    #
		    return err({callstack=>2}, "param '$_[1]' is $ref ref, should be ARRAY ref")

		}
	    }


	    #  Check for some content
	    #
	    if ($noempty_hr->{$_[1]} && !scalar(@{$_[0]->{$_[1]}})) {

		#  Empty array ref, fail
		#
		return err({callstack=>2}, "param '$_[1]' cannot be an empty ARRAY ref")

	    }
	    return 1;
	},


	'%' => sub {

	    #  Check content is a hash ref
	    #
	    unless ((my $ref=ref($_[0]->{$_[1]})) eq 'HASH') {

		#  Not quite a fail yet. If an array ref we will convert to hash
		#
		if ($ref eq 'ARRAY') {

		    #  Yep, array ref, coerce to hash ref
		    #
		    $_[0]->{$_[1]}={@{$_[0]->{$_[1]}}};

		} else {

		    #  Fail
		    #
		    unless ($ref) {
			return err({callstack=>2}, "param '$_[1]' is not a ref, should be HASH ref")
		    } else {
			return err({callstack=>2}, "param '$_[1]' is $ref ref, should be HASH ref")
		    }
		}
	    }

	    #  Check for some content
	    #
	    if ($noempty_hr->{$_[1]} && !scalar(keys %{$_[0]->{$_[1]}})) {

		#  Empty HASH ref, fail
		#
		return err({callstack=>2}, "param '$_[1]' cannot be an empty HASH ref")

	    }
	    return 1;

	},


	'&' => sub {

	    #  Must be a code ref
	    #
	    unless ((my $ref=ref($_[0]->{$_[1]})) eq 'CODE') {

		#  Fail
		#
		unless ($ref) {
		    return err({callstack=>2}, "param '$_[1]' is not a ref, should be CODE ref")
		} else {
		    return err({callstack=>2}, "param '$_[1]' is $ref ref, should be CODE ref")
		}
	    }
	},



	'.' => sub {

	    #  Always true, any value (even undef) allowed
	    #
	    return 1;

	},


	'!' => sub {

	    #  Always true, but convert to boolean value
	    #
	    if ($_[0]->{$_[1]}=~/no|off|disable/i) {
		$_[0]->{$_[1]}=undef
	    }
	    else {
		$_[0]->{$_[1]}=$_[0]->{$_[1]} ? 1 : undef;
	    }

	    return 1;
	},


	'*' => sub {

	    #  Must be a class ref
	    #
	    my $refclass=$protoclass_hr->{$_[1]} ||
		return err({callstack=>2}, "unable to determine ref type for param '$_[1]'".
			       'from proto template');

	    #  Check that the ref types match
	    #
	    unless ((my $ref=ref($_[0]->{$_[1]})) eq $refclass) {
		unless ($ref) {
		    return err({callstack=>2}, "param '$_[1]' is not a ref at all (%s), ".
				   'but a $refclass ref is required',
			       $_[0]->{$_[1]})
		}
		else {
		    return err({callstack=>2}, "param '$_[1]' is $ref ref, not $refclass ".
				   'ref as required')
		}
	    }
	    return 1;
	},

    });


    #  OK, by now we should have taken care of assigning loose paramaters etc,
    #  now we need to do some prototype checking, using the above code hash
    #
    foreach my $param (keys %{$template_hr}) {


	#  Check mandatory status first
	#
	if ($mandatory_hr->{$param}) {
	    unless (exists( $param{$param} ) ) {
		return err({callstack=>1}, "missing mandatory paramater '$param'") 
	    }
	}


	#  Skip if empty
	#
	next unless $param{$param};


	#  And run
	#
	$proto_cr_hr->{$template_hr->{$param}}->(\%param, $param) ||
	    return err();


    }


    #  All done, return the hash ref
    #
    return \%param;


}


sub self_or_class {


    #  Get the method the caller was looking for
    #
    my ($class, $caller_ar, $param)=@_;


    #  Set default caller info if not supplied
    #
    $caller_ar ||= [caller(1)];


    #  Get the method that the caller wants to use
    #
    my $method=(reverse split(/::/,$caller_ar->[3]))[0];


    #  Work out if we want the self or class for a method. Default
    #  is class
    #
    my $return=$class;


    #  Small block to work out if the caller can entertain the
    #  method asked for with its self ref
    #
    {

	if (UNIVERSAL::can($param->[0], $method)) {
	    $return=shift(@{$param}); last;
	}

	if (UNIVERSAL::isa($param->[0], $caller_ar->[0])) {
	    $return=shift(@{$param}); last;
	}

	if (ref($param->[0]) eq $caller_ar->[0]) {
	    shift(@{$param}); last
	}

    }


    #  And return
    #
    return $return

}


#sub debug {

#    use Data::Dumper;
#    printf(shift()."\n", @_);

#}


sub err {


    #  Go direct to the WebMod::Err::err routine, using goto
    #  to preserve important callback info
    #
    goto(&WebMod::Err::err);


}


