package Promises6::Resolver;
use Evo::Base -base;
use Promises6::Util ':all';
use Scalar::Util;
use Hash::Util::FieldHash 'fieldhash';

# that's the harder part of specification, because some paragraphs are not so obvious
# even JS Q doesn't implement it in the right way. But we'll take a try
#
# SpecA+ .. a thenable that participates in a circular thenable chain..
# we do reject in cb because if somehow the same thenable won't call
# resolve cb, than there are no infinite. Also race can call the same
# thenable twice.

# if then will call resolve or reject with a value and then die,
# our deferred will remain unchanged, because will be fulfilled with a value
# but we need to track a case when then calls $resolved and die, and send_msg
# is subclassed with next_tick. because we use the same deferred, wich
# remains PENDING.
# So we're going to use a counter and ignore an error if $resolve was called
#
# Also we have cover a case when thenable calls our callback twice, first time
# with thenable or other promise, and second with a plane value. We should
# ignore the second call. So we track the fact that callback was called on both
# onResolve and onReject. see then/thenable_call_cbs_several_times.t


fieldhash my %THENABLES;
has 'builder';

sub thenables { $THENABLES{$_[1]} //= {}; }

sub resolve($self, $deferred, $val) {

  # not a promise, but deferred protect from mem leaks
  return $deferred->reject("circular reference detected $val")
    if $val && Scalar::Util::blessed $val && $val eq $deferred;

  return $self->resolve_promise($deferred, $val) if is_promise($val);
  return $self->resolve_thenable($deferred, $val) if is_thenable($val);
  $deferred->change_state(FULFILLED, $val);
}

sub resolve_promise($self, $deferred, $promise) {
  my $xd = $promise->deferred;

  return $deferred->reject("circular reference detected $promise")
    if $xd == $deferred;
  return $deferred->change_state($xd->result->@*) if $xd->state != PENDING;

  $xd->subscribe($deferred->builder->listener(deferred => $deferred));
  $deferred->is_adaptor(1);
}

sub resolve_thenable($self, $deferred, $thenable) {

  my $was_called = 0;
  my $registered = $self->thenables($deferred);
  $registered->{$thenable}++;

  # resolved cb, if called exceptions will be skipped after that
  my $resolve = sub($v) {
    return if $was_called++;
    return $deferred->reject("Circular thenable $v") if $registered->{$v}++;
    $deferred->resolve($v);
  };

  # reject immidiately changes a state
  my $reject = sub($r) {
    return if $was_called++;
    $deferred->reject($r);
  };

  # resolution 3.III.d
  my $err;
CATCH: {
    local $@;
    eval { $thenable->then($resolve, $reject) };
    $err = $@;
  }
  $deferred->reject($err) if $err && !$was_called;
}


1;

__END__

=pod

=encoding UTF-8

=head1 NAME

Promises6::Resolver

=head1 VERSION

version 0.005

=head1 ATTRIBUTES

=head2 builder

An instance of L<Promises6::Builder>

=head1 METHODS

=head2 resolve
=method resolve_promise
=method resolve_thenable
=method thenables

Internal methods

=head1 AUTHOR

alexbyk <alexbyk.com>

=head1 COPYRIGHT AND LICENSE

This software is copyright (c) 2015 by alexbyk.

This is free software; you can redistribute it and/or modify it under
the same terms as the Perl 5 programming language system itself.

=cut
