# Licensed to the Apache Software Foundation (ASF) under one or more
# contributor license agreements.  See the NOTICE file distributed with
# this work for additional information regarding copyright ownership.
# The ASF licenses this file to You under the Apache License, Version 2.0
# (the "License"); you may not use this file except in compliance with
# the License.  You may obtain a copy of the License at
#
#     http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.

use strict;
use warnings;

package Clownfish;
use Clownfish::Base;
our $VERSION = '0.01';

BEGIN {
    push @Clownfish::CBlock::ISA,      "Clownfish::Base";
    push @Clownfish::DocuComment::ISA, "Clownfish::Base";
    push @Clownfish::File::ISA,        "Clownfish::Base";
    push @Clownfish::Parcel::ISA,      "Clownfish::Base";
    push @Clownfish::Symbol::ISA,      "Clownfish::Base";
    push @Clownfish::Type::ISA,        "Clownfish::Base";
}

use XSLoader;
BEGIN { XSLoader::load( 'Clownfish', '0.01' ) }

1;

=head1 NAME

Clownfish - A small OO language that forms symbiotic relationships with "host"
languages.

=head1 PRIVATE API

Clownfish is an Apache Lucy implementation detail.  This documentation is partial --
enough for the curious hacker, but not a full API.

=head1 DESCRIPTION

=head2 Overview.

Clownfish is a small language for declaring an object oriented interface and a
compiler which allows classes to be implemented either in C, in a "host"
language, or a combination of both. 

=head2 Why use Clownfish?

=over

=item *

Clownfish-based projects give users the ability to write full subclasses
in any "host" language for which a binding has been prepared.

=item *

Pure C Clownfish class implementations are very fast.

=item *

Users can perform rapid prototyping in their language of choice, then port
their classes to C either for speed or to make them available across multiple
language platforms.

=item *

=back

=head2 Object Model

Clownfish is single-inheritance and class based -- a minimalist design which
makes it as compatible as possible with a broad range of hosts.

Subclasses may be created either at compile time or at run time.

=back

=head2 C method invocation syntax.

Methods are differentiated from functions via capitalization:
Boat_capsize() is a function, Boat_Capsize() is a method.

    // Base method.
    void
    Boat_capsize(Boat *self)
    {
        self->upside_down = true;
    }

    // Implementing function, in Boat/Battleship.c
    void
    Battleship_capsize(Battleship *self) 
    {
        // Superclass method invocation.
        Boat_capsize_t capsize = (Boat_capsize_t)SUPER_METHOD(
            BATTLESHIP, Battleship, Capsize);
        capsize((Boat*)self);  

        // Subclass-specific behavior.
        Battleship_Sink(self);
    }

    // Implementing function, in Boat/RubberDinghy.c
    void
    RubDing_capsize(RubberDinghy *self) 
    {
        // Superclass method invocation.
        Boat_capsize_t capsize = (Boat_capsize_t)SUPER_METHOD(
            RUBBERDINGHY, RubDing, Capsize);
        capsize((Boat*)self);  

        // Subclass-specific behavior.
        RubDing_Drift(self);
    }

=head2 Class declaration syntax

    [final] [inert] class CLASSNAME [cnick CNICK] 
        [inherits PARENT] [ : ATTRIBUTE ]* {
    
        [declarations]
    
    }

Example:

    class Boat::RubberDinghy cnick RubDing inherits Boat {
        
        public inert incremented RubberDinghy*
        new();
        
        void 
        Capsize(RubberDinghy *self);
    }

=over

=item * B<CLASSNAME> - The name of this class.  The last string of characters
will be used as the object's C struct name.

=item * B<CNICK> - A recognizable abbreviation of the class name, used as a
prefix for every function and method.

=item * B<PARENT> - The full name of the parent class.

=item * B<ATTRIBUTE> - An arbitrary attribute, e.g. "dumpable", or perhaps
"serializable".  A class may have multiple attributes, each preceded by a
colon.

=back

=head2 Memory management

At present, memory is managed via a reference counting scheme, but this is not
inherently part of Clownfish.

=head2 Namespaces, parcels, prefixes, and "short names"

There are two levels of namespacing in Clownfish: parcels and classes.

Clownfish classes intended to be published as a single unit may be grouped
together using a "parcel".  Parcel directives need to go at the top of each
class file.

    parcel Crustacean cnick Crust;

All symbols generated by Clownfish for classes within a parcel will be
prefixed by varying capitalizations of the parcel's C-nickname or "cnick" in
order to avoid namespace collisions with other projects.

Within a parcel, the last part of each class name must be unique.

    class Crustacean::Lobster::Claw { ... }
    class Crustacean::Crab::Claw    { ... } // Illegal, "Claw" already used

"Short names" -- names minus the parcel prefix -- will be auto-generated for
all class symbols.  When there is no danger of namespace collision, typically
because no third-party non-system libraries are being pound-included, the
short names can be used after a USE_SHORT_NAMES directive:

    #define CRUST_USE_SHORT_NAMES

The USE_SHORT_NAMES directives do not affect class prefixes, only parcel
prefixes.

    // No short names.
    crust_LobsterClaw *claw = crust_LobClaw_new();
    
    // With short names.
    #define CRUST_USE_SHORT_NAMES
    LobsterClaw *claw = LobClaw_new();

=head2 Inclusion

C header code generated by the Clownfish compiler is written to a file with
whose name is the same as the .cfh file, but with an extension of ".h".  C
code should pound-include "Crustacean/Lobster.h" for a class defined in
"Crustacean/Lobster.cfh".

=head1 COPYRIGHT 
 
Clownfish is distributed under the Apache License, Version 2.0, as 
described in the file C<LICENSE> included with the distribution. 

=cut

