// 
//  Copyright (C) 1995,2007,2010  Smithsonian Astrophysical Observatory
//
//
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 3 of the License, or
//  (at your option) any later version.
//
//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License along
//  with this program; if not, write to the Free Software Foundation, Inc.,
//  51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
//

/*H****************************************************************************
 
* FILE NAME: %M%
 
* DEVELOPEMENT: UI
 
* DESCRIPTION:
 
        This file conatins the class definition for the Host Level 
	Parameter interface base class. This class incapsulates a parameter
	in the IRAF Compatible Parameter interface library into a class.
 
* NOTES:
 
        This class is based on the IRAF Compatible Parameter interface.
	This class does not support the structure option.
 
* REVISION HISTORY:
 
        Ref. No.        Date
        --------        ----
        %I%             %G%
 
*H****************************************************************************/

#ifndef PARAMETER_ITEM_HH
#define PARAMETER_ITEM_HH

#include <list>
#include <string>

#include <string.h>

using namespace std;

#include "parameter.h"		/* needs the ansi compliant prototypes */

// delimeters for the print function
extern const char DEFAULT_DELIM;	// comma
extern const char FORMAT_DELIM;	// another character such as '`'

// The max length a parameter enter can be in a file.
extern const int MAX_PARAM_LENGTH;

// mode settings for parameter entries 
extern const int NO_MODE;
extern const int AUTO_MODE;
extern const int HIDDEN_MODE;
extern const int LEARN_MODE;
extern const int QUERY_MODE;

// parameter INDEF values 
extern const double INDEF_REAL;	// defined as INDEFD in parameter interface  
extern const int INDEF_INT;	// defined as INDEFI in parameter interface  

// extern our string conversion utils and strcmp
extern int _cxcparam_string_to_double (string val, double &outv);
extern int _cxcparam_string_to_long (string val, long &outv);
extern int _cxcparam_strcmp_loose (string str1, string str2);


// for now these are #defines, but perhaps they should be const's
#define MIN_MAX_CHECK            0
#define MIN_ONLY_CHECK           1
#define MAX_ONLY_CHECK           2
#define DEFAULT_CHECK            3
#define SUBSET_CHECK             4
#define NO_RANGE_CHECK           5

#ifdef STRING_TYPE
#undef STRING_TYPE
#endif

class ParamItem
{

  friend ostream & operator<< (ostream &, ParamItem &);
  friend istream & operator>> (istream &, ParamItem &);

public:
  // The different type of parameters
  enum ParamType
  { STRING_TYPE,
    FILENAME_TYPE,
    PSET_TYPE,
    BOOLEAN_TYPE,
    INTEGER_TYPE,
    REAL_TYPE,
    NO_PARAM_TYPE
  };

  // Constructor
    ParamItem ();
    ParamItem (const ParamItem & item);

  // Destructor
    virtual ~ ParamItem ();

  // Accessor functions
  bool Changed ()
  {
    return changed;
  };
  string GetName ()
  {
    return name;
  };
  ParamType GetType ()
  {
    return type;
  };
  int GetMode ()
  {
    return mode;
  };
  string GetPrompt ()
  {
    return prompt;
  };
  int GetPosition ()
  {
    return position;
  };

  // mode related methods 
  int CalculateMode (char *val);
  bool IsAutoMode ()
  {
    return ((mode & AUTO_MODE) == AUTO_MODE);
  }
  bool IsHiddenMode ()
  {
    return ((mode & HIDDEN_MODE) == HIDDEN_MODE);
  }
  bool IsLearnMode ()
  {
    return ((mode & LEARN_MODE) == LEARN_MODE);
  }
  bool IsQueryMode ()
  {
    return ((mode & QUERY_MODE) == QUERY_MODE);
  }

  // set the delimiter if not using the default
  void SetDelim (char tmp_delim)
  {
    delim = tmp_delim;
  }

  // set position of parameter item for display
  void SetPosition (int tmp_pos)
  {
    position = tmp_pos;
  }

  // set the parameter file descriptor
  void SetParam (paramfile * par_file)
  {
    parfile = par_file;
  }
  paramfile *GetParam ()
  {
    return parfile;
  };
  void SetFilename (char *filename);
  const char *GetFilename ()
  {
    return file_name.c_str ();
  }

  // Returns a char** of the fields in the parameter
  char **GetChars (int *count = NULL);

  // Returns the character lengths of each field in the parameter.
  short *GetWidths ();

  // Returns a True if the ParamItem is valid, otherwise it returns a False
  bool IsValid ()
  {
    return ((name.length () > 0) && (type != NO_PARAM_TYPE));
  }

  // Sets the value for the parameter, returns a 0 status if the new
  // value doesn't fall with in the limits.
  virtual int SetValue (string val);
  virtual int SetName (const char *passed_name)
  {
    name = passed_name;
    return (0);
  }
  virtual int SetPrompt (const char *passed_prompt)
  {
    prompt = passed_prompt;
    return (0);
  }

  // Returns the new value in string format.
  virtual string GetNewValue ();

  // evaluate indirection as a string 
  string ComputeIndirString (char *redir);

  // Print the parameter to the output stream
  // Note: Prints only the name, type, and mode of the parameter.
  virtual void Write (ostream &);
  virtual void Print (ostream &)
  {
  };
  virtual void PrintType (ostream &);
  virtual void PrintMode (ostream &);

  // Read the parameter from the input stream.
  // Note: Reads only the name, type, and mode of the parameter.
  virtual void Read (istream &);
  virtual void Read (char *line);

protected:
  // Objects
  bool changed;
  string name;
  ParamType type;
  int mode;
  string prompt;
  int position;			// display index for parameter 
  // (original position in file) 
  string delim;

  paramfile *parfile;
  string file_name;
};

class ParamBoolean:public ParamItem
{
public:
  // Constructor
  ParamBoolean ();
  ParamBoolean (const ParamItem & item);
    ParamBoolean (const ParamBoolean & item);

  // Destructor
    virtual ~ ParamBoolean ();

  // Sets the value for the parameter, returns a 0 status if the new
  // value doesn't fall with in the limits.
  int SetValue (bool val);
  virtual int SetValue (string val);

  bool EvaluateIndir (char *val);	// determine value of indirection

  // Accessor functions
  bool GetValue ()
  {
    return value;
  };
  bool GetMinValue ()
  {
    return min_value;
  };
  bool GetMaxValue ()
  {
    return max_value;
  };
  bool GetCurrent ()
  {
    return current;
  };
  char *GetRedirectCurrent ()
  {
    return cur_redir;
  };
  bool RedirectChange ()
  {
    return (strcmp (cur_redir, val_redir));
  }
  char *GetRedirectValue ()
  {
    return val_redir;
  };

  // Returns the new value in string format.
  virtual string GetNewValue ();

  // Print the parameter to the output stream (default and current value)
  virtual void Print (ostream &);

  // Write the parameter to the output stream in valid format (current only)
  virtual void Write (ostream &);

  // Read the parameter from the input stream.
  // Note: Reads only the value, min_value, max_value, and prompt
  virtual void Read (istream &);
  virtual void Read (char *line)
  {
    ParamItem::Read (line);
  }				//Avoid compiler warning.

protected:
  // Object
  bool value;
  bool min_value;
  bool max_value;
  bool current;
  char cur_redir[80];		// current field redirect
  char val_redir[80];		// value field redirect
  int mode;

};

class ParamString:public ParamItem
{
public:
  // Constructor
  ParamString ();
  ParamString (const ParamItem & item);
    ParamString (const ParamString & item);

  // Destructor
    virtual ~ ParamString ();

  // Sets the value for the parameter, returns a 0 status if the new
  // value doesn't fall with in the limits.
  virtual int SetValue (string val);

  string EvaluateIndir (char *val);	// determine value of indirection

  // Accessor functions
  string GetValue ()
  {
    return value;
  };
  string GetMinValue ()
  {
    return min_value;
  };
  string GetMaxValue ()
  {
    return max_value;
  };
  string GetCurrent ()
  {
    return current;
  };
  bool RedirectChange ()
  {
    return (strcmp (cur_redir, val_redir));
  }
  int GetSubsetCount ()
  {
    return subset_cnt;
  };
  char **GetSubset ()
  {
    return subset;
  };
  string GetSubsetString ()
  {
    return subset_str;
  };
  int GetRangeCheck ()
  {
    return range_check;
  };
  char *GetRedirectCurrent ()
  {
    return cur_redir;
  };
  char *GetRedirectValue ()
  {
    return val_redir;
  };
  char *GetRedirectMax ()
  {
    return max_redir;
  };
  char *GetRedirectMin ()
  {
    return min_redir;
  };

  // Returns the new value in string format.
  virtual string GetNewValue ();

  // Print the parameter to the output stream
  virtual void Print (ostream &);

  // Write the parameter to the output stream in valid format (current only)
  virtual void Write (ostream &);

  // Read the parameter from the input stream.
  // Note: Reads only the value, min_value, max_value, and prompt
  virtual void Read (istream &);
  virtual void Read (char *line);

  bool is_string_subset (char *buffer);
  void store_subset (char *buffer);

protected:
  // Object
  string value;
  string min_value;
  string max_value;
  string current;
  char **subset;		// array containing subset values
  int subset_cnt;		// number of items in the subset
  int range_check;		// type of range checking- min, max, min&max, subset
  int mode;
  char cur_redir[80];		// current field redirect
  char val_redir[80];		// value field redirect
  char max_redir[80];		// max field redirect
  char min_redir[80];		// min field redirect
  string subset_str;		// string containing subset list 
};

class ParamInteger:public ParamItem
{
public:
  // Constructor
  ParamInteger ();
  ParamInteger (const ParamItem & item);
    ParamInteger (const ParamInteger & item);

  // Destructor
    virtual ~ ParamInteger ();

  // Sets the value for the parameter, returns a 0 status if the new
  // value doesn't fall with in the limits.
  int SetValue (int val);
  virtual int SetValue (string val);

  int EvaluateIndir (char *val);	// determine value of indirection

  // Accessor functions
  int GetValue ()
  {
    return value;
  };
  int GetMinValue ()
  {
    return min_value;
  };
  int GetMaxValue ()
  {
    return max_value;
  };
  int GetCurrent ()
  {
    return current;
  };
  int GetRangeCheck ()
  {
    return range_check;
  };
  bool RedirectChange ()
  {
    return (strcmp (cur_redir, val_redir));
  }
  int GetSubsetCount ()
  {
    return subset_cnt;
  };
  int *GetSubset ()
  {
    return subset;
  };
  string GetSubsetString ()
  {
    return subset_str;
  };
  char *GetRedirectCurrent ()
  {
    return cur_redir;
  };
  char *GetRedirectValue ()
  {
    return val_redir;
  };
  char *GetRedirectMax ()
  {
    return max_redir;
  };
  char *GetRedirectMin ()
  {
    return min_redir;
  };

  // Returns the new value in string format.
  virtual string GetNewValue ();

  // Print the parameter to the output stream
  virtual void Print (ostream &);

  // Write the parameter to the output stream in valid format (current only)
  virtual void Write (ostream &);

  // Read the parameter from the input stream.
  // Note: Reads only the value, min_value, max_value, and prompt
  virtual void Read (istream &);
  virtual void Read (char *line);

  bool is_numeric (char *buffer);	// 0 = no, 1 = yes  
  bool is_numeric_subset (char *buffer);
  void store_subset (char *buffer);


protected:
  // Object
  int value;
  int min_value;
  int max_value;
  int current;
  int *subset;			// array containing subset values 
  int subset_cnt;		// number of items in the subset 
  int range_check;		// type of range checking- min, max, min&max, subset 
  int mode;
  char cur_redir[80];		// current field redirect 
  char val_redir[80];		// value field redirect 
  char max_redir[80];		// max field redirect 
  char min_redir[80];		// min field redirect 
  string subset_str;		// string containing subset list 
};

class ParamReal:public ParamItem
{
public:
  // Constructor
  ParamReal ();
  ParamReal (const ParamItem & item);
    ParamReal (const ParamReal & item);

  // Destructor
    virtual ~ ParamReal ();

  // Sets the value for the parameter, returns a 0 status if the new
  // value doesn't fall with in the limits.
  int SetValue (double val);
  virtual int SetValue (string val);

  double EvaluateIndir (char *val);	// determine value of indirection

  // Accessor functions
  double GetValue ()
  {
    return value;
  };
  double GetMinValue ()
  {
    return min_value;
  };
  double GetMaxValue ()
  {
    return max_value;
  };
  double GetCurrent ()
  {
    return current;
  };
  bool RedirectChange ()
  {
    return (strcmp (cur_redir, val_redir));
  }
  int GetSubsetCount ()
  {
    return subset_cnt;
  };
  double *GetSubset ()
  {
    return subset;
  };
  string GetSubsetString ()
  {
    return subset_str;
  };
  int GetPrecision ()
  {
    return precision;
  };
  int GetRangeCheck ()
  {
    return range_check;
  };
  char *GetRedirectValue ()
  {
    return val_redir;
  };
  char *GetRedirectCurrent ()
  {
    return cur_redir;
  };
  char *GetRedirectMax ()
  {
    return max_redir;
  };
  char *GetRedirectMin ()
  {
    return min_redir;
  };

  int SetPrecision (int val);

  // Returns the new value in string format.
  virtual string GetNewValue ();

  // Print the parameter to the output stream
  virtual void Print (ostream &);

  // Write the parameter to the output stream in valid format (current only)
  virtual void Write (ostream &);

  // Read the parameter from the input stream.
  // Note: Reads only the value, min_value, max_value, and prompt
  virtual void Read (istream &);
  virtual void Read (char *line);

  bool is_decimal_number (char *buffer);
  bool is_decimal_subset (char *buffer);
  void store_subset (char *buffer);

protected:
  // Object
  double value;
  double min_value;
  double max_value;
  double current;
  int mode;
  double *subset;		// array containing subset values 
  int subset_cnt;		// number of items in the subset 
  int range_check;		// type of range checking- min, max, min&max, subset
  int precision;		// number of signif digits after decimal 
  char cur_redir[80];		// current field redirect
  char val_redir[80];		// value field redirect
  char max_redir[80];		// max field redirect
  char min_redir[80];		// min field redirect
  string subset_str;		// string containing subset list 
};

#endif
