#!/usr/bin/env perl

use Bio::GeneDesign;
use Getopt::Long;
use File::Basename;
use Pod::Usage;

use strict;
use warnings;

my $VERSION = '5.55';
my $GDV = "GD_Repeat_Smash_$VERSION";
my $GDS = "_RS";

local $| = 1;

##Get Arguments
my %p = ();
GetOptions (
      'input=s'     => \$p{INPUT},
      'output=s'    => \$p{OUTPUT},
      'format=s'    => \$p{FORMAT},
      'rscu=s'      => \$p{FILES},
      'organism=s'  => \$p{ORGS},
      'help'        => \$p{HELP}
);

################################################################################
################################ SANITY  CHECK #################################
################################################################################
pod2usage(-verbose => 99, -sections => 'NAME|VERSION|DESCRIPTION|ARGUMENTS|USAGE')
  if ($p{HELP});

my $GD = Bio::GeneDesign->new();

#The input file must exist and be a format we care to read.
die "\n GDERROR: You must supply an input file.\n"
  if (! $p{INPUT});
my ($iterator, $filename, $suffix) = $GD->import_seqs($p{INPUT});

$p{FORMAT} = $p{FORMAT} || $suffix || 'genbank';

#The output path must exist, and we'll need it to end with a slash
$p{OUTPUT} = $p{OUTPUT} || q{.};
$p{OUTPUT} = $GD->endslash($p{OUTPUT});
die "\n GDERROR: $p{OUTPUT} does not exist.\n"
  if ($p{OUTPUT} && ! -e $p{OUTPUT});

#We must get a list of organisms or a set of rscu files
die "\n GDERROR: Neither an organism nor an RSCU table were supplied.\n"
  if (! $p{ORGS} && ! $p{FILES});
$p{ORGS}  = $p{ORGS}   || q{};
$p{FILES} = $p{FILES}  || q{};

################################################################################
################################# CONFIGURING ##################################
################################################################################
my @fileswritten;
my @seqstowrite;

my %works = ();
foreach my $org ( split q{,}, $p{ORGS} )
{
  $works{$org} = {on => $org, path => undef};
}
foreach my $file ( split q{,}, $p{FILES} )
{
  $works{$file} = {on => basename($file), path => $file};
}

################################################################################
############################### CODON  JUGGLING ################################
################################################################################
while ( my $obj = $iterator->next_seq() )
{
  foreach my $work (keys %works)
  {
    $GD->set_organism
    (
        -organism_name => $works{$work}->{on},
        -rscu_path     => $works{$work}->{path}
    );
    my $newobj = $GD->repeat_smash(-sequence => $obj);
    push @seqstowrite, $newobj;
    print "$obj; $newobj\n\n";
  }
}
if (scalar @seqstowrite)
{
  push @fileswritten, $GD->export_seqs
  (
    -filepath  => $p{OUTPUT} . $filename . $GDS,
    -sequences => \@seqstowrite,
    -format    => $p{FORMAT},
  );
}

print "\n";
print "Wrote $_\n" foreach @fileswritten;
print "\n";
print $GD->attitude() . " brought to you by $GDV\n\n";

exit;

__END__

=head1 NAME

  GD_Repeat_Smash.pl

=head1 VERSION

  Version 5.55

=head1 DESCRIPTION

  Output will be named according to the name of the input file, and will be
  tagged with _RS.

=head1 USAGE

=head1 ARGUMENTS

Required arguments:

  -i,   --input : a file containing nucleotide sequences.
  -org, --organism : an organism whose RSCU table can be found in the config
      directory, or several separated by commas
    AND/OR
  -r,   --rscu : path to an RSCU table generated by GD_Generate_RSCU_Table.pl

Optional arguments:

  -out, --output : path to an output directory
  -f,   --format : default genbank
  -h,   --help : Display this message

=head1 COPYRIGHT AND LICENSE

Copyright (c) 2015, GeneDesign developers
All rights reserved.

Redistribution and use in source and binary forms, with or without modification,
are permitted provided that the following conditions are met:

* Redistributions of source code must retain the above copyright notice, this
list of conditions and the following disclaimer.

* Redistributions in binary form must reproduce the above copyright notice, this
list of conditions and the following disclaimer in the documentation and/or
other materials provided with the distribution.

* The names of Johns Hopkins, the Joint Genome Institute, the Lawrence Berkeley
National Laboratory, the Department of Energy, and the GeneDesign developers may
not be used to endorse or promote products derived from this software without
specific prior written permission.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND
ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
DISCLAIMED. IN NO EVENT SHALL THE DEVELOPERS BE LIABLE FOR ANY DIRECT, INDIRECT,
INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE
OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF
ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

=cut
