/*
 * ddev.c - SCO UnixWare device support functions for lsof
 */


/*
 * Copyright 1996 Purdue Research Foundation, West Lafayette, Indiana
 * 47907.  All rights reserved.
 *
 * Written by Victor A. Abell
 *
 * This software is not subject to any license of the American Telephone
 * and Telegraph Company or the Regents of the University of California.
 *
 * Permission is granted to anyone to use this software for any purpose on
 * any computer system, and to alter it and redistribute it freely, subject
 * to the following restrictions:
 *
 * 1. Neither the authors nor Purdue University are responsible for any
 *    consequences of the use of this software.
 *
 * 2. The origin of this software must not be misrepresented, either by
 *    explicit claim or by omission.  Credit to the authors and Purdue
 *    University must appear in documentation and sources.
 *
 * 3. Altered versions must be plainly marked as such, and must not be
 *    misrepresented as being the original software.
 *
 * 4. This notice may not be removed or altered.
 */

#ifndef lint
static char copyright[] =
"@(#) Copyright 1996 Purdue Research Foundation.\nAll rights reserved.\n";
static char *rcsid = "$Id: ddev.c,v 1.5 98/03/06 08:39:51 abe Exp $";
#endif


#include "lsof.h"


_PROTOTYPE(static int rmdupdev,(struct l_dev ***dp, int n, char *nm));
_PROTOTYPE(static void stkdir,(char ***d, int *n, int *x, char *p));


#if	defined(HASDCACHE)
/*
 * clr_sect() - clear cached clone and pseudo sections
 */

void
clr_sect()
{
	struct clone *c, *c1;

	if (Clone) {
	    for (c = Clone; c; c = c1) {
		c1 = c->next;
		(void) free((FREE_P *)c);
	    }
	    Clone = NULL;
	}
}
#endif  /* defined(HASDCACHE) */


/*
 * printchdevname() - print character device name
 */

int
printchdevname(rdev, f)
	dev_t *rdev;			/* device */
	int f;				/* 1 = print trailing '\n' */
{
	struct clone *c;
	struct l_dev *dp;

	readdev(0);
/*
 * Search for clone.
 */

#if	defined(HASDCACHE)

printchdevname_again:

#endif	/* defined(HASDCACHE) */

	if (Lf->is_stream && Clone) {
	    for (c = Clone; c; c = c->next) {
		if (major(*rdev) == minor(Devtp[c->dx].rdev)) {

#if	defined(HASDCACHE)
		    if (DCunsafe && !Devtp[c->dx].v && !vfy_dev(&Devtp[c->dx]))
			goto printchdevname_again;
#endif	/* defined(HASDCACHE) */

		    safestrprt(Devtp[c->dx].name, stdout, f);
		    return(1);
		}
	    }
	}
/*
 * Search device table for match.
 */
	if ((dp = lkupdev(rdev, 1, 0))) {
	    safestrprt(dp->name, stdout, f);
	    return(1);
	}

#if	defined(HASDCACHE)
/*
 * If the device cache is "unsafe" and we haven't found any match, reload
 * the device cache.
 */
	if (DCunsafe) {
	    (void) rereaddev();
	    goto printchdevname_again;
	}
#endif	/* defined(HASDCACHE) */

	return(0);
}


/*
 * readdev() - read names, modes and device types of everything in /dev
 */

void
readdev(skip)
	int	skip;			/* skip device cache read if 1 */
{
	struct clone *c;

#if	defined(HASDCACHE)
	int dcrd;
#endif	/* defined(HASDCACHE) */

	DIR *dfp;
	int dn = 0;
	struct dirent *dp;
	char **dstk = (char **)NULL;
	int dx = 0;
	char *fp = (char *)NULL;
	int i = 0;

#if	defined(HASBLKDEV)
	int j = 0;
#endif	/* defined(HASBLKDEV) */

	char *path = (char *)NULL;
	MALLOC_S pl;
	struct stat sb;

	if (Sdev)
	    return;

#if	defined(HASDCACHE)
/*
 * Read device cache, as directed.
 */
	if (!skip) {
	    if (DCstate == 2 || DCstate == 3) {
		if ((dcrd = read_dcache()) == 0)
		    return;
	    }
	} else
	    dcrd = 1;
#endif	/* defined(HASDCACHE) */

	(void) stkdir(&dstk, &dn, &dx, "/dev");
/*
 * Unstack the next /dev or /dev/<subdirectory> directory.
 */
	while (--dx >= 0) {
	    if (!(dfp = opendir(dstk[dx]))) {

#if	defined(WARNDEVACCESS)
		if (!Fwarn) {
		    (void) fprintf(stderr, "%s: WARNING: can't open: ", Pn);
		    safestrprt(dstk[dx], stderr, 1);
		}
#endif	/* defined(WARNDEVACCESS) */

		continue;
	    }
	    if (path) {
		(void) free((FREE_P *)path);
		path = (char *)NULL;
	    }
	    if (!(path = mkstrcat(dstk[dx], -1, "/", 1, (char *)NULL, -1, &pl)))
	    {
		(void) fprintf(stderr, "%s: no space for: ", Pn);
		safestrprt(dstk[dx], stderr, 0);
		(void) fprintf(stderr, "/\n");
	    }
	    (void) free((FREE_P *)dstk[dx]);
	    dstk[dx] = (char *)NULL;
	/*
	 * Scan the directory.
	 */
	    for (dp = readdir(dfp); dp; dp = readdir(dfp)) {
		if (dp->d_ino == 0 || dp->d_name[0] == '.')
		    continue;
	    /*
	     * Form the full path name and get its status.
	     */
		if (fp) {
		    (void) free((FREE_P *)fp);
		    fp = (char *)NULL;
		}
		if (!(fp = mkstrcat(path, pl, dp->d_name, -1, (char *)NULL, -1,
			   (MALLOC_S *)NULL)))
		{
		    (void) fprintf(stderr, "%s: no space for: ", Pn);
		    safestrprt(path, stderr, 0);
		    safestrprt(dp->d_name, stderr, 1);
		    Exit(1);
		}

#if	defined(USE_STAT)
		if (stat(fp, &sb))
#else	/* !defined(USE_STAT) */
		if (lstat(fp, &sb))
#endif	/* defined(USE_STAT) */

		{
		    if (errno == ENOENT)	/* a symbolic link */
			continue;		/* to nowhere? */

#if	defined(WARNDEVACCESS)
		    if (!Fwarn) {
			int errno_save = errno;

			(void) fprintf(stderr, "%s: can't stat ", Pn);
			safestrprt(fp, stderr, 0);
			(void) fprintf(stderr, ": %s\n", strerror(errno_save));
		    }
#endif	/* defined(WARNDEVACCESS) */

		    continue;
		}
	    /*
	     * If it's a subdirectory, stack its name for later
	     * processing.
	     */
		if ((sb.st_mode & S_IFMT) == S_IFDIR) {
		    (void) stkdir(&dstk, &dn, &dx, fp);
			continue;
		}
		if ((sb.st_mode & S_IFMT) == S_IFCHR) {

		/*
		 * Save character device information in Devtp[].
		 */
		    if (i >= Ndev) {
			Ndev += DEVINCR;
			if (!Devtp)
			    Devtp = (struct l_dev *)malloc(
				    (MALLOC_S)(sizeof(struct l_dev)*Ndev));
			else
			    Devtp = (struct l_dev *)realloc((MALLOC_P *)Devtp,
				    (MALLOC_S)(sizeof(struct l_dev)*Ndev));
			if (!Devtp) {
			    (void) fprintf(stderr,
				"%s: no space for character device\n", Pn);
			    Exit(1);
			}
		    }
		    Devtp[i].inode = sb.st_ino;
		    if (!(Devtp[i].name = mkstrcpy(fp, (MALLOC_S *)NULL))) {
			(void) fprintf(stderr, "%s: no space for: ", Pn);
			safestrprt(fp, stderr, 1);
			Exit(1);
		    }
		    Devtp[i].rdev = sb.st_rdev;
		    Devtp[i].v = 0;
		    if (HaveCloneMaj && major(Devtp[i].rdev) == CloneMaj) {

		    /*
		     * Record clone device information.
		     */
			if (!(c = (struct clone *)malloc(sizeof(struct clone))))
			{
			    (void) fprintf(stderr,
				"%s: no space for clone device: ", Pn);
			    safestrprt(fp, stderr, 1);
			    Exit(1);
			}
			c->dx = i;
			c->next = Clone;
			Clone = c;
		    }
		    i++;
		}

#if	defined(HASBLKDEV)
		if ((sb.st_mode & S_IFMT) == S_IFBLK) {

		/*
		 * Save block device information in BDevtp[].
		 */
		    if (j >= BNdev) {
			BNdev += DEVINCR;
			if (!BDevtp)
			    BDevtp = (struct l_dev *)malloc(
				     (MALLOC_S)(sizeof(struct l_dev)*BNdev));
			else
			    BDevtp = (struct l_dev *)realloc((MALLOC_P *)BDevtp,
				     (MALLOC_S)(sizeof(struct l_dev)*BNdev));
			if (!BDevtp) {
			    (void) fprintf(stderr,
				"%s: no space for block device\n", Pn);
			    Exit(1);
			}
		    }
		    BDevtp[j].inode = sb.st_ino;
		    BDevtp[j].name = fp;
		    fp = (char *)NULL;
		    BDevtp[j].rdev = sb.st_rdev;
		    BDevtp[j].v = 0;
		    j++;
		}
#endif	/* defined(HASBLKDEV) */

	    }
	    (void) closedir(dfp);
	}
/*
 * Free any allocated space.
 */
	if (dstk)
	    (void) free((FREE_P *)dstk);
	if (fp)
	    (void) free((FREE_P *)fp);
	if (path)
	    (void) free((FREE_P *)path);
/*
 * Reduce the BDevtp[] (optional) and Devtp[] tables to their minimum
 * sizes; allocate and build sort pointer lists; and sort the tables by
 * device number.
 */

#if	defined(HASBLKDEV)
	if (BNdev) {
	    if (BNdev > j) {
		BNdev = j;
		BDevtp = (struct l_dev *)realloc((MALLOC_P *)BDevtp,
			 (MALLOC_S)(sizeof(struct l_dev) * BNdev));
	    }
	    if (!(BSdev = (struct l_dev **)malloc(
			  (MALLOC_S)(sizeof(struct l_dev *) * BNdev))))
	    {
		(void) fprintf(stderr,
		    "%s: no space for block device sort pointers\n", Pn);
		Exit(1);
	    }
	    for (j = 0; j < BNdev; j++) {
		BSdev[j] = &BDevtp[j];
	    }
	    (void) qsort((QSORT_P *)BSdev, (size_t)BNdev,
		(size_t)sizeof(struct l_dev *), compdev);
	    BNdev = rmdupdev(&BSdev, BNdev, "block");
	} else {
	    if (!Fwarn)
		(void) fprintf(stderr,
		    "%s: WARNING: no block devices found\n", Pn);
	}
#endif	/* defined(HASBLKDEV) */

	if (Ndev) {
	    if (Ndev > i) {
		Ndev = i;
		Devtp = (struct l_dev *)realloc((MALLOC_P *)Devtp,
			(MALLOC_S)(sizeof(struct l_dev) * Ndev));
	    }
	    if (!(Sdev = (struct l_dev **)malloc(
			 (MALLOC_S)(sizeof(struct l_dev *) * Ndev))))
	    {
		(void) fprintf(stderr,
		    "%s: no space for character device sort pointers\n", Pn);
		Exit(1);
	    }
	    for (i = 0; i < Ndev; i++) {
		Sdev[i] = &Devtp[i];
	    }
	    (void) qsort((QSORT_P *)Sdev, (size_t)Ndev,
		(size_t)sizeof(struct l_dev *), compdev);
	    Ndev = rmdupdev(&Sdev, Ndev, "char");
	} else {
	    (void) fprintf(stderr, "%s: no character devices found\n", Pn);
	    Exit(1);
	}

#if	defined(HASDCACHE)
/*
 * Write device cache file, as required.
 */
	if (DCstate == 1 || (DCstate == 3 && dcrd))
	    write_dcache();
#endif	/* defined(HASDCACHE) */

}


#if	defined(HASDCACHE)
/*
 * rereaddev() - reread device names, modes and types
 */

void
rereaddev()
{
	(void) clr_devtab();

# if	defined(DCACHE_CLR)
	(void) DCACHE_CLR();
# endif	/* defined(DCACHE_CLR) */

	readdev(1);
	DCunsafe = 0;
}


/*
 * rw_clone_sect() - read/write the device cache file clone section
 */

int
rw_clone_sect(m)
	int m;				/* mode: 1 = read; 2 = write */
{
	char buf[MAXPATHLEN*2], *cp, *cp1;
	struct clone *c;
	struct l_dev *dp;
	int i, j, len, n;

	if (m == 1) {

	/*
	 * Read the clone section header and validate it.
	 */
	    if (!fgets(buf, sizeof(buf), DCfs)) {

bad_clone_sect:
		if (!Fwarn) {
		    (void) fprintf(stderr,
			"%s: bad clone section header in %s: ",
			Pn, DCpath[DCpathX]);
		    safestrprt(buf, stderr, 1);
		}
		return(1);
	    }
	    (void) crc(buf, strlen(buf), &DCcksum);
	    len = strlen("clone section: ");
	    if (strncmp(buf, "clone section: ", len) != 0)
		goto bad_clone_sect;
	    if ((n = atoi(&buf[len])) < 0)
		goto bad_clone_sect;
	/*
	 * Read the clone section lines and create the Clone list.
	 */
	    for (i = 0; i < n; i++) {
		if (fgets(buf, sizeof(buf), DCfs) == NULL) {
		    if (!Fwarn) {
			(void) fprintf(stderr,
			    "%s: bad clone line in %s: ", Pn, DCpath[DCpathX]);
			safestrprt(buf, stderr, 1);
		    }
		    return(1);
		}
		(void) crc(buf, strlen(buf), &DCcksum);
	    /*
	     * Assemble Devtp[] index and make sure it's correct.
	     */
		for (cp = buf, j = 0; *cp != ' '; cp++) {
		    if (*cp < '0' || *cp > '9') {

bad_clone_index:
			if (!Fwarn) {
			    (void) fprintf(stderr,
				"%s: bad cached clone device index: ", Pn);
			    safestrprt(buf, stderr, 1);
			}
			return(1);
		    }
		    j = (j * 10) + (int)(*cp - '0');
		}
		if (j < 0 || j >= Ndev || (cp1 = strchr(++cp, '\n')) == NULL)
		    goto bad_clone_index;
		if (strncmp(cp, Devtp[j].name, (cp1 - cp)) != 0)
		    goto bad_clone_index;
	    /*
	     * Allocate and complete a clone structure.
	     */
		if (!(c = (struct clone *)malloc(sizeof(struct clone)))) {
		    (void) fprintf(stderr,
			"%s: no space for cached clone: ", Pn);
		    safestrprt(buf, stderr, 1);
		    Exit(1);
		}
		c->dx = j;
		c->next = Clone;
		Clone = c;
	    }
	    return(0);
	} else if (m == 2) {

	/*
	 * Write the clone section header.
	 */
	    for (c = Clone, n = 0; c; c = c->next, n++)
		;
	    (void) sprintf(buf, "clone section: %d\n", n);
	    if (wr2DCfd(buf, &DCcksum))
		return(1);
	/*
	 * Write the clone section lines.
	 */
	    for (c = Clone; c; c = c->next) {
		for (dp = &Devtp[c->dx], j = 0; j < Ndev; j++) {
		    if (dp == Sdev[j])
			break;
		}
		if (j >= Ndev) {
		    if (!Fwarn) {
			(void) fprintf(stderr,
			    "%s: can't make index for clone: ", Pn);
			safestrprt(dp->name, stderr, 1);
		    }
		    (void) unlink(DCpath[DCpathX]);
		    (void) close(DCfd);
		    DCfd = -1;
		    return(1);
		}
		(void) sprintf(buf, "%d %s\n", j, dp->name);
		if (wr2DCfd(buf, &DCcksum))
		    return(1);
	    }
	    return(0);
	}
/*
 * A shouldn't-happen case: mode neither 1 nor 2.
 */
	(void) fprintf(stderr, "%s: internal rw_clone_sect error: %d\n",
	    Pn, m);
	Exit(1);
}
#endif	/* defined(HASDCACHE) */


/*
 * rmdupdev() - remove duplicate (major/minor/inode) devices
 */

static int
rmdupdev(dp, n, nm)
	struct l_dev ***dp;	/* device table pointers address */
	int n;			/* number of pointers */
	char *nm;		/* device table name for error message */
{
	struct clone *c, *cp;
	int i, j, k;
	struct l_dev **p;

	for (i = j = 0, p = *dp; i < n ;) {
	    for (k = i + 1; k < n; k++) {
		if (p[i]->rdev != p[k]->rdev || p[i]->inode != p[k]->inode)
		    break;
	    /*
	     * See if we're deleting a duplicate clone device.  If so,
	     * delete its clone table entry.
	     */
		for (c = Clone, cp = (struct clone *)NULL;
		     c;
		     cp = c, c = c->next)
		{
		    if (&Devtp[c->dx] != p[k])
			continue;
		    if (!cp)
			Clone = c->next;
		    else
			cp->next = c->next;
		    (void) free((FREE_P *)c);
		    break;
		}
	    }
	    if (i != j)
		p[j] = p[i];
	    j++;
	    i = k;
	}
	if (n == j)
	    return(n);
	if (!(*dp = (struct l_dev **)realloc((MALLOC_P *)*dp,
	    (MALLOC_S)(j * sizeof(struct l_dev *)))))
	{
	    (void) fprintf(stderr, "%s: can't realloc %s device pointers\n",
		Pn, nm);
	    Exit(1);
	}
	return(j);
}


/*
 * stkdir() - stack directory name
 */

static void
stkdir(d, n, x, p)
	char ***d;		/* array of directory pointers */
	int *n;			/* number of pointers */
	int *x;			/* current index */
	char *p;		/* directory path */
{
	MALLOC_S len;
/*
 * Provide adequate directory stack space.
 */
	if (*x >= *n) {
	    *n += 10;
	    len = (MALLOC_S)(*n * sizeof(char *));
	    if (!*d)
		*d = (char **)malloc(len);
	    else
		*d = (char **)realloc((MALLOC_P *)*d, len);
	    if (!*d) {
		(void) fprintf(stderr,
		    "%s: no space for directory stack at ", Pn);
		safestrprt(p, stderr, 1);
		Exit(1);
	    }
	}
/*
 * Allocate space for the name, copy it there and put its pointer on the stack.
 */
	if (!((*d)[*x] = mkstrcpy(p, (MALLOC_S *)NULL))) {
	    (void) fprintf(stderr, "%s: no space for: ", Pn);
	    safestrprt(p, stderr, 1);
	    Exit(1);
	}
	*x += 1;
}


#if	defined(HASDCACHE)
/*
 * vfy_dev() - verify a device table entry (usually when DCunsafe == 1)
 *
 * Note: rereads entire device table when an entry can't be verified.
 */

int
vfy_dev(dp)
	struct l_dev *dp;		/* device table pointer */
{
	struct stat sb;

	if (!DCunsafe || dp->v)
	    return(1);

#if	defined(USE_STAT)
	if (stat(dp->name, &sb)
#else	/* !defined(USE_STAT) */
	if (lstat(dp->name, &sb)
#endif	/* defined(USE_STAT) */

	||  dp->rdev != sb.st_rdev
	||  dp->inode != sb.st_ino) {
	   (void) rereaddev();
	   return(0);
	}
	dp->v = 1;
	return(1);
}
#endif  /* defined(HASDCACHE) */
