;;; xwem-register.el --- Registers support for XWEM.

;; Copyright (C) 2004 by Free Software Foundation, Inc.

;; Author: Zajcev Evgeny <zevlg@yandex.ru>
;; Created: Fri Feb  6 08:04:24 MSK 2004
;; Keywords: xwem
;; X-CVS: $Id: xwem-register.el,v 1.3 2004/05/05 22:43:06 lg Exp $

;; This file is part of XWEM.

;; XWEM is free software; you can redistribute it and/or modify it
;; under the terms of the GNU General Public License as published by
;; the Free Software Foundation; either version 2, or (at your option)
;; any later version.

;; XWEM is distributed in the hope that it will be useful, but WITHOUT
;; ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
;; or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public
;; License for more details.

;; You should have received a copy of the GNU General Public License
;; along with XEmacs; see the file COPYING.  If not, write to the Free
;; Software Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA
;; 02111-1307, USA.

;;; Synched up with: Not in FSF

;;; Commentary:

;; Just like `register' packege for Emacs, but for XWEM.
;;
;; To start using it add:
;;
;;    (require 'xwem-register)
;;    (xwem-register-install-bindings)
;;
;; to your xwemrc.  That will add bonus bindings to `xwem-global-map',
;; such as `H-x 6' to store current window configuration to register,
;; `H-x /' to store current client to register and `H-x j' to jump to
;; register, i.e. set saved window config or pop to saved client, and
;; others.

;;; Code:


(defcustom xwem-registers-frame-config-no-delete t
  "This valued passed as NO-DELETE argument to `xwem-set-frame-configuration'."
  :type 'boolean
  :group 'xwem-misc)

(defvar xwem-registers nil
  "XWEM registers alist.
Each element in form \(NAME . VALUE\), one for each register.
NAME is a character. VALUE is a string, number, client or a list.
A list in form (XWEM-WIN-CONFIG CONFIG) represent a window
configuration.")

(defun xwem-register-set (register value)
  "Set contents of XWEM register named REGISTER to VALUE.
Return VALUE, see documentation for `xwem-registers' for possible VALUE."
  (let ((aelt (assq register xwem-registers)))
    (if aelt
	(setcdr aelt value)
      (setq aelt (cons register value))
      (setq xwem-registers (cons aelt xwem-registers)))
    value))

(defun xwem-register-get (reg)
  "Return contents of XWEM register named REG, or nil if none."
  (cdr (assq reg xwem-registers)))

;;;###autoload(autoload 'xwem-register-client "xwem-register" "" t)
(define-xwem-command xwem-register-client (register)
  "Store selected client to REGISTER."
  (xwem-interactive "kClient to register: ")

  (let ((reg (event-key register)))
    (xwem-register-set reg (xwem-cl-selected))))

;;;###autoload(autoload 'xwem-register-win-config "xwem-register" "" t)
(define-xwem-command xwem-register-win-config (register)
  "Store window configuration in REGISTER."
  (xwem-interactive "kWindow Configuration to register: ")

  (let ((reg (event-key register)))
    (xwem-register-set reg (list 'XWEM-WIN-CONFIG
				 (xwem-window-configuration)))))

;;;###autoload(autoload 'xwem-register-frame-config "xwem-register" "" t)
(define-xwem-command xwem-register-frame-config (register)
  "Store frame configuration to REGISTER."
  (xwem-interactive "kFrame Configuration to register: ")

  (let ((reg (event-key register)))
    (xwem-register-set reg (list 'XWEM-FRAME-CONFIG
                                 (xwem-frame-configuration)))))

;;;###autoload(autoload 'xwem-register-jump "xwem-register" "" t)
(define-xwem-command xwem-register-jump (register)
  "Jump to REGISTER.
If prefix ARG is supplied remove REGISTER from `xwem-registers' alist."
  (xwem-interactive "kJump to register: ")

  (let* ((reg (event-key register))
	 (rval (xwem-register-get reg)))
    (cond ((xwem-cl-p rval)
	   (xwem-cl-pop-to-client rval))

	  ((and (listp rval) (eq 'XWEM-WIN-CONFIG (car rval)))
	   (xwem-set-window-configuration (cadr rval)))

          ((and (listp rval) (eq 'XWEM-FRAME-CONFIG (car rval)))
           (xwem-set-frame-configuration (cadr rval) xwem-registers-frame-config-no-delete))

	  (t (xwem-message 'todo "Hanle register value: %S" rval)))
    
    ))

;;;###autoload
(defun xwem-register-install-bindings ()
  "Install default register bindings in `xwem-global-map'."
  (define-key xwem-global-map (xwem-kbd "H-x 6") 'xwem-register-win-config)
  (define-key xwem-global-map (xwem-kbd "H-x 8") 'xwem-register-frame-config)

  (define-key xwem-global-map (xwem-kbd "H-x /") 'xwem-register-client)
  (define-key xwem-global-map (xwem-kbd "H-x j") 'xwem-register-jump))


(provide 'xwem-register)

;;; xwem-register.el ends here
