/*
 This file is part of GNU Taler
 (C) 2020 Taler Systems S.A.

 GNU Taler is free software; you can redistribute it and/or modify it under the
 terms of the GNU General Public License as published by the Free Software
 Foundation; either version 3, or (at your option) any later version.

 GNU Taler is distributed in the hope that it will be useful, but WITHOUT ANY
 WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
 A PARTICULAR PURPOSE.  See the GNU General Public License for more details.

 You should have received a copy of the GNU General Public License along with
 GNU Taler; see the file COPYING.  If not, see <http://www.gnu.org/licenses/>
 */

/**
 * Imports.
 */
import {
  ConfirmPayResultType,
  PreparePayResultType,
  TalerMerchantInstanceHttpClient,
  URL,
  codecForMerchantOrderStatusUnpaid,
  succeedOrThrow,
} from "@gnu-taler/taler-util";
import { WalletApiOperation } from "@gnu-taler/taler-wallet-core";
import {
  createSimpleTestkudosEnvironmentV3,
  withdrawViaBankV3,
} from "../harness/environments.js";
import { GlobalTestState, harnessHttpLib } from "../harness/harness.js";

/**
 * Run test for basic, bank-integrated withdrawal.
 */
export async function runMerchantLongpollingTest(t: GlobalTestState) {
  // Set up test environment
  const { walletClient, bankClient, exchange, merchant, merchantAdminAccessToken } =
    await createSimpleTestkudosEnvironmentV3(t);

  // Withdraw digital cash into the wallet.

  const wres = await withdrawViaBankV3(t, {
    walletClient,
    bankClient,
    exchange,
    amount: "TESTKUDOS:20",
  });

  await wres.withdrawalFinishedCond;

  const merchantClient = new TalerMerchantInstanceHttpClient(
    merchant.makeInstanceBaseUrl(),
  );

  /*
   * =========================================================================
   * Create an order and let the wallet pay under a session ID
   *
   * We check along the way that the JSON response to /orders/{order_id}
   * returns the right thing.
   * =========================================================================
   */

  let orderResp = succeedOrThrow(
    await merchantClient.createOrder(merchantAdminAccessToken, {
      order: {
        summary: "Buy me!",
        amount: "TESTKUDOS:5",
        fulfillment_url: "https://example.com/article42",
      },
      create_token: false,
    }),
  );

  let orderStatus = succeedOrThrow(
    await merchantClient.getOrderDetails(merchantAdminAccessToken, orderResp.order_id, {
      sessionId: "mysession-one",
    }),
  );

  t.assertTrue(orderStatus.order_status === "unpaid");

  t.assertTrue(orderStatus.already_paid_order_id === undefined);
  let publicOrderStatusUrl = new URL(orderStatus.order_status_url);

  // First, request order status without longpolling
  {
    console.log("requesting", publicOrderStatusUrl.href);
    let publicOrderStatusResp = await harnessHttpLib.fetch(
      publicOrderStatusUrl.href,
    );

    if (publicOrderStatusResp.status != 402) {
      throw Error(
        `expected status 402 (before claiming, no long polling), but got ${publicOrderStatusResp.status}`,
      );
    }
  }

  // Now do long-polling for half a second!
  publicOrderStatusUrl.searchParams.set("timeout_ms", "500");

  console.log("requesting", publicOrderStatusUrl.href);
  let publicOrderStatusResp = await harnessHttpLib.fetch(
    publicOrderStatusUrl.href,
  );

  if (publicOrderStatusResp.status != 402) {
    throw Error(
      `expected status 402 (before claiming, with long-polling), but got ${publicOrderStatusResp.status}`,
    );
  }

  let pubUnpaidStatus = codecForMerchantOrderStatusUnpaid().decode(
    await publicOrderStatusResp.json(),
  );

  console.log(pubUnpaidStatus);

  /**
   * =========================================================================
   * Now actually pay, but WHILE a long poll is active!
   * =========================================================================
   */

  let preparePayResp = await walletClient.call(
    WalletApiOperation.PreparePayForUri,
    {
      talerPayUri: pubUnpaidStatus.taler_pay_uri,
    },
  );

  t.assertTrue(preparePayResp.status === PreparePayResultType.PaymentPossible);

  publicOrderStatusUrl.searchParams.set("timeout_ms", "5000");
  publicOrderStatusUrl.searchParams.set(
    "h_contract",
    preparePayResp.contractTermsHash,
  );

  let publicOrderStatusPromise = harnessHttpLib.fetch(
    publicOrderStatusUrl.href,
  );

  t.assertTrue(preparePayResp.status === PreparePayResultType.PaymentPossible);

  const proposalTransactionId = preparePayResp.transactionId;

  publicOrderStatusResp = await publicOrderStatusPromise;

  if (publicOrderStatusResp.status != 402) {
    throw Error(
      `expected status 402 (after claiming), but got ${publicOrderStatusResp.status}`,
    );
  }

  pubUnpaidStatus = codecForMerchantOrderStatusUnpaid().decode(
    await publicOrderStatusResp.json(),
  );

  const confirmPayRes = await walletClient.call(WalletApiOperation.ConfirmPay, {
    transactionId: proposalTransactionId,
  });

  t.assertTrue(confirmPayRes.type === ConfirmPayResultType.Done);
}

runMerchantLongpollingTest.suites = ["merchant"];
