/*
  This file is part of TALER
  Copyright (C) 2025 Taler Systems SA

  TALER is free software; you can redistribute it and/or modify it under the
  terms of the GNU Affero General Public License as published by the Free Software
  Foundation; either version 3, or (at your option) any later version.

  TALER is distributed in the hope that it will be useful, but WITHOUT ANY
  WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
  A PARTICULAR PURPOSE.  See the GNU Affero General Public License for more details.

  You should have received a copy of the GNU Affero General Public License along with
  TALER; see the file COPYING.  If not, see <http://www.gnu.org/licenses/>
*/
/**
 * @file taler-exchange-httpd_melt_v27.c
 * @brief Handle /melt requests
 * @note This endpoint is active since v27 of the protocol API
 * @author Özgür Kesim
 */

#include "taler/platform.h"
#include <gnunet/gnunet_util_lib.h>
#include <jansson.h>
#include "taler-exchange-httpd.h"
#include "taler/taler_json_lib.h"
#include "taler/taler_mhd_lib.h"
#include "taler-exchange-httpd_melt_v27.h"
#include "taler-exchange-httpd_responses.h"
#include "taler-exchange-httpd_keys.h"
#include "taler/taler_util.h"

/**
 * The different type of errors that might occur, sorted by name.
 * Some of them require idempotency checks, which are marked
 * in @e idempotency_check_required below.
 */
enum MeltError
{
  MELT_ERROR_NONE = 0,
  MELT_ERROR_AGE_RESTRICTION_COMMITMENT_INVALID,
  MELT_ERROR_AGE_RESTRICTION_NOT_SUPPORTED_BY_DENOMINATION,
  MELT_ERROR_AMOUNT_OVERFLOW,
  MELT_ERROR_AMOUNT_PLUS_FEE_OVERFLOW,
  MELT_ERROR_AMOUNT_WITH_FEE_INCORRECT,
  MELT_ERROR_BLINDING_SEED_REQUIRED,
  MELT_ERROR_COIN_CIPHER_MISMATCH,
  MELT_COIN_CONFLICTING_DENOMINATION_KEY,
  MELT_ERROR_COIN_EXPIRED_NO_ZOMBIE,
  MELT_ERROR_COIN_SIGNATURE_INVALID,
  MELT_ERROR_COIN_UNKNOWN,
  MELT_ERROR_CONFIRMATION_SIGN,
  MELT_ERROR_CRYPTO_HELPER,
  MELT_ERROR_DB_FETCH_FAILED,
  MELT_ERROR_DB_INVARIANT_FAILURE,
  MELT_ERROR_DB_MAKE_COIN_KNOW_FAILURE,
  MELT_ERROR_DB_PREFLIGHT_FAILURE,
  MELT_ERROR_DENOMINATION_EXPIRED,
  MELT_ERROR_DENOMINATION_KEY_UNKNOWN,
  MELT_ERROR_DENOMINATION_REVOKED,
  MELT_ERROR_DENOMINATION_SIGN,
  MELT_ERROR_DENOMINATION_SIGNATURE_INVALID,
  MELT_ERROR_DENOMINATION_VALIDITY_IN_FUTURE,
  MELT_ERROR_IDEMPOTENT_PLANCHET,
  MELT_ERROR_INSUFFICIENT_FUNDS,
  MELT_ERROR_KEYS_MISSING,
  MELT_ERROR_FEES_EXCEED_CONTRIBUTION,
  MELT_ERROR_NONCE_RESUSE,
  MELT_ERROR_REQUEST_PARAMETER_MALFORMED,
};

/**
 * With the bits set in this value will be mark the errors
 * that require a check for idempotency before actually
 * returning an error.
 */
static const uint64_t idempotency_check_required =
  0
  | (1 << MELT_ERROR_DENOMINATION_EXPIRED)
  | (1 << MELT_ERROR_DENOMINATION_KEY_UNKNOWN)
  | (1 << MELT_ERROR_DENOMINATION_REVOKED)
  | (1 << MELT_ERROR_INSUFFICIENT_FUNDS) /* TODO: is this still correct? Compare exchange_do_refresh.sql */
  | (1 << MELT_ERROR_KEYS_MISSING);

#define IDEMPOTENCY_CHECK_REQUIRED(error) \
        (0 != (idempotency_check_required & (1 << (error))))

/**
 * Context for a /melt request
 */
struct MeltContext
{

  /**
   * This struct is kept in a DLL.
   */
  struct MeltContext *prev;
  struct MeltContext *next;

  /**
     * Processing phase we are in.
     * The ordering here partially matters, as we progress through
     * them by incrementing the phase in the happy path.
     */
  enum MeltPhase
  {
    MELT_PHASE_PARSE,
    MELT_PHASE_CHECK_MELT_VALID,
    MELT_PHASE_CHECK_KEYS,
    MELT_PHASE_CHECK_COIN_SIGNATURE,
    MELT_PHASE_PREPARE_TRANSACTION,
    MELT_PHASE_RUN_TRANSACTION,
    MELT_PHASE_GENERATE_REPLY_SUCCESS,
    MELT_PHASE_GENERATE_REPLY_ERROR,
    MELT_PHASE_RETURN_NO,
    MELT_PHASE_RETURN_YES,
  } phase;


  /**
   * Request context
   */
  const struct TEH_RequestContext *rc;

  /**
   * Current time for the DB transaction.
   */
  struct GNUNET_TIME_Timestamp now;

  /**
   * The current key state
   */
  struct TEH_KeyStateHandle *ksh;

  /**
   * The melted coin's denomination key
   */
  struct TEH_DenominationKey *melted_coin_denom;

  /**
   * Set to true if this coin's denomination was revoked and the operation
   * is thus only allowed for zombie coins where the transaction
   * history includes a #TALER_EXCHANGEDB_TT_OLD_COIN_RECOUP.
   */
  bool zombie_required;

  /**
   * We already checked and noticed that the coin is known. Hence we
   * can skip the "ensure_coin_known" step of the transaction.
   */
  bool coin_is_dirty;

  /**
   * UUID of the coin in the known_coins table.
   */
  uint64_t known_coin_id;

  /**
   * Captures all parameters provided in the JSON request
   */
  struct
  {

    /**
     * All fields (from the request or computed)
     * that we persist in the database.
     */
    struct TALER_EXCHANGEDB_Refresh_v27 refresh;

    /**
     * In some error cases we check for idempotency.
     * If we find an entry in the database, we mark this here.
     */
    bool is_idempotent;

    /**
     * In some error conditions the request is checked
     * for idempotency and the result from the database
     * is stored here.
     */
    struct TALER_EXCHANGEDB_Refresh_v27 refresh_idem;

    /**
     * True if @e blinding_seed is missing in the request
     */
    bool no_blinding_seed;

    /**
     * Array @e persis.num_coins of hashes of the public keys
     * of the denominations to refresh.
     */
    struct TALER_DenominationHashP *denoms_h;

    /**
     * Array of @e num_planchets coin planchets, arranged
     * in runs of @e num_coins coins, [0..num_coins)..[0..num_coins),
     * one for each kappa value.
     */
    struct TALER_BlindedPlanchet *planchets[TALER_CNC_KAPPA];

    /**
     * #TALER_CNC_KAPPA hashes of the batches of @e num_coins coins.
     */
    struct TALER_KappaHashBlindedPlanchetsP kappa_planchets_h;

    /**
     * Array @e withdraw.num_r_pubs of indices into @e denoms_h
     * of CS denominations.
     */
    uint32_t *cs_indices;

    /**
     * Total (over all coins) amount (excluding fee) committed for the refresh
     */
    struct TALER_Amount amount;

  } request;

  /**
   * Errors occurring during evaluation of the request are captured in this
   * struct.  In phase WITHDRAW_PHASE_GENERATE_REPLY_ERROR an appropriate error
   * message is prepared and sent to the client.
   */
  struct
  {
    /* The (internal) error code */
    enum MeltError code;

    /**
     * Some errors require details to be sent to the client.
     * These are captured in this union.
     * Each field is named according to the error that is using it, except
     * commented otherwise.
     */
    union
    {
      const char *request_parameter_malformed;

      /**
       * For all errors related to a particular denomination, i.e.
       *  #MELT_ERROR_DENOMINATION_KEY_UNKNOWN,
       *  #MELT_ERROR_DENOMINATION_EXPIRED,
       *  #MELT_ERROR_DENOMINATION_VALIDITY_IN_FUTURE,
       *  #MELT_ERROR_AGE_RESTRICTION_NOT_SUPPORTED_BY_DENOMINATION,
       * we use this one field.
       */
      struct TALER_DenominationHashP denom_h;

      const char *db_fetch_context;

      enum TALER_ErrorCode ec_confirmation_sign;

      enum TALER_ErrorCode ec_denomination_sign;

      /* remaining value of the coin */
      struct TALER_Amount insufficient_funds;

    } details;
  } error;
};

/**
 * The following macros set the given error code,
 * set the phase to Melt_PHASE_GENERATE_REPLY_ERROR,
 * and optionally set the given field (with an optionally given value).
 */
#define SET_ERROR(mc, ec) \
        do \
        { GNUNET_static_assert (MELT_ERROR_NONE != ec); \
          (mc)->error.code = (ec);                      \
          (mc)->phase = MELT_PHASE_GENERATE_REPLY_ERROR; } while (0)

#define SET_ERROR_WITH_FIELD(mc, ec, field) \
        do \
        { GNUNET_static_assert (MELT_ERROR_NONE != ec); \
          (mc)->error.code = (ec);                      \
          (mc)->error.details.field = (field);          \
          (mc)->phase = MELT_PHASE_GENERATE_REPLY_ERROR; } while (0)

#define SET_ERROR_WITH_DETAIL(mc, ec, field, value) \
        do \
        { GNUNET_static_assert (MELT_ERROR_NONE != ec); \
          (mc)->error.code = (ec);                      \
          (mc)->error.details.field = (value);          \
          (mc)->phase = MELT_PHASE_GENERATE_REPLY_ERROR; } while (0)


/**
 * All melt context is kept in a DLL.
 */
static struct MeltContext *mc_head;
static struct MeltContext *mc_tail;

void
TEH_melt_v27_cleanup ()
{
  struct MeltContext *mc;

  while (NULL != (mc = mc_head))
  {
    GNUNET_CONTAINER_DLL_remove (mc_head,
                                 mc_tail,
                                 mc);
    MHD_resume_connection (mc->rc->connection);
  }
}


/**
 * Terminate the main loop by returning the final result.
 *
 * @param[in,out] mc context to update phase for
 * @param mres MHD status to return
 */
static void
finish_loop (struct MeltContext *mc,
             MHD_RESULT mres)
{
  mc->phase = (MHD_YES == mres)
                         ? MELT_PHASE_RETURN_YES
                         : MELT_PHASE_RETURN_NO;
}


/**
 * Free information in @a re, but not @a re itself.
 *
 * @param[in] re refresh data to free
 */
static void
free_refresh (struct TALER_EXCHANGEDB_Refresh_v27 *re)
{
  if (NULL != re->denom_sigs)
  {
    for (size_t i = 0; i<re->num_coins; i++)
      TALER_blinded_denom_sig_free (&re->denom_sigs[i]);
    GNUNET_free (re->denom_sigs);
  }
  GNUNET_free (re->cs_r_values);
  GNUNET_free (re->denom_serials);
  GNUNET_free (re->denom_pub_hashes);
  TALER_denom_sig_free (&re->coin.denom_sig);
}


/**
 * Cleanup routine for melt request.
 * The function is called upon completion of the request
 * that should clean up @a rh_ctx.
 *
 * @param rc request context to clean up
 */
static void
clean_melt_rc (struct TEH_RequestContext *rc)
{
  struct MeltContext *mc = rc->rh_ctx;

  GNUNET_free (mc->request.denoms_h);
  for (uint8_t k = 0; k<TALER_CNC_KAPPA; k++)
  {
    for (size_t i = 0; i<mc->request.refresh.num_coins; i++)
      TALER_blinded_planchet_free (&mc->request.planchets[k][i]);
    GNUNET_free (mc->request.planchets[k]);
  }
  free_refresh (&mc->request.refresh);
  if (mc->request.is_idempotent)
  {
    free_refresh (&mc->request.refresh_idem);
  }
  GNUNET_free (mc->request.cs_indices);
  GNUNET_free (mc);
}


/**
 * Creates a new context for the incoming melt request
 *
 * @param mc melt request context
 * @param root json body of the request
 */
static void
phase_parse_request (
  struct MeltContext *mc,
  const json_t *root)
{
  const json_t *j_denoms_h;
  const json_t *j_coin_evs;
  enum GNUNET_GenericReturnValue res;
  struct GNUNET_JSON_Specification spec[] = {
    GNUNET_JSON_spec_fixed_auto ("old_coin_pub",
                                 &mc->request.refresh.coin.coin_pub),
    GNUNET_JSON_spec_fixed_auto ("old_denom_pub_h",
                                 &mc->request.refresh.coin.denom_pub_hash),
    GNUNET_JSON_spec_mark_optional (
      GNUNET_JSON_spec_fixed_auto ("old_age_commitment_h",
                                   &mc->request.refresh.coin.h_age_commitment),
      &mc->request.refresh.coin.no_age_commitment),
    TALER_JSON_spec_denom_sig ("old_denom_sig",
                               &mc->request.refresh.coin.denom_sig),
    GNUNET_JSON_spec_fixed_auto ("refresh_seed",
                                 &mc->request.refresh.refresh_seed),
    GNUNET_JSON_spec_mark_optional (
      GNUNET_JSON_spec_fixed_auto ("blinding_seed",
                                   &mc->request.refresh.blinding_seed),
      &mc->request.refresh.no_blinding_seed),
    TALER_JSON_spec_amount ("value_with_fee",
                            TEH_currency,
                            &mc->request.refresh.amount_with_fee),
    GNUNET_JSON_spec_array_const ("denoms_h",
                                  &j_denoms_h),
    GNUNET_JSON_spec_array_const ("coin_evs",
                                  &j_coin_evs),
    GNUNET_JSON_spec_fixed_auto ("confirm_sig",
                                 &mc->request.refresh.coin_sig),
    GNUNET_JSON_spec_end ()
  };

  res = TALER_MHD_parse_json_data (mc->rc->connection,
                                   root,
                                   spec);
  if (GNUNET_OK != res)
  {
    GNUNET_break_op (0);
    mc->phase = (GNUNET_NO == res)
      ? MELT_PHASE_RETURN_YES
      : MELT_PHASE_RETURN_NO;
    return;
  }

  /* validate array size */
  GNUNET_static_assert (
    TALER_MAX_COINS < INT_MAX / TALER_CNC_KAPPA);

  mc->request.refresh.num_coins = json_array_size (j_denoms_h);
  if (0 == mc->request.refresh.num_coins)
  {
    GNUNET_break_op (0);
    SET_ERROR_WITH_DETAIL (mc,
                           MELT_ERROR_REQUEST_PARAMETER_MALFORMED,
                           request_parameter_malformed,
                           "denoms_h must not be empty");
    return;
  }
  else if (TALER_MAX_COINS < mc->request.refresh.num_coins)
  {
    /**
     * The wallet had committed to more than the maximum coins allowed, the
     * reserve has been charged, but now the user can not melt any money
     * from it.  Note that the user can't get their money back in this case!
     */
    GNUNET_break_op (0);
    SET_ERROR_WITH_DETAIL (mc,
                           MELT_ERROR_REQUEST_PARAMETER_MALFORMED,
                           request_parameter_malformed,
                           "maximum number of coins that can be refreshed has been exceeded");
    return;
  }
  else if (TALER_CNC_KAPPA != json_array_size (j_coin_evs))
  {
    GNUNET_break_op (0);
    SET_ERROR_WITH_DETAIL (mc,
                           MELT_ERROR_REQUEST_PARAMETER_MALFORMED,
                           request_parameter_malformed,
                           "coin_evs must be an array of length "TALER_CNC_KAPPA_STR);
    return;
  }

  /* Extract the denomination hashes */
  {
    size_t idx;
    json_t *value;

    mc->request.denoms_h
      = GNUNET_new_array (mc->request.refresh.num_coins,
                          struct TALER_DenominationHashP);

    json_array_foreach (j_denoms_h, idx, value) {
      struct GNUNET_JSON_Specification ispec[] = {
        GNUNET_JSON_spec_fixed_auto (NULL,
                                     &mc->request.denoms_h[idx]),
        GNUNET_JSON_spec_end ()
      };

      res = TALER_MHD_parse_json_data (mc->rc->connection,
                                       value,
                                       ispec);
      if (GNUNET_YES != res)
      {
        GNUNET_break_op (0);
        mc->phase = (GNUNET_NO == res)
          ? MELT_PHASE_RETURN_YES
          : MELT_PHASE_RETURN_NO;
        return;
      }
    }
  }

  /* Calculate the hash over the blinded coin envelopes */
  for (size_t k = 0; k<TALER_CNC_KAPPA; k++)
  {
    mc->request.planchets[k] =
      GNUNET_new_array (mc->request.refresh.num_coins,
                        struct  TALER_BlindedPlanchet);
  }

  /* Parse blinded envelopes. */
  {
    json_t *j_kappa_planchets;
    size_t kappa;
    struct GNUNET_HashContext *ctx;

    /* ctx to calculate the planchet_h */
    ctx = GNUNET_CRYPTO_hash_context_start ();
    GNUNET_assert (NULL != ctx);

    json_array_foreach (j_coin_evs, kappa, j_kappa_planchets)
    {
      json_t *j_cev;
      size_t idx;

      if (mc->request.refresh.num_coins != json_array_size (j_kappa_planchets))
      {
        GNUNET_break_op (0);
        SET_ERROR_WITH_DETAIL (mc,
                               MELT_ERROR_REQUEST_PARAMETER_MALFORMED,
                               request_parameter_malformed,
                               "coin_evs[] size");
        return;
      }

      json_array_foreach (j_kappa_planchets, idx, j_cev)
      {
        /* Now parse the individual envelopes and calculate the hash of
         * the commitment along the way. */
        struct GNUNET_JSON_Specification kspec[] = {
          TALER_JSON_spec_blinded_planchet (NULL,
                                            &mc->request.planchets[kappa][idx]),
          GNUNET_JSON_spec_end ()
        };

        res = TALER_MHD_parse_json_data (mc->rc->connection,
                                         j_cev,
                                         kspec);
        if (GNUNET_OK != res)
        {
          GNUNET_break_op (0);
          mc->phase = (GNUNET_NO == res)
            ? MELT_PHASE_RETURN_YES
            : MELT_PHASE_RETURN_NO;
          return;
        }
        /* Check for duplicate planchets. Technically a bug on
         * the client side that is harmless for us, but still
         * not allowed per protocol */
        for (size_t k = 0; k <= kappa; k++)
        {
          size_t max = (k == kappa)
                          ? idx
                          : mc->request.refresh.num_coins;
          for (size_t i = 0; i < max; i++)
          {
            if (0 ==
                TALER_blinded_planchet_cmp (
                  &mc->request.planchets[kappa][idx],
                  &mc->request.planchets[k][i]))
            {
              GNUNET_break_op (0);
              GNUNET_JSON_parse_free (kspec);
              SET_ERROR (mc,
                         MELT_ERROR_IDEMPOTENT_PLANCHET);
              return;
            }
          }
        }
      }
      /* Save the hash of the batch of planchets for index kappa */
      TALER_wallet_blinded_planchets_hash (
        mc->request.refresh.num_coins,
        mc->request.planchets[kappa],
        mc->request.denoms_h,
        &mc->request.kappa_planchets_h.tuple[kappa]);
      GNUNET_CRYPTO_hash_context_read (
        ctx,
        &mc->request.kappa_planchets_h.tuple[kappa],
        sizeof(mc->request.kappa_planchets_h.tuple[kappa]));
    }

    /* Finally calculate the total hash over all planchets */
    GNUNET_CRYPTO_hash_context_finish (
      ctx,
      &mc->request.refresh.planchets_h.hash);
  }
  mc->ksh = TEH_keys_get_state ();
  if (NULL == mc->ksh)
  {
    GNUNET_break (0);
    SET_ERROR (mc,
               MELT_ERROR_KEYS_MISSING);
    return;
  }
  mc->phase = MELT_PHASE_CHECK_MELT_VALID;
}


/**
 * Check if the given denomination is still or already valid, has not been
 * revoked and potentically supports age restriction.
 *
 * @param[in,out] mc context for the melt operation
 * @param denom_h Hash of the denomination key to check
 * @param[out] pdk denomination key found, might be NULL
 * @return #GNUNET_OK when denomation was found and valid,
 *         #GNUNET_NO when denomination is not valid at this time
 *         #GNUNET_SYSERR otherwise (denomination invalid), with finish_loop called.
 */
static enum GNUNET_GenericReturnValue
find_denomination (
  struct MeltContext *mc,
  const struct TALER_DenominationHashP *denom_h,
  struct TEH_DenominationKey **pdk)
{
  struct TEH_DenominationKey *dk;

  *pdk = NULL;
  GNUNET_assert (NULL != mc->ksh);
  dk = TEH_keys_denomination_by_hash_from_state (
    mc->ksh,
    denom_h,
    NULL,
    NULL);
  if (NULL == dk)
  {
    SET_ERROR_WITH_DETAIL (mc,
                           MELT_ERROR_DENOMINATION_KEY_UNKNOWN,
                           denom_h,
                           *denom_h);
    return GNUNET_SYSERR;
  }
  *pdk = dk;

  if (GNUNET_TIME_absolute_is_past (
        dk->meta.expire_withdraw.abs_time))
  {
    SET_ERROR_WITH_DETAIL (mc,
                           MELT_ERROR_DENOMINATION_EXPIRED,
                           denom_h,
                           *denom_h);
    /**
     * Note that we return GNUNET_NO here.
     * This way phase_check_melt_valid can react
     * to it as a non-error case and do the zombie check.
     */
    return GNUNET_NO;
  }

  if (GNUNET_TIME_absolute_is_future (
        dk->meta.start.abs_time))
  {
    GNUNET_break_op (0);
    SET_ERROR_WITH_DETAIL (mc,
                           MELT_ERROR_DENOMINATION_VALIDITY_IN_FUTURE,
                           denom_h,
                           *denom_h);
    return GNUNET_SYSERR;
  }

  if (dk->recoup_possible)
  {
    SET_ERROR (mc,
               MELT_ERROR_DENOMINATION_REVOKED);
    return GNUNET_SYSERR;
  }

  /* In case of age melt, make sure that the denomitation supports age restriction */
  if (! (mc->request.refresh.coin.no_age_commitment) &&
      (0 == dk->denom_pub.age_mask.bits))
  {
    GNUNET_break_op (0);
    SET_ERROR_WITH_DETAIL (mc,
                           MELT_ERROR_AGE_RESTRICTION_NOT_SUPPORTED_BY_DENOMINATION,
                           denom_h,
                           *denom_h);
    return GNUNET_SYSERR;
  }
  if ((mc->request.refresh.coin.no_age_commitment) &&
      (0 != dk->denom_pub.age_mask.bits))
  {
    GNUNET_break_op (0);
    SET_ERROR (mc,
               MELT_ERROR_AGE_RESTRICTION_COMMITMENT_INVALID);
    return GNUNET_SYSERR;
  }

  return GNUNET_OK;
}


/**
 * Check if the given array of hashes of denomination_keys
 * - belong to valid denominations
 * - calculate the total amount of the denominations including fees
 * for melt.
 *
 * @param mc context of the melt to check keys for
 */
static void
phase_check_keys (
  struct MeltContext *mc)
{
  bool is_cs_denom[mc->request.refresh.num_coins];

  memset (is_cs_denom,
          0,
          sizeof(is_cs_denom));

  mc->request.refresh.denom_serials =
    GNUNET_new_array (mc->request.refresh.num_coins,
                      uint64_t);
  GNUNET_assert (GNUNET_OK ==
                 TALER_amount_set_zero (TEH_currency,
                                        &mc->request.amount));

  /* Calculate the total value and withdraw fees for the fresh coins */
  for (size_t i = 0; i < mc->request.refresh.num_coins; i++)
  {
    struct TEH_DenominationKey *dk;

    if (GNUNET_OK !=
        find_denomination (
          mc,
          &mc->request.denoms_h[i],
          &dk))
      return;

    if (GNUNET_CRYPTO_BSA_CS ==
        dk->denom_pub.bsign_pub_key->cipher)
    {
      if (mc->request.refresh.no_blinding_seed)
      {
        GNUNET_break_op (0);
        SET_ERROR (mc,
                   MELT_ERROR_BLINDING_SEED_REQUIRED);
        return;
      }
      mc->request.refresh.num_cs_r_values++;
      is_cs_denom[i] = true;
    }
    /* Ensure the ciphers from the planchets match the denominations'. */
    {
      for (uint8_t k = 0; k < TALER_CNC_KAPPA; k++)
      {
        if (dk->denom_pub.bsign_pub_key->cipher !=
            mc->request.planchets[k][i].blinded_message->cipher)
        {
          GNUNET_break_op (0);
          SET_ERROR (mc,
                     MELT_ERROR_COIN_CIPHER_MISMATCH);
          return;
        }
      }
    }
    /* Accumulate the values */
    if (0 > TALER_amount_add (&mc->request.amount,
                              &mc->request.amount,
                              &dk->meta.value))
    {
      GNUNET_break_op (0);
      SET_ERROR (mc,
                 MELT_ERROR_AMOUNT_OVERFLOW);
      return;
    }
    /* Accumulate the withdraw fees for the fresh coins */
    if (0 > TALER_amount_add (&mc->request.amount,
                              &mc->request.amount,
                              &dk->meta.fees.withdraw))
    {
      GNUNET_break_op (0);
      SET_ERROR (mc,
                 MELT_ERROR_AMOUNT_PLUS_FEE_OVERFLOW);
      return;
    }
    mc->request.refresh.denom_serials[i] = dk->meta.serial;
  }

  /**
   * Calculate the amount (with withdraw fee) plus refresh fee and
   * compare with the value provided by the client in the request.
   */
  {
    struct TALER_Amount amount_with_fee;

    if (0 > TALER_amount_add (&amount_with_fee,
                              &mc->request.amount,
                              &mc->melted_coin_denom->meta.fees.refresh))
    {
      GNUNET_break_op (0);
      SET_ERROR (mc,
                 MELT_ERROR_AMOUNT_PLUS_FEE_OVERFLOW);
      return;
    }

    if (0 != TALER_amount_cmp (&amount_with_fee,
                               &mc->request.refresh.amount_with_fee))
    {
      GNUNET_break_op (0);
      SET_ERROR (mc,
                 MELT_ERROR_AMOUNT_WITH_FEE_INCORRECT);
      return;
    }
  }

  /* Save the indices of CS denominations */
  if (0 < mc->request.refresh.num_cs_r_values)
  {
    size_t j = 0;

    mc->request.cs_indices = GNUNET_new_array (
      mc->request.refresh.num_cs_r_values,
      uint32_t);

    for (size_t i = 0; i < mc->request.refresh.num_coins; i++)
    {
      if (is_cs_denom[i])
        mc->request.cs_indices[j++] = i;
    }
  }
  mc->phase++;
}


/**
 * Check that the client signature authorizing the melt is valid.
 *
 * @param[in,out] mc request context to check
 */
static void
phase_check_coin_signature (
  struct MeltContext *mc)
{
  /* We can now compute the commitment */
  {
    struct TALER_KappaHashBlindedPlanchetsP k_bps_h = {0};

    for (uint8_t k = 0; k < TALER_CNC_KAPPA; k++)
      TALER_wallet_blinded_planchets_hash (
        mc->request.refresh.num_coins,
        mc->request.planchets[k],
        mc->request.denoms_h,
        &k_bps_h.tuple[k]);

    TALER_refresh_get_commitment_v27 (
      &mc->request.refresh.rc,
      &mc->request.refresh.refresh_seed,
      mc->request.no_blinding_seed
           ? NULL
           : &mc->request.refresh.blinding_seed,
      &k_bps_h,
      &mc->request.refresh.coin.coin_pub,
      &mc->request.refresh.amount_with_fee);
  }


  TEH_METRICS_num_verifications[TEH_MT_SIGNATURE_EDDSA]++;
  if (GNUNET_OK !=
      TALER_wallet_melt_verify (
        &mc->request.refresh.amount_with_fee,
        &mc->melted_coin_denom->meta.fees.refresh,
        &mc->request.refresh.rc,
        &mc->request.refresh.coin.denom_pub_hash,
        &mc->request.refresh.coin.h_age_commitment,
        &mc->request.refresh.coin.coin_pub,
        &mc->request.refresh.coin_sig))
  {
    GNUNET_break_op (0);
    SET_ERROR (mc,
               MELT_ERROR_COIN_SIGNATURE_INVALID);
    return;
  }

  mc->phase++;
}


/**
 * Check for information about the melted coin's denomination,
 * extracting its validity status and fee structure.
 * Baseline: check if deposits/refreshes are generally
 * simply still allowed for this denomination.
 *
 * @param mc parsed request information
 */
static void
phase_check_melt_valid (struct MeltContext *mc)
{
  enum MeltPhase current_phase = mc->phase;
  /**
   * Find the old coin's denomination.
   * Note that we return only on GNUNET_SYSERR,
   * because GNUNET_NO for the expired denomination
   * will be handled below, with the zombie-check.
   */
  if (GNUNET_SYSERR ==
      find_denomination (mc,
                         &mc->request.refresh.coin.denom_pub_hash,
                         &mc->melted_coin_denom))
    return;

  GNUNET_log (GNUNET_ERROR_TYPE_INFO,
              "Melted coin's denomination is worth %s\n",
              TALER_amount2s (&mc->melted_coin_denom->meta.value));

  /* sanity-check that "total melt amount > melt fee" */
  if (0 <
      TALER_amount_cmp (&mc->melted_coin_denom->meta.fees.refresh,
                        &mc->request.refresh.amount_with_fee))
  {
    GNUNET_break_op (0);
    SET_ERROR (mc,
               MELT_ERROR_FEES_EXCEED_CONTRIBUTION);
    return;
  }

  if (GNUNET_OK !=
      TALER_test_coin_valid (&mc->request.refresh.coin,
                             &mc->melted_coin_denom->denom_pub))
  {
    GNUNET_break_op (0);
    SET_ERROR (mc,
               MELT_ERROR_DENOMINATION_SIGNATURE_INVALID);
    return;
  }

  /**
   * find_denomination might have set the phase to
   * produce an error, but we are still investigating.
   * We reset the phase.
   */
  mc->phase = current_phase;
  mc->error.code = MELT_ERROR_NONE;

  if (GNUNET_TIME_absolute_is_past (
        mc->melted_coin_denom->meta.expire_deposit.abs_time))
  {
    /**
     * We are past deposit expiration time, but maybe this is a zombie?
     */
    struct TALER_DenominationHashP denom_hash;
    enum GNUNET_DB_QueryStatus qs;

    /* Check that the coin is dirty (we have seen it before), as we will
       not just allow melting of a *fresh* coin where the denomination was
       revoked (those must be recouped) */
    qs = TEH_plugin->get_coin_denomination (
      TEH_plugin->cls,
      &mc->request.refresh.coin.coin_pub,
      &mc->known_coin_id,
      &denom_hash);
    if (0 > qs)
    {
      /* There is no good reason for a serialization failure here: */
      GNUNET_break (GNUNET_DB_STATUS_SOFT_ERROR != qs);
      SET_ERROR (mc,
                 MELT_ERROR_DB_FETCH_FAILED);
      return;
    }
    if (GNUNET_DB_STATUS_SUCCESS_ONE_RESULT != qs)
    {
      /* We never saw this coin before, so _this_ justification is not OK.
       * Note that the error was already set in find_denominations. */
      GNUNET_assert (MELT_ERROR_DENOMINATION_EXPIRED ==
                     mc->error.code);
      GNUNET_assert (MELT_PHASE_GENERATE_REPLY_ERROR ==
                     mc->phase);
      return;
    }
    /* sanity check */
    if (0 !=
        GNUNET_memcmp (&denom_hash,
                       &mc->request.refresh.coin.denom_pub_hash))
    {
      GNUNET_break_op (0);
      SET_ERROR_WITH_DETAIL (mc,
                             MELT_COIN_CONFLICTING_DENOMINATION_KEY,
                             denom_h,
                             denom_hash);
      return;
    }
    /* Minor optimization: no need to run the
       "ensure_coin_known" part of the transaction */
    mc->coin_is_dirty = true;
    /* check later that zombie is satisfied */
    mc->zombie_required = true;
  }
  mc->phase++;
}


/**
 * The request for melt was parsed successfully.
 * Sign and persist the chosen blinded coins for the reveal step.
 *
 * @param mc The context for the current melt request
 */
static void
phase_prepare_transaction (
  struct MeltContext *mc)
{
  mc->request.refresh.denom_sigs
    = GNUNET_new_array (
        mc->request.refresh.num_coins,
        struct TALER_BlindedDenominationSignature);
  mc->request.refresh.noreveal_index =
    GNUNET_CRYPTO_random_u32 (GNUNET_CRYPTO_QUALITY_STRONG,
                              TALER_CNC_KAPPA);

  /* Choose and sign the coins */
  {
    struct TEH_CoinSignData csds[mc->request.refresh.num_coins];
    enum TALER_ErrorCode ec_denomination_sign;
    size_t noreveal_idx = mc->request.refresh.noreveal_index;

    memset (csds,
            0,
            sizeof(csds));

    /* Pick the chosen blinded coins */
    for (size_t i = 0; i<mc->request.refresh.num_coins; i++)
    {
      csds[i].bp = &mc->request.planchets[noreveal_idx][i];
      csds[i].h_denom_pub = &mc->request.denoms_h[i];
    }

    ec_denomination_sign = TEH_keys_denomination_batch_sign (
      mc->request.refresh.num_coins,
      csds,
      true, /* for melt */
      mc->request.refresh.denom_sigs);
    if (TALER_EC_NONE != ec_denomination_sign)
    {
      GNUNET_break (0);
      SET_ERROR_WITH_FIELD (mc,
                            MELT_ERROR_DENOMINATION_SIGN,
                            ec_denomination_sign);
      return;
    }

    /* Save the hash of chosen planchets */
    mc->request.refresh.selected_h =
      mc->request.kappa_planchets_h.tuple[noreveal_idx];

    /**
     * For the denominations with cipher CS, calculate the R-values
     * and save the choices we made now, as at a later point, the
     * private keys for the denominations might now be available anymore
     * to make the same choice again.
     */
    if (0 <  mc->request.refresh.num_cs_r_values)
    {
      size_t num_cs_r_values = mc->request.refresh.num_cs_r_values;
      struct TEH_CsDeriveData cdds[num_cs_r_values];
      struct GNUNET_CRYPTO_CsSessionNonce nonces[num_cs_r_values];

      memset (nonces, 0, sizeof(nonces));
      mc->request.refresh.cs_r_values
        = GNUNET_new_array (
            num_cs_r_values,
            struct GNUNET_CRYPTO_CSPublicRPairP);
      mc->request.refresh.cs_r_choices = 0;

      GNUNET_assert (! mc->request.refresh.no_blinding_seed);
      TALER_cs_derive_nonces_from_seed (
        &mc->request.refresh.blinding_seed,
        true, /* for melt */
        num_cs_r_values,
        mc->request.cs_indices,
        nonces);

      for (size_t i = 0; i < num_cs_r_values; i++)
      {
        size_t idx = mc->request.cs_indices[i];

        GNUNET_assert (idx < mc->request.refresh.num_coins);
        cdds[i].h_denom_pub = &mc->request.denoms_h[idx];
        cdds[i].nonce = &nonces[i];
      }

      /**
       * Let the crypto helper generate the R-values and
       * make the choices
       */
      if (TALER_EC_NONE !=
          TEH_keys_denomination_cs_batch_r_pub_simple (
            mc->request.refresh.num_cs_r_values,
            cdds,
            true, /* for melt */
            mc->request.refresh.cs_r_values))
      {
        GNUNET_break (0);
        SET_ERROR (mc,
                   MELT_ERROR_CRYPTO_HELPER);
        return;
      }

      /* Now save the choices for the selected bits */
      for (size_t i = 0; i < num_cs_r_values; i++)
      {
        size_t idx = mc->request.cs_indices[i];

        struct TALER_BlindedDenominationSignature *sig =
          &mc->request.refresh.denom_sigs[idx];
        uint8_t bit = sig->blinded_sig->details.blinded_cs_answer.b;

        mc->request.refresh.cs_r_choices |= bit << i;
        GNUNET_static_assert (
          TALER_MAX_COINS <=
          sizeof(mc->request.refresh.cs_r_choices) * 8);
      }
    }
  }
  mc->phase++;
}


/**
 * Generates response for the melt request.
 *
 * @param mc melt operation context
 */
static void
phase_generate_reply_success (struct MeltContext *mc)
{
  struct TALER_EXCHANGEDB_Refresh_v27 *db_obj;
  struct TALER_ExchangePublicKeyP pub;
  struct TALER_ExchangeSignatureP sig;
  enum TALER_ErrorCode ec_confirmation_sign;

  db_obj = mc->request.is_idempotent
    ? &mc->request.refresh_idem
    : &mc->request.refresh;
  ec_confirmation_sign =
    TALER_exchange_online_melt_confirmation_sign (
      &TEH_keys_exchange_sign_,
      &db_obj->rc,
      db_obj->noreveal_index,
      &pub,
      &sig);
  if (TALER_EC_NONE != ec_confirmation_sign)
  {
    SET_ERROR_WITH_FIELD (mc,
                          MELT_ERROR_CONFIRMATION_SIGN,
                          ec_confirmation_sign);
    return;
  }

  finish_loop (mc,
               TALER_MHD_REPLY_JSON_PACK (
                 mc->rc->connection,
                 MHD_HTTP_OK,
                 GNUNET_JSON_pack_uint64 ("noreveal_index",
                                          db_obj->noreveal_index),
                 GNUNET_JSON_pack_data_auto ("exchange_sig",
                                             &sig),
                 GNUNET_JSON_pack_data_auto ("exchange_pub",
                                             &pub)));
}


/**
 * Check if the melt request is replayed and we already have an answer.
 * If so, replay the existing answer and return the HTTP response.
 *
 * @param[in,out] mc parsed request data
 * @return true if the request is idempotent with an existing request
 *    false if we did not find the request in the DB and did not set @a mret
 */
static bool
melt_is_idempotent (
  struct MeltContext *mc)
{
  enum GNUNET_DB_QueryStatus qs;

  qs = TEH_plugin->get_refresh (
    TEH_plugin->cls,
    &mc->request.refresh.rc,
    &mc->request.refresh_idem);
  if (0 > qs)
  {
    /* FIXME: soft error not handled correctly! */
    GNUNET_break (GNUNET_DB_STATUS_SOFT_ERROR == qs);
    if (GNUNET_DB_STATUS_HARD_ERROR == qs)
      SET_ERROR_WITH_DETAIL (mc,
                             MELT_ERROR_DB_FETCH_FAILED,
                             db_fetch_context,
                             "get_refresh");
    return true; /* Well, kind-of. */
  }
  if (GNUNET_DB_STATUS_SUCCESS_NO_RESULTS == qs)
    return false;

  mc->request.is_idempotent = true;
  GNUNET_log (GNUNET_ERROR_TYPE_DEBUG,
              "request is idempotent\n");

  /* Generate idempotent reply */
  TEH_METRICS_num_requests[TEH_MT_REQUEST_IDEMPOTENT_MELT]++;
  mc->phase = MELT_PHASE_GENERATE_REPLY_SUCCESS;
  mc->error.code = MELT_ERROR_NONE;
  return true;
}


/**
 * Reports an error, potentially with details.
 * That is, it puts a error-type specific response into the MHD queue.
 * It will do a idempotency check first, if needed for the error type.
 *
 * @param mc melt context
 */
static void
phase_generate_reply_error (
  struct MeltContext *mc)
{
  GNUNET_assert (MELT_PHASE_GENERATE_REPLY_ERROR == mc->phase);
  GNUNET_assert (MELT_ERROR_NONE != mc->error.code);

  if (IDEMPOTENCY_CHECK_REQUIRED (mc->error.code) &&
      melt_is_idempotent (mc))
  {
    return;
  }

  switch (mc->error.code)
  {
  case MELT_ERROR_NONE:
    break;
  case MELT_ERROR_REQUEST_PARAMETER_MALFORMED:
    finish_loop (mc,
                 TALER_MHD_reply_with_error (
                   mc->rc->connection,
                   MHD_HTTP_BAD_REQUEST,
                   TALER_EC_GENERIC_PARAMETER_MALFORMED,
                   mc->error.details.request_parameter_malformed));
    return;
  case MELT_ERROR_KEYS_MISSING:
    finish_loop (mc,
                 TALER_MHD_reply_with_error (
                   mc->rc->connection,
                   MHD_HTTP_INTERNAL_SERVER_ERROR,
                   TALER_EC_EXCHANGE_GENERIC_KEYS_MISSING,
                   NULL));
    return;
  case MELT_ERROR_DB_FETCH_FAILED:
    finish_loop (mc,
                 TALER_MHD_reply_with_error (
                   mc->rc->connection,
                   MHD_HTTP_INTERNAL_SERVER_ERROR,
                   TALER_EC_GENERIC_DB_FETCH_FAILED,
                   mc->error.details.db_fetch_context));
    return;
  case MELT_ERROR_DB_INVARIANT_FAILURE:
    finish_loop (mc,
                 TALER_MHD_reply_with_error (
                   mc->rc->connection,
                   MHD_HTTP_INTERNAL_SERVER_ERROR,
                   TALER_EC_GENERIC_DB_INVARIANT_FAILURE,
                   NULL));
    return;
  case MELT_ERROR_DB_PREFLIGHT_FAILURE:
    finish_loop (mc,
                 TALER_MHD_reply_with_error (
                   mc->rc->connection,
                   MHD_HTTP_INTERNAL_SERVER_ERROR,
                   TALER_EC_GENERIC_DB_COMMIT_FAILED,
                   "make_coin_known"));
    return;
  case MELT_ERROR_DB_MAKE_COIN_KNOW_FAILURE:
    finish_loop (mc,
                 TALER_MHD_reply_with_error (
                   mc->rc->connection,
                   MHD_HTTP_INTERNAL_SERVER_ERROR,
                   TALER_EC_GENERIC_DB_START_FAILED,
                   "preflight failure"));
    return;
  case MELT_ERROR_COIN_UNKNOWN:
    finish_loop (mc,
                 TALER_MHD_reply_with_ec (
                   mc->rc->connection,
                   TALER_EC_EXCHANGE_GENERIC_COIN_UNKNOWN,
                   NULL));
    return;
  case MELT_COIN_CONFLICTING_DENOMINATION_KEY:
    finish_loop (mc,
                 TALER_MHD_reply_with_ec (
                   mc->rc->connection,
                   TALER_EC_EXCHANGE_GENERIC_COIN_CONFLICTING_DENOMINATION_KEY,
                   TALER_B2S (&mc->error.details.denom_h)));
    return;
  case MELT_ERROR_COIN_EXPIRED_NO_ZOMBIE:
    finish_loop (mc,
                 TALER_MHD_reply_with_error (
                   mc->rc->connection,
                   MHD_HTTP_BAD_REQUEST,
                   TALER_EC_EXCHANGE_MELT_COIN_EXPIRED_NO_ZOMBIE,
                   NULL));
    return;
  case MELT_ERROR_DENOMINATION_SIGN:
    finish_loop (mc,
                 TALER_MHD_reply_with_ec (
                   mc->rc->connection,
                   mc->error.details.ec_denomination_sign,
                   NULL));
    return;
  case MELT_ERROR_DENOMINATION_SIGNATURE_INVALID:
    finish_loop (mc,
                 TALER_MHD_reply_with_error (mc->rc->connection,
                                             MHD_HTTP_FORBIDDEN,
                                             TALER_EC_EXCHANGE_DENOMINATION_SIGNATURE_INVALID,
                                             NULL));
    return;
  case MELT_ERROR_DENOMINATION_KEY_UNKNOWN:
    GNUNET_break_op (0);
    finish_loop (mc,
                 TEH_RESPONSE_reply_unknown_denom_pub_hash (
                   mc->rc->connection,
                   &mc->error.details.denom_h));
    return;
  case MELT_ERROR_DENOMINATION_EXPIRED:
    GNUNET_break_op (0);
    finish_loop (mc,
                 TEH_RESPONSE_reply_expired_denom_pub_hash (
                   mc->rc->connection,
                   &mc->error.details.denom_h,
                   TALER_EC_EXCHANGE_GENERIC_DENOMINATION_EXPIRED,
                   "MELT"));
    return;
  case MELT_ERROR_DENOMINATION_VALIDITY_IN_FUTURE:
    finish_loop (mc,
                 TEH_RESPONSE_reply_expired_denom_pub_hash (
                   mc->rc->connection,
                   &mc->error.details.denom_h,
                   TALER_EC_EXCHANGE_GENERIC_DENOMINATION_VALIDITY_IN_FUTURE,
                   "MELT"));
    return;
  case MELT_ERROR_DENOMINATION_REVOKED:
    GNUNET_break_op (0);
    finish_loop (mc,
                 TALER_MHD_reply_with_ec (
                   mc->rc->connection,
                   TALER_EC_EXCHANGE_GENERIC_DENOMINATION_REVOKED,
                   NULL));
    return;
  case MELT_ERROR_COIN_CIPHER_MISMATCH:
    finish_loop (mc,
                 TALER_MHD_reply_with_ec (
                   mc->rc->connection,
                   TALER_EC_EXCHANGE_GENERIC_CIPHER_MISMATCH,
                   NULL));
    return;
  case MELT_ERROR_BLINDING_SEED_REQUIRED:
    finish_loop (mc,
                 TALER_MHD_reply_with_ec (
                   mc->rc->connection,
                   TALER_EC_GENERIC_PARAMETER_MISSING,
                   "blinding_seed"));
    return;
  case MELT_ERROR_CRYPTO_HELPER:
    finish_loop (mc,
                 TALER_MHD_reply_with_ec (
                   mc->rc->connection,
                   TALER_EC_GENERIC_INTERNAL_INVARIANT_FAILURE,
                   NULL));
    return;
  case MELT_ERROR_AGE_RESTRICTION_NOT_SUPPORTED_BY_DENOMINATION:
    {
      char msg[256];

      GNUNET_snprintf (msg,
                       sizeof(msg),
                       "denomination %s does not support age restriction",
                       GNUNET_h2s (&mc->error.details.denom_h.hash));
      finish_loop (mc,
                   TALER_MHD_reply_with_ec (
                     mc->rc->connection,
                     TALER_EC_EXCHANGE_GENERIC_DENOMINATION_KEY_UNKNOWN,
                     msg));
      return;
    }
  case  MELT_ERROR_AGE_RESTRICTION_COMMITMENT_INVALID:
    finish_loop (mc,
                 TALER_MHD_reply_with_ec (
                   mc->rc->connection,
                   TALER_EC_EXCHANGE_REFRESHES_REVEAL_AGE_RESTRICTION_COMMITMENT_INVALID,
                   "old_age_commitment_h"));
    return;
  case MELT_ERROR_AMOUNT_OVERFLOW:
    finish_loop (mc,
                 TALER_MHD_reply_with_error (
                   mc->rc->connection,
                   MHD_HTTP_BAD_REQUEST,
                   TALER_EC_EXCHANGE_REFRESHES_REVEAL_COST_CALCULATION_OVERFLOW,
                   "amount"));
    return;
  case MELT_ERROR_AMOUNT_PLUS_FEE_OVERFLOW:
    finish_loop (mc,
                 TALER_MHD_reply_with_error (
                   mc->rc->connection,
                   MHD_HTTP_INTERNAL_SERVER_ERROR,
                   TALER_EC_EXCHANGE_REFRESHES_REVEAL_COST_CALCULATION_OVERFLOW,
                   "amount+fee"));
    return;
  case MELT_ERROR_FEES_EXCEED_CONTRIBUTION:
    finish_loop (mc,
                 TALER_MHD_reply_with_error (mc->rc->connection,
                                             MHD_HTTP_BAD_REQUEST,
                                             TALER_EC_EXCHANGE_MELT_FEES_EXCEED_CONTRIBUTION,
                                             NULL));
    return;
  case MELT_ERROR_AMOUNT_WITH_FEE_INCORRECT:
    finish_loop (mc,
                 TALER_MHD_reply_with_error (
                   mc->rc->connection,
                   MHD_HTTP_BAD_REQUEST,
                   TALER_EC_EXCHANGE_REFRESHES_REVEAL_COST_CALCULATION_OVERFLOW,
                   "value_with_fee incorrect"));
    return;
  case MELT_ERROR_CONFIRMATION_SIGN:
    finish_loop (mc,
                 TALER_MHD_reply_with_ec (
                   mc->rc->connection,
                   mc->error.details.ec_confirmation_sign,
                   NULL));
    return;
  case MELT_ERROR_INSUFFICIENT_FUNDS:
    finish_loop (mc,
                 TEH_RESPONSE_reply_coin_insufficient_funds (
                   mc->rc->connection,
                   TALER_EC_EXCHANGE_GENERIC_INSUFFICIENT_FUNDS,
                   &mc->request.refresh.coin.denom_pub_hash,
                   &mc->request.refresh.coin.coin_pub));
    return;
  case MELT_ERROR_IDEMPOTENT_PLANCHET:
    finish_loop (mc,
                 TALER_MHD_reply_with_error (
                   mc->rc->connection,
                   MHD_HTTP_BAD_REQUEST,
                   TALER_EC_GENERIC_PARAMETER_MALFORMED,   /* FIXME: new error! */
                   "idempotent planchet"));
    return;
  case MELT_ERROR_NONCE_RESUSE:
    finish_loop (mc,
                 TALER_MHD_reply_with_error (
                   mc->rc->connection,
                   MHD_HTTP_BAD_REQUEST,
                   TALER_EC_GENERIC_PARAMETER_MALFORMED, /* FIXME: new error */
                   "nonce reuse"));
    return;
  case MELT_ERROR_COIN_SIGNATURE_INVALID:
    finish_loop (mc,
                 TALER_MHD_reply_with_ec (
                   mc->rc->connection,
                   TALER_EC_EXCHANGE_MELT_COIN_SIGNATURE_INVALID,
                   NULL));
    return;
  }
  GNUNET_break (0);
  finish_loop (mc,
               TALER_MHD_reply_with_error (
                 mc->rc->connection,
                 MHD_HTTP_INTERNAL_SERVER_ERROR,
                 TALER_EC_GENERIC_INTERNAL_INVARIANT_FAILURE,
                 "error phase without error"));
}


/**
 * Function implementing melt transaction.  Runs the
 * transaction logic; IF it returns a non-error code, the transaction
 * logic MUST NOT queue a MHD response.  IF it returns an hard error,
 * the transaction logic MUST queue a MHD response and set @a mhd_ret.
 * IF it returns the soft error code, the function MAY be called again
 * to retry and MUST not queue a MHD response.
 *
 * @param cls a `struct MeltContext *`
 * @param connection MHD request which triggered the transaction
 * @param[out] mhd_ret set to MHD response status for @a connection,
 *             if transaction failed (!)
 * @return transaction status
 */
static enum GNUNET_DB_QueryStatus
melt_transaction (
  void *cls,
  struct MHD_Connection *connection,
  MHD_RESULT *mhd_ret)
{
  struct MeltContext *mc = cls;
  enum GNUNET_DB_QueryStatus qs;
  bool balance_ok;
  bool found;
  bool nonce_reuse;
  uint32_t noreveal_index;
  struct TALER_Amount insufficient_funds;

  qs = TEH_plugin->do_refresh (TEH_plugin->cls,
                               &mc->request.refresh,
                               &mc->now,
                               &found,
                               &noreveal_index,
                               &mc->zombie_required,
                               &nonce_reuse,
                               &balance_ok,
                               &insufficient_funds);
  if (0 > qs)
  {
    if (GNUNET_DB_STATUS_HARD_ERROR == qs)
      SET_ERROR_WITH_DETAIL (mc,
                             MELT_ERROR_DB_FETCH_FAILED,
                             db_fetch_context,
                             "do_refresh");
    return qs;
  }
  if (GNUNET_DB_STATUS_SUCCESS_NO_RESULTS == qs)
  {
    GNUNET_break_op (0);
    SET_ERROR (mc,
               MELT_ERROR_COIN_UNKNOWN);
    return GNUNET_DB_STATUS_HARD_ERROR;
  }
  if (found)
  {
    /**
     * This request is idempotent, set the nonreveal_index
     * to the previous one and reply success.
     */
    mc->request.refresh.noreveal_index = noreveal_index;
    mc->phase = MELT_PHASE_GENERATE_REPLY_SUCCESS;
    mc->error.code = MELT_ERROR_NONE;
    return GNUNET_DB_STATUS_SUCCESS_ONE_RESULT;
  }
  if (nonce_reuse)
  {
    GNUNET_break_op (0);
    SET_ERROR (mc,
               MELT_ERROR_NONCE_RESUSE);
    return GNUNET_DB_STATUS_HARD_ERROR;
  }
  if (! balance_ok)
  {
    GNUNET_break_op (0);
    SET_ERROR_WITH_FIELD (mc,
                          MELT_ERROR_INSUFFICIENT_FUNDS,
                          insufficient_funds);
    return GNUNET_DB_STATUS_HARD_ERROR;
  }
  if (mc->zombie_required)
  {
    GNUNET_break_op (0);
    SET_ERROR (mc,
               MELT_ERROR_COIN_EXPIRED_NO_ZOMBIE);
    return GNUNET_DB_STATUS_HARD_ERROR;
  }

  if (GNUNET_DB_STATUS_SUCCESS_ONE_RESULT == qs)
    TEH_METRICS_num_success[TEH_MT_SUCCESS_MELT]++;
  return qs;
}


/**
 * The request was prepared successfully.
 * Run the main DB transaction.
 *
 * @param mc The context for the current melt request
 */
static void
phase_run_transaction (
  struct MeltContext *mc)
{
  if (GNUNET_SYSERR ==
      TEH_plugin->preflight (TEH_plugin->cls))
  {
    GNUNET_break (0);
    SET_ERROR (mc,
               MELT_ERROR_DB_PREFLIGHT_FAILURE);
    return;
  }

  /* first, make sure coin is known */
  if (! mc->coin_is_dirty)
  {
    MHD_RESULT mhd_ret = -1;
    enum GNUNET_DB_QueryStatus qs;

    for (unsigned int tries = 0; tries<MAX_TRANSACTION_COMMIT_RETRIES; tries++)
    {
      qs = TEH_make_coin_known (&mc->request.refresh.coin,
                                mc->rc->connection,
                                &mc->known_coin_id,
                                &mhd_ret);
      if (GNUNET_DB_STATUS_SOFT_ERROR != qs)
        break;
    }
    if (0 > qs)
    {
      GNUNET_break (0);
      /* Check if an answer has been queued */
      switch (mhd_ret)
      {
      case MHD_NO:
        mc->phase = MELT_PHASE_RETURN_NO;
        return;
      case MHD_YES:
        mc->phase = MELT_PHASE_RETURN_YES;
        return;
      default:
        /* ignore */
      }
      SET_ERROR (mc,
                 MELT_ERROR_DB_MAKE_COIN_KNOW_FAILURE);
      return;
    }
  }

  /* run main database transaction */
  {
    MHD_RESULT mhd_ret = -1;
    enum GNUNET_GenericReturnValue ret;
    enum MeltPhase current_phase = mc->phase;

    GNUNET_assert (MELT_PHASE_RUN_TRANSACTION ==
                   current_phase);
    ret = TEH_DB_run_transaction (mc->rc->connection,
                                  "run melt",
                                  TEH_MT_REQUEST_MELT,
                                  &mhd_ret,
                                  &melt_transaction,
                                  mc);
    if (GNUNET_OK != ret)
    {
      GNUNET_break (0);
      /* Check if an answer has been queued */
      switch (mhd_ret)
      {
      case MHD_NO:
        mc->phase = MELT_PHASE_RETURN_NO;
        return;
      case MHD_YES:
        mc->phase = MELT_PHASE_RETURN_YES;
        return;
      default:
        /* ignore */
      }
      GNUNET_assert (MELT_ERROR_NONE != mc->error.code);
      GNUNET_assert (MELT_PHASE_GENERATE_REPLY_ERROR == mc->phase);
      return;
    }
    /**
     * In case of idempotency (which is not an error condition),
     * the phase has changed in melt_transaction.
     * We simple return.
     */
    if (current_phase != mc->phase)
      return;
  }
  mc->phase++;
}


MHD_RESULT
TEH_handler_melt_v27 (
  struct TEH_RequestContext *rc,
  const json_t *root,
  const char *const args[0])
{
  struct MeltContext *mc = rc->rh_ctx;

  (void) args;
  if (NULL == mc)
  {
    mc = GNUNET_new (struct MeltContext);
    rc->rh_ctx = mc;
    rc->rh_cleaner = &clean_melt_rc;
    mc->rc = rc;
    mc->now = GNUNET_TIME_timestamp_get ();
  }

  while (true)
  {
    GNUNET_log (GNUNET_ERROR_TYPE_INFO,
                "melt processing in phase %d\n",
                mc->phase);
    switch (mc->phase)
    {
    case MELT_PHASE_PARSE:
      phase_parse_request (mc,
                           root);
      break;
    case MELT_PHASE_CHECK_MELT_VALID:
      phase_check_melt_valid (mc);
      break;
    case MELT_PHASE_CHECK_KEYS:
      phase_check_keys (mc);
      break;
    case MELT_PHASE_CHECK_COIN_SIGNATURE:
      phase_check_coin_signature (mc);
      break;
    case MELT_PHASE_PREPARE_TRANSACTION:
      phase_prepare_transaction (mc);
      break;
    case MELT_PHASE_RUN_TRANSACTION:
      phase_run_transaction (mc);
      break;
    case MELT_PHASE_GENERATE_REPLY_SUCCESS:
      phase_generate_reply_success (mc);
      break;
    case MELT_PHASE_GENERATE_REPLY_ERROR:
      phase_generate_reply_error (mc);
      break;
    case MELT_PHASE_RETURN_YES:
      return MHD_YES;
    case MELT_PHASE_RETURN_NO:
      return MHD_NO;
    }
  }
}
