/*
  This file is part of TALER
 Copyright (C) 2025 Taler Systems SA

  TALER is free software; you can redistribute it and/or modify it under the
  terms of the GNU Lesser General Public License as published by the Free Software
  Foundation; either version 2.1, or (at your option) any later version.

  TALER is distributed in the hope that it will be useful, but WITHOUT ANY
  WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
  A PARTICULAR PURPOSE.  See the GNU Lesser General Public License for more details.

  You should have received a copy of the GNU Lesser General Public License along with
  TALER; see the file COPYING.LGPL.  If not, see
  <http://www.gnu.org/licenses/>
*/
/**
 * @file merchant_api_get_product_image.c
 * @brief Implementation of the GET /products/$HASH/image request
 * @author Bohdan Potuzhnyi
 */
#include "platform.h"
#include <curl/curl.h>
#include <jansson.h>
#include <microhttpd.h>
#include <gnunet/gnunet_util_lib.h>
#include <gnunet/gnunet_curl_lib.h>
#include <taler/taler_error_codes.h>
#include <taler/taler_json_lib.h>
#include "taler_merchant_service.h"
#include "merchant_api_curl_defaults.h"


/**
 * Handle for a GET /products/$HASH/image operation.
 */
struct TALER_MERCHANT_ProductImageGetHandle
{
  /**
   * The url for this request.
   */
  char *url;

  /**
   * Handle for the request.
   */
  struct GNUNET_CURL_Job *job;

  /**
   * Function to call with the result.
   */
  TALER_MERCHANT_ProductImageGetCallback cb;

  /**
   * Closure for @a cb.
   */
  void *cb_cls;

  /**
   * Reference to the execution context.
   */
  struct GNUNET_CURL_Context *ctx;
};


/**
 * Function called when we're done processing the
 * HTTP GET /products/$HASH/image request.
 *
 * @param cls the `struct TALER_MERCHANT_ProductImageGetHandle`
 * @param response_code HTTP response code, 0 on error
 * @param response response body, NULL if not in JSON
 */
static void
handle_get_product_image_finished (void *cls,
                                   long response_code,
                                   const void *response)
{
  struct TALER_MERCHANT_ProductImageGetHandle *pigh = cls;
  const json_t *json = response;
  struct TALER_MERCHANT_ProductImageGetResponse pir = {
    .hr.http_status = (unsigned int) response_code,
    .hr.reply = json
  };

  pigh->job = NULL;
  GNUNET_log (GNUNET_ERROR_TYPE_DEBUG,
              "Got /products/$HASH/image response with status code %u\n",
              (unsigned int) response_code);
  switch (response_code)
  {
  case MHD_HTTP_OK:
    {
      struct GNUNET_JSON_Specification spec[] = {
        GNUNET_JSON_spec_string ("image",
                                 &pir.details.ok.image),
        GNUNET_JSON_spec_end ()
      };

      if (GNUNET_OK ==
          GNUNET_JSON_parse (json,
                             spec,
                             NULL, NULL))
      {
        pigh->cb (pigh->cb_cls,
                  &pir);
        GNUNET_JSON_parse_free (spec);
        TALER_MERCHANT_product_image_get_cancel (pigh);
        return;
      }
      pir.hr.http_status = 0;
      pir.hr.ec = TALER_EC_GENERIC_INVALID_RESPONSE;
      break;
    }
  case MHD_HTTP_NOT_FOUND:
  case MHD_HTTP_BAD_REQUEST:
    pir.hr.ec = TALER_JSON_get_error_code (json);
    pir.hr.hint = TALER_JSON_get_error_hint (json);
    break;
  default:
    pir.hr.ec = TALER_JSON_get_error_code (json);
    pir.hr.hint = TALER_JSON_get_error_hint (json);
    GNUNET_log (GNUNET_ERROR_TYPE_ERROR,
                "Unexpected response code %u/%d\n",
                (unsigned int) response_code,
                (int) pir.hr.ec);
    break;
  }
  pigh->cb (pigh->cb_cls,
            &pir);
  TALER_MERCHANT_product_image_get_cancel (pigh);
}


struct TALER_MERCHANT_ProductImageGetHandle *
TALER_MERCHANT_product_image_get (
  struct GNUNET_CURL_Context *ctx,
  const char *backend_url,
  const char *image_hash,
  TALER_MERCHANT_ProductImageGetCallback cb,
  void *cb_cls)
{
  struct TALER_MERCHANT_ProductImageGetHandle *pigh;
  CURL *eh;

  pigh = GNUNET_new (struct TALER_MERCHANT_ProductImageGetHandle);
  pigh->ctx = ctx;
  pigh->cb = cb;
  pigh->cb_cls = cb_cls;
  {
    char *path;

    GNUNET_asprintf (&path,
                     "products/%s/image",
                     image_hash);
    pigh->url = TALER_url_join (backend_url,
                                path,
                                NULL);
    GNUNET_free (path);
  }
  if (NULL == pigh->url)
  {
    GNUNET_log (GNUNET_ERROR_TYPE_ERROR,
                "Could not construct request URL.\n");
    GNUNET_free (pigh);
    return NULL;
  }
  GNUNET_log (GNUNET_ERROR_TYPE_DEBUG,
              "Requesting URL '%s'\n",
              pigh->url);
  eh = TALER_MERCHANT_curl_easy_get_ (pigh->url);
  pigh->job = GNUNET_CURL_job_add (ctx,
                                   eh,
                                   &handle_get_product_image_finished,
                                   pigh);
  return pigh;
}


void
TALER_MERCHANT_product_image_get_cancel (
  struct TALER_MERCHANT_ProductImageGetHandle *pigh)
{
  if (NULL != pigh->job)
    GNUNET_CURL_job_cancel (pigh->job);
  GNUNET_free (pigh->url);
  GNUNET_free (pigh);
}