/* tag: kerberos utility functions (source)
 *
 * Copyright (C) 2000  Andrew Chatham
 * Copyright (C) 2003  Charles Duffy
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */

#include <stdio.h>
#include <stdlib.h>
#include <com_err.h>
#include <config.h>

#include "krb-util.h"

#include <krb5.h>

#ifdef DO_KRB524
#include <kerberosIV/krb.h>
#endif

#ifdef DO_KRBAFS
#include <krbafs.h>
#endif

#include <libgnome/libgnome.h>

#ifdef DO_KRB524
extern int krb524_init_ets(krb5_context context);
extern int krb524_convert_creds_kdc(krb5_context, krb5_creds *, CREDENTIALS*);
#endif

char * progname = "ticket_applet";


#define clear_error(e) memset((char *) &e, 0, sizeof(krb_error))

static krb_error
make_error(krb5_error_code code)
{
  krb_error error;

  strncpy(error.message, error_message(code), 255);
  error.code = code;
  return error;
}

krb_error
krb_get_tickets(char *princ, char *pass)
{
  krb5_context context;
  krb5_principal me = NULL;
  krb5_creds creds;
  krb5_get_init_creds_opt opts;
  krb5_error_code code;
  krb5_ccache ccache = NULL;
#ifdef DO_KRB524
  CREDENTIALS v4_creds;
#endif

  krb_error error;
  clear_error(error);

  if ((code = krb5_init_context(&context))) {
    error = make_error(code);
    return error;
  }
  
  krb5_get_init_creds_opt_init(&opts);
  
  if ((code = krb5_parse_name(context, princ, &me))) {
    error = make_error(code);
    goto bail_rest;
  }

  if ((code = krb5_cc_default(context, &ccache))) {
    error = make_error(code);
    goto bail_rest;
  }
  
  if ((code = krb5_get_init_creds_password(context, &creds, 
					   me, pass,
					   NULL, NULL, 
					   0, NULL, 
					   &opts))) {
    error = make_error(code);
    if (code == KRB5KRB_AP_ERR_BAD_INTEGRITY) {
      strcpy(error.message, _("Password incorrect"));
    }
    goto bail_rest;
  }

  if ((code = krb5_cc_initialize(context, ccache, me))) {
    error = make_error(code);
    goto bail_creds;
  }

  if ((code = krb5_cc_store_cred(context, ccache, &creds))) {
    error = make_error(code);
    goto bail_creds;
  }

#ifdef DO_KRB524
  if ((code = krb524_convert_creds_kdc(context, &creds, &v4_creds))) {
    error = make_error(code);
    goto bail_creds;
  }

  if ((code = in_tkt(v4_creds.pname, v4_creds.pinst) != KSUCCESS)) {
    error = make_error(code);
    goto bail_creds;
  }
  
  if ((code = krb_save_credentials(v4_creds.service, v4_creds.instance,
				   v4_creds.realm, v4_creds.session,
				   v4_creds.lifetime, v4_creds.kvno,
				   &(v4_creds.ticket_st), 
				   v4_creds.issue_date))) {
    error = make_error(code);
    goto bail_creds;
  }

#endif //DO_KRB524

 bail_creds:

  krb5_free_cred_contents(context, &creds);

 bail_rest:

  if (me)
    krb5_free_principal(context, me);
  if (ccache)
    krb5_cc_close(context, ccache);

  krb5_free_context(context);

  return error;
}


krb_error
krb_time_left(int * t)
{
  krb5_context context;
  krb5_error_code code;
  krb5_ccache ccache = NULL;
  krb5_principal me = NULL;
  krb5_cc_cursor cur;
  krb5_creds creds;
  krb5_int32 now;
  krb_error error;

  *t = 0;
  clear_error(error);

  if ((code = krb5_init_context(&context))) {
    error = make_error(code);
    return error;
  }

  if ((code = krb5_cc_default(context, &ccache))) {
    error = make_error(code);
    goto bail;
  }

  if ((code = krb5_cc_get_principal(context, ccache, &me))) {
    error = make_error(code);
    goto bail;
  }

  if ((code = krb5_cc_start_seq_get(context, ccache, &cur))) {
    error = make_error(code);
    goto bail;
  }

  if ((code = krb5_timeofday(context, &now))) {
    error = make_error(code);
  }
  else
    while (!(code = krb5_cc_next_cred(context, ccache, &cur, &creds))) {
      *t = creds.times.endtime - now;
      *t = *t > 0 ? *t : 0;
      krb5_free_cred_contents(context, &creds);
    }

  if ((code = krb5_cc_end_seq_get(context, ccache, &cur))) {
    error = make_error(code);
  }

  krb5_free_cred_contents(context, &creds);

 bail:


  if (me)
    krb5_free_principal(context, me);
  if (ccache)
    krb5_cc_close(context, ccache);

  krb5_free_context(context);

  return error;

}

char *
krb_time_left_as_string(void)
{
  int t;
  int min,hr;
  char *msg;
  krb_error error;
  
  error = krb_time_left(&t);

  if (error.code) {
    msg = strdup("???");
    fprintf(stderr, "WARNING " PACKAGE ": %s\n\n", 
	    error.message);
  } else { 
    min = t / 60 % 60;
    hr = t / 60 / 60 % 60;
    msg = malloc(15);
    snprintf(msg, 15, " %d : %.2d ", hr, min);
  }
  
  return msg;
}

#ifdef DO_KRBAFS
int get_afs_tokens(void)
{
  if (!k_hasafs()) {
    printf("Not k_hasafs\n");
    return 1;
  }
  if(krb_afslog(0, 0)) {
    printf("Failed getting tokens for default cell.\n");
    return 1;
  }
  return 0;
}
#endif // DO_KRBAFS
