;;; GNU Guix --- Functional package management for GNU
;;; Copyright © 2013 Andreas Enge <andreas@enge.fr>
;;; Copyright © 2013 Joshua Grant <tadni@riseup.net>
;;; Copyright © 2014 David Thompson <davet@gnu.org>
;;; Copyright © 2014 Mark H Weaver <mhw@netris.org>
;;;
;;; This file is part of GNU Guix.
;;;
;;; GNU Guix is free software; you can redistribute it and/or modify it
;;; under the terms of the GNU General Public License as published by
;;; the Free Software Foundation; either version 3 of the License, or (at
;;; your option) any later version.
;;;
;;; GNU Guix is distributed in the hope that it will be useful, but
;;; WITHOUT ANY WARRANTY; without even the implied warranty of
;;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
;;; GNU General Public License for more details.
;;;
;;; You should have received a copy of the GNU General Public License
;;; along with GNU Guix.  If not, see <http://www.gnu.org/licenses/>.

(define-module (gnu packages gl)
  #:use-module (ice-9 match)
  #:use-module ((guix licenses) #:prefix l:)
  #:use-module (guix packages)
  #:use-module (guix download)
  #:use-module (guix build-system gnu)
  #:use-module (guix packages)
  #:use-module (gnu packages bison)
  #:use-module (gnu packages flex)
  #:use-module (gnu packages pkg-config)
  #:use-module (gnu packages gettext)
  #:use-module (gnu packages linux)
  #:use-module (gnu packages python)
  #:use-module (gnu packages xorg)
  #:use-module (gnu packages xml)
  #:use-module (gnu packages fontutils)
  #:use-module (gnu packages guile)
  #:use-module (gnu packages xdisorg))

(define-public glu
  (package
    (name "glu")
    (version "9.0.0")
    (source (origin
	     (method url-fetch)
	     (uri (string-append "ftp://ftp.freedesktop.org/pub/mesa/glu/glu-"
				  version ".tar.gz"))
	     (sha256
	      (base32 "0r72yyhj09x3krn3kn629jqbwyq50ji8w5ri2pn6zwrk35m4g1s3"))))
    (build-system gnu-build-system)
    (inputs `(("mesa" ,mesa)))
    (home-page "http://www.opengl.org/archives/resources/faq/technical/glu.htm")
    (synopsis "Mesa OpenGL Utility library")
    (description
     "GLU, or OpenGL Utility Library provides some higher-level functionality
not provided by just OpenGL itself.  Some of GLU's Features
include: Scaling of 2D images and creation of mipmap pyramids,
Transformation of object coordinates into device coordinates and
vice versa, Support for NURBS surfaces, Support for tessellation
of concave or bow tie polygonal primitives, Specialty transformation
matrices for creating perspective and orthographic projections,
positioning a camera, and selection/picking, Rendering of disk,
cylinder, and sphere primitives, Interpreting OpenGL error values
as ASCII text.")
    (license (l:x11-style "http://directory.fsf.org/wiki/License:SGIFreeBv2"))))

(define-public freeglut
  (package
    (name "freeglut")
    (version "2.8.1")
    (source (origin
	     (method url-fetch)
	     (uri (string-append "mirror://sourceforge/project/freeglut/freeglut/"
				  version "/freeglut-" version ".tar.gz"))
	     (sha256
	      (base32 "16lrxxxd9ps9l69y3zsw6iy0drwjsp6m26d1937xj71alqk6dr6x"))))
    (build-system gnu-build-system)
    (inputs `(("mesa" ,mesa)
	      ("libx11" ,libx11)
	      ("libxi" ,libxi)
	      ("libxrandr" ,libxrandr)
	      ("libxxf86vm" ,libxxf86vm)
	      ("inputproto" ,inputproto)
	      ("xinput" ,xinput)
	      ("glu" ,glu)))
    (home-page "http://freeglut.sourceforge.net/")
    (synopsis "Alternative to the OpenGL Utility Toolkit (GLUT)")
    (description
     "Freeglut is a completely Free/OpenSourced alternative to
the OpenGL Utility Toolkit (GLUT) library.  GLUT was originally
written by Mark Kilgard to support the sample programs in the
second edition OpenGL 'RedBook'.  Since then, GLUT has been used
in a wide variety of practical applications because it is simple,
widely available and highly portable.

GLUT (and hence freeglut) allows the user to create and manage windows
containing OpenGL contexts on a wide range of platforms and also read
the mouse, keyboard and joystick functions.  Freeglut is released under
the X-Consortium license.")
    (license l:x11)))

(define-public ftgl
  (package
    (name "ftgl")
    (version "2.1.3-rc5")
    (source (origin
	     (method url-fetch)
	     (uri (string-append "mirror://sourceforge/project/ftgl/FTGL%20Source/2.1.3~rc5/ftgl-"
				  version ".tar.gz"))
	     (sha256
	      (base32 "0nsn4s6vnv5xcgxcw6q031amvh2zfj2smy1r5mbnjj2548hxcn2l"))))
    (build-system gnu-build-system)
    (inputs `(("freetype" ,freetype)
	      ("libx11" ,libx11)
	      ("mesa" ,mesa)
	      ("glu" ,glu)))
    (home-page "http://ftgl.sourceforge.net")
    (synopsis "Font rendering library for OpenGL applications")
    (description
     "FTGL is a font rendering library for OpenGL applications.  Supported
rendering modes are: Bitmaps, Anti-aliased pixmaps, Texture maps, Outlines,
Polygon meshes, and Extruded polygon meshes")
    (license l:x11)))

(define-public mesa
  (package
    (name "mesa")
    (version "10.4.0")
    (source
      (origin
        (method url-fetch)
        (uri (string-append "ftp://ftp.freedesktop.org/pub/mesa/"
                            version "/MesaLib-" version ".tar.bz2"))
        (sha256
         (base32
          "069j4ck51hc70gryhw3z0rkyhhl0bnhbks4xg1wqqw56l7rxz9wq"))))
    (build-system gnu-build-system)
    (propagated-inputs
      `(("glproto" ,glproto)
        ("libdrm" ,libdrm)
        ("libx11" ,libx11)
        ("libxdamage" ,libxdamage)
        ("libxxf86vm" ,libxxf86vm)))
    (inputs
      `(("udev" ,eudev)
        ("dri2proto" ,dri2proto)
        ("dri3proto" ,dri3proto)
        ("presentproto" ,presentproto)
        ("libxshmfence" ,libxshmfence)
        ("expat" ,expat)
        ("libxfixes" ,libxfixes)
        ("libxml2" ,libxml2)
        ;; TODO: Add 'libva'
        ;; TODO: Add 'libxml2-python' for OpenGL ES 1.1 and 2.0 support
        ("makedepend" ,makedepend)))
    (native-inputs
      `(("pkg-config" ,pkg-config)
        ("gettext" ,gnu-gettext)
        ("flex" ,flex)
        ("bison" ,bison)
        ("python" ,python-2))) ; incompatible with Python 3 (print syntax)
    (arguments
     `(#:configure-flags
       '(;; drop r300 from default gallium drivers, as it requires llvm
         "--with-gallium-drivers=r600,svga,swrast"
         "--enable-xa"

         ;; on non-intel systems, drop i915 and i965
         ;; from the default dri drivers
         ,@(match (%current-system)
             ((or "x86_64-linux" "i686-linux")
              '())
             (_
              '("--with-dri-drivers=nouveau,r200,radeon,swrast"))))
       #:phases (alist-cons-after
                 'unpack 'add-missing-m4-files
                 (lambda _
                   ;; When these files are missing, make tries to rebuild
                   ;; several parts of the build system.
                   (zero? (system* "touch" "--date=@0"
                                   "m4/libtool.m4" "m4/ltoptions.m4"
                                   "m4/ltsugar.m4" "m4/ltversion.m4"
                                   "m4/lt~obsolete.m4")))
                 (alist-cons-after
                  'unpack 'patch-create_test_cases
                  (lambda _
                    (substitute* "src/glsl/tests/lower_jumps/create_test_cases.py"
                      (("/usr/bin/env bash") (which "bash"))))
                  %standard-phases))))
    (home-page "http://mesa3d.org/")
    (synopsis "OpenGL implementation")
    (description "Mesa is a free implementation of the OpenGL specification -
a system for rendering interactive 3D graphics.  A variety of device drivers
allows Mesa to be used in many different environments ranging from software
emulation to complete hardware acceleration for modern GPUs.")
    (license l:x11)))

(define-public glew
  (package
    (name "glew")
    (version "1.11.0")
    (source (origin
              (method url-fetch)
              (uri (string-append
                    "mirror://sourceforge/glew/glew-"
                    version
                    ".tgz"))
              (sha256
               (base32
                "1mhkllxz49l1x680dmzrv2i82qjrq017sykah3xc90f2d8qcxfv9"))
              (modules '((guix build utils)))
              (snippet
               '(substitute* "config/Makefile.linux"
                  (("= cc") "= gcc")
                  (("/lib64") "/lib")))))
    (build-system gnu-build-system)
    (arguments
     '(#:phases (alist-delete 'configure %standard-phases)
       #:make-flags (list (string-append "GLEW_PREFIX="
                                         (assoc-ref %outputs "out"))
                          (string-append "GLEW_DEST="
                                         (assoc-ref %outputs "out")))
       #:tests? #f))                              ;no 'check' target
    (inputs
     `(("libxi" ,libxi)
       ("libxmu" ,libxmu)
       ("libx11" ,libx11)
       ("mesa" ,mesa)))

    ;; <GL/glew.h> includes <GL/glu.h>.
    (propagated-inputs `(("glu" ,glu)))

    (home-page "http://glew.sourceforge.net/")
    (synopsis "OpenGL extension loading library for C and C++")
    (description
     "The OpenGL Extension Wrangler Library (GLEW) is a C/C++ extension
loading library.  GLEW provides efficient run-time mechanisms for determining
which OpenGL extensions are supported on the target platform.  OpenGL core and
extension functionality is exposed in a single header file.")
    (license l:bsd-3)))

(define-public guile-opengl
  (package
    (name "guile-opengl")
    (version "0.1.0")
    (source (origin
             (method url-fetch)
             (uri (string-append "mirror://gnu/guile-opengl/guile-opengl-"
                                 version ".tar.gz"))
             (sha256
              (base32
               "13qfx4xh8baryxqrv986l848ygd0piqwm6s2s90pxk9c0m9vklim"))))
    (build-system gnu-build-system)
    (native-inputs `(("pkg-config" ,pkg-config)))
    (inputs `(("guile" ,guile-2.0)
              ("mesa" ,mesa)
              ("freeglut" ,freeglut)))
    (arguments
     '(#:phases (alist-cons-before
                 'build 'patch-dynamic-link
                 (lambda* (#:key inputs outputs #:allow-other-keys)
                   (define (dynamic-link-substitute file lib input)
                     (substitute* file
                       (("dynamic-link \"lib([a-zA-Z]+)\"" _ lib)
                        (string-append "dynamic-link \""
                                       (assoc-ref inputs input)
                                       "/lib/lib" lib "\""))))
                   ;; Replace dynamic-link calls for libGL, libGLU, and
                   ;; libglut with absolute paths to the store.
                   (dynamic-link-substitute "glx/runtime.scm" "GL" "mesa")
                   (dynamic-link-substitute "glu/runtime.scm" "GLU" "mesa")
                   (dynamic-link-substitute "glut/runtime.scm" "glut"
                                            "freeglut"))
                 %standard-phases)))
    (home-page "http://gnu.org/s/guile-opengl")
    (synopsis "Guile binding for the OpenGL graphics API")
    (description
     "Guile-OpenGL is a library for Guile that provides bindings to the
OpenGL graphics API.")
    (license l:lgpl3+)))
