/*
 This file is part of GNU Taler
 (C) 2022 Taler Systems S.A.

 GNU Taler is free software; you can redistribute it and/or modify it under the
 terms of the GNU General Public License as published by the Free Software
 Foundation; either version 3, or (at your option) any later version.

 GNU Taler is distributed in the hope that it will be useful, but WITHOUT ANY
 WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
 A PARTICULAR PURPOSE.  See the GNU General Public License for more details.

 You should have received a copy of the GNU General Public License along with
 GNU Taler; see the file COPYING.  If not, see <http://www.gnu.org/licenses/>
 */

import { ExchangeTosStatus } from "@gnu-taler/taler-util";
import { useTranslationContext } from "@gnu-taler/web-util/browser";
import { Fragment, h, VNode } from "preact";
import { CheckboxOutlined } from "../../components/CheckboxOutlined.js";
import { ExchangeXmlTos } from "../../components/ExchangeToS.js";
import {
  Input,
  LinkSuccess,
  TermsOfServiceStyle,
  WarningBox,
} from "../../components/styled/index.js";
import { Button } from "../../mui/Button.js";
import { State } from "./index.js";
import { SelectList } from "../SelectList.js";
import { EnabledBySettings } from "../EnabledBySettings.js";

export function ShowButtonsAcceptedTosView({
  termsAccepted,
  showingTermsOfService,
  children,
}: State.ShowButtonsAccepted): VNode {
  const { i18n } = useTranslationContext();

  return (
    <Fragment>
      {showingTermsOfService.button.onClick !== undefined && (
        <Fragment>
          <section style={{ justifyContent: "space-around", display: "flex" }}>
            <LinkSuccess
              upperCased
              onClick={showingTermsOfService.button.onClick}
            >
              <i18n.Translate>Show terms of service</i18n.Translate>
            </LinkSuccess>
          </section>
          {termsAccepted.button.onClick !== undefined && (
            <section
              style={{ justifyContent: "space-around", display: "flex" }}
            >
              <CheckboxOutlined
                name="terms"
                enabled={termsAccepted.value}
                label={
                  <i18n.Translate>
                    I accept the exchange terms of service
                  </i18n.Translate>
                }
                onToggle={termsAccepted.button.onClick}
              />
            </section>
          )}
        </Fragment>
      )}
      {children}
    </Fragment>
  );
}

export function ShowButtonsNonAcceptedTosView({
  showingTermsOfService,
  terms,
}: State.ShowButtonsNotAccepted): VNode {
  const { i18n } = useTranslationContext();

  return (
    <Fragment>
      <section style={{ justifyContent: "space-around", display: "flex" }}>
        <Button
          variant="contained"
          color="success"
          onClick={showingTermsOfService.button.onClick}
        >
          <i18n.Translate>Review exchange terms of service</i18n.Translate>
        </Button>
      </section>
    </Fragment>
  );
}

export function ShowTosContentView({
  termsAccepted,
  showingTermsOfService,
  terms,
  tosLang,
  tosFormat,
  skipTos,
}: State.ShowContent): VNode {
  const { i18n } = useTranslationContext();
  const ableToReviewTermsOfService = termsAccepted.button.onClick !== undefined;

  if (terms.status === ExchangeTosStatus.MissingTos) {
    return (
      <section>
        <section style={{ justifyContent: "space-around", display: "flex" }}>
          <WarningBox>
            <i18n.Translate>
              The exchange doesn't have a terms of service.
            </i18n.Translate>
          </WarningBox>
        </section>

        <section style={{ justifyContent: "space-around", display: "flex" }}>
          <Button
            variant="contained"
            color="success"
            disabled={!skipTos.onClick}
            onClick={skipTos.onClick}
          >
            <i18n.Translate>Skip it for now.</i18n.Translate>
          </Button>
        </section>
      </section>
    );
  }
  return (
    <section>
      <Input style={{ display: "flex", justifyContent: "end" }}>
        <EnabledBySettings name="selectTosFormat">
          <SelectList
            label={i18n.str`Format`}
            list={tosFormat.list}
            name="format"
            value={tosFormat.value}
            onChange={tosFormat.onChange}
          />
        </EnabledBySettings>
        <SelectList
          label={i18n.str`Language`}
          list={tosLang.list}
          name="lang"
          value={tosLang.value}
          onChange={tosLang.onChange}
        />
      </Input>

      {!terms.content && (
        <section style={{ justifyContent: "space-around", display: "flex" }}>
          <WarningBox>
            <i18n.Translate>
              The exchange replied with a empty terms of service
            </i18n.Translate>
          </WarningBox>
        </section>
      )}
      {terms.content && (
        <section
          style={{
            justifyContent: "space-around",
            display: "flex",
            position: "relative",
            resize: "vertical",
            overflow: "hidden",
            marginTop: "4px",
            minHeight: "120px",
            height: "240px",
          }}
        >
          {terms.content.type === "xml" &&
            (!terms.content.document ? (
              <WarningBox>
                <i18n.Translate>
                  No terms of service. The exchange replied with a empty
                  document
                </i18n.Translate>
              </WarningBox>
            ) : (
              <TermsOfServiceStyle>
                <ExchangeXmlTos doc={terms.content.document} />
              </TermsOfServiceStyle>
            ))}
          {terms.content.type === "plain" &&
            (!terms.content.content ? (
              <WarningBox>
                <i18n.Translate>
                  No terms of service. The exchange replied with a empty text
                </i18n.Translate>
              </WarningBox>
            ) : (
              <div style={{ textAlign: "left" }}>
                <pre>{terms.content.content}</pre>
              </div>
            ))}
          {terms.content.type === "html" && (
            <iframe
              style={{
                width: "100%",
                height: "100%",
                border: "2px solid #0003",
                borderRadius: "4px",
                boxSizing: "border-box",
              }}
              src={`data:text/html;utf-8,${encodeURIComponent(
                terms.content.html,
              )}`}
            />
          )}
          {terms.content.type === "pdf" && (
            <a href={terms.content.location.toString()} download="tos.pdf">
              <i18n.Translate>Download Terms of Service</i18n.Translate>
            </a>
          )}
        </section>
      )}
      {showingTermsOfService && ableToReviewTermsOfService && (
        <section style={{ justifyContent: "space-around", display: "flex" }}>
          <LinkSuccess
            upperCased
            onClick={showingTermsOfService.button.onClick}
          >
            <i18n.Translate>Hide terms of service</i18n.Translate>
          </LinkSuccess>
        </section>
      )}
      {termsAccepted.button.onClick &&
        terms.status !== ExchangeTosStatus.Accepted && (
          <section style={{ justifyContent: "space-around", display: "flex" }}>
            <CheckboxOutlined
              name="terms"
              enabled={termsAccepted.value}
              label={
                <i18n.Translate>
                  I accept the exchange terms of service
                </i18n.Translate>
              }
              onToggle={termsAccepted.button.onClick}
            />
          </section>
        )}
    </section>
  );
}
