/* ncftpget.c
 *
 * A non-interactive utility to grab files from a remote FTP server.
 * Very useful in shell scripts!
 */

#ifdef HAVE_CONFIG_H
#	include <Config.h>
#endif

#include <unistd.h>
#include <sys/types.h>
#include <stdio.h>
#include <string.h>
#include <ctype.h>
#include <stdlib.h>
#include <signal.h>
#include <setjmp.h>

#include <ncftp.h>				/* Library header. */
#include <Strn.h>				/* Library header. */
#include "gpshare.h"

jmp_buf gJmp;
int gGotSig = 0;
int gCanJmp = 0;

extern char *optarg;
extern int optind;

static void
Usage(void)
{
	FILE *fp;

	fp = popen("more", "w");
	if (fp == NULL)
		fp = stderr;

	fprintf(fp, "Usages:\n");
	fprintf(fp, "  ncftpget [flags] remote-host local-dir remote-path-names...   (mode 1)\n");
	fprintf(fp, "  ncftpget -f login.cfg [flags] local-dir remote-path-names...  (mode 2)\n");
	fprintf(fp, "  ncftpget [flags] ftp://url.style.host/path/name               (mode 3)\n");
	fprintf(fp, "\nFlags:\n\
  -u XX  Use username XX instead of anonymous.\n\
  -p XX  Use password XX with the username.\n\
  -P XX  Use port number XX instead of the default FTP service port (21).\n\
  -d XX  Use the file XX for debug logging.\n\
  -a     Use ASCII transfer type instead of binary.\n");
	fprintf(fp, "\
  -t XX  Timeout after XX seconds.\n\
  -v/-V  Do (do not) use progress meters.\n\
  -f XX  Read the file XX for host, user, and password information.\n\
  -A     Append to local files, instead of overwriting them.\n\
  -Z     Do not try to resume transfers.\n\
  -R     Recursive mode; copy whole directory trees.");
	fprintf(fp, "\nExamples:\n\
  ncftpget ftp.wustl.edu . /pub/README /pub/README.too\n\
  ncftpget ftp.wustl.edu . '/pub/README*'\n\
  ncftpget -R ftp.probe.net /tmp /pub/ncftpd  (ncftpd is a directory)\n\
  ncftpget ftp://ftp.wustl.edu/pub/README\n\
  ncftpget -u gleason -p my.password Bozo.probe.net . '/home/mjg/.*rc'\n\
  ncftpget -u gleason Bozo.probe.net . /home/mjg/foo.txt  (prompt for password)\n\
  ncftpget -f Bozo.cfg '/home/mjg/.*rc'\n\
  ncftpget -a -d /tmp/debug.log -t 60 ftp.wustl.edu . '/pub/README*'\n");

	fprintf(fp, "\nLibrary version: %s.\n", gLibNcFTPVersion + 5);
#ifdef UNAME
	fprintf(fp, "System: %s.\n", UNAME);
#endif
	fprintf(fp, "\nThis is a freeware program by Mike Gleason (mgleason@probe.net).\n");

	if (fp != stderr)
		pclose(fp);
	exit(kExitUsage);
}	/* Usage */



static void
Abort(int sigNum)
{
	if (gCanJmp == 0) {
		gCanJmp = 0;
		gGotSig = sigNum;
		longjmp(gJmp, 1);
	}
}	/* Abort */




static int 
Copy(FTPCIPtr cip, char *dstdir, const char ** volatile files, int rflag, int xtype, int resumeflag, int appendflag)
{
	int i;
	int result;
	const char *file;
	int rc = 0;

	for (i=0; ; i++) {
		file = files[i];
		if (file == NULL)
			break;
		result = FTPGetFiles2(cip, file, dstdir, rflag, kGlobYes, xtype, resumeflag, appendflag);
		if (result != 0) {
			fprintf(stderr, "ncftpget: file retrieval error: %s.\n", FTPStrError(result));
			rc = result;
		}
	}
	return (rc);
}	/* Copy */




void
main(int argc, char **argv)
{
	int result, c;
	volatile int rflag = 0;
	volatile int xtype = kTypeBinary;
	volatile int appendflag = kAppendNo;
	volatile int resumeflag = kResumeYes;
	int progmeters;
	FTPLibraryInfo li;
	FTPConnectionInfo fi;
	char * volatile dstdir;
	const char ** volatile flist;
	const char * volatile errstr;
	volatile ExitStatus es;
	char url[256];
	char urlfile[128];
	LineList cdlist;
	LinePtr lp;
	int rc;
#ifdef HAVE_GETPASS
	char *password;
#endif

	result = FTPInitLibrary(&li);
	if (result < 0) {
		fprintf(stderr, "ncftpget: init library error %d (%s).\n", result, FTPStrError(result));
		exit(kExitInitLibraryFailed);
	}
	result = FTPInitConnectionInfo(&li, &fi, kDefaultFTPBufSize);
	if (result < 0) {
		fprintf(stderr, "ncftpget: init connection info error %d (%s).\n", result, FTPStrError(result));
		exit(kExitInitConnInfoFailed);
	}

	fi.debugLog = NULL;
	fi.errLog = stderr;
	STRNCPY(fi.user, "anonymous");
	fi.netTimeout = (unsigned int) 180;
	fi.host[0] = '\0';
	progmeters = ((isatty(2) != 0) && (getppid() > 1)) ? 1 : 0;
	urlfile[0] = '\0';
	InitLineList(&cdlist);

	while ((c = getopt(argc, argv, "P:u:p:e:d:t:aRrvVf:AZ")) > 0) switch(c) {
		case 'P':
			fi.port = atoi(optarg);	
			break;
		case 'u':
			STRNCPY(fi.user, optarg);
			break;
		case 'p':
			STRNCPY(fi.pass, optarg);	/* Don't recommend doing this! */
			break;
		case 'e':
			if (strcmp(optarg, "stdout") == 0)
				fi.errLog = stdout;
			else if (optarg[0] == '-')
				fi.errLog = stdout;
			else if (strcmp(optarg, "stderr") == 0)
				fi.errLog = stderr;
			else
				fi.errLog = fopen(optarg, "a");
			break;
		case 'd':
			if (strcmp(optarg, "stdout") == 0)
				fi.debugLog = stdout;
			else if (optarg[0] == '-')
				fi.debugLog = stdout;
			else if (strcmp(optarg, "stderr") == 0)
				fi.debugLog = stderr;
			else
				fi.debugLog = fopen(optarg, "a");
			break;
		case 't':
			fi.netTimeout = (unsigned int) atoi(optarg);	
			break;
		case 'a':
			xtype = kTypeAscii;
			break;
		case 'R':
		case 'r':
			rflag = 1;
			break;
		case 'v':
			progmeters = 1;
			break;
		case 'V':
			progmeters = 0;
			break;
		case 'f':
			ReadConfigFile(optarg, &fi);
			break;
		case 'A':
			appendflag = kAppendYes;
			break;
		case 'Z':
			resumeflag = kResumeNo;
			break;
		default:
			Usage();
	}
	if (optind > argc - 1)
		Usage();

	if (progmeters != 0)
		fi.progress = PrStatBar;

	if (fi.host[0] == '\0') {
		STRNCPY(url, argv[optind]);
		rc = FTPDecodeURL(&fi, url, &cdlist, urlfile, sizeof(urlfile), (int *) &xtype, NULL);
		if (rc == kMalformedURL) {
			fprintf(stderr, "Malformed URL: %s\n", url);
			exit(kExitMalformedURL);
		} else if (rc == kNotURL) {
			/* This is what should happen most of the time. */
			if (optind > argc - 3)
				Usage();
			STRNCPY(fi.host, argv[optind]);
			dstdir = argv[optind + 1];
			flist = (const char ** volatile) argv + optind + 2;
		} else {
			/* URL okay */
			flist = NULL;
			if ((urlfile[0] == '\0') && (rflag == 0)) {
				/* It was obviously a directory, and they didn't say -R. */
				fprintf(stderr, "ncftpget: Use -R if you want the whole directory tree.\n");
				es = kExitUsage;
				exit(es);
			}
		}
	} else {
		if (optind > argc - 2)
			Usage();
		dstdir = argv[optind + 0];
		flist = (const char ** volatile) argv + optind + 1;
	}

#ifdef HAVE_GETPASS
	if (strcmp(fi.user, "anonymous") && strcmp(fi.user, "ftp")) {
		if ((fi.pass[0] == '\0') && (isatty(2) != 0)) {
			password = getpass("Password: ");		
			if (password != NULL) {
				STRNCPY(fi.pass, password);
				/* Don't leave cleartext password in memory. */
				memset(password, 0, strlen(fi.pass));
			}
		}
	}
#endif

	if (setjmp(gJmp) == 0) {
		signal(SIGINT, Abort);
		signal(SIGTERM, Abort);
		signal(SIGALRM, Abort);
		gCanJmp = 1;
		es = kExitOpenTimedOut;
		errstr = "could not open remote host";
		if ((result = FTPOpenHost(&fi)) < 0) {
			fprintf(stderr, "ncftpget: cannot open %s: %s.\n", fi.host, FTPStrError(result));
			es = kExitOpenFailed;
			exit(es);
		}
		if (flist == NULL) {
			/* URL mode */
			errstr = "could not change directory on remote host";
			es = kExitChdirTimedOut;
			for (lp = cdlist.first; lp != NULL; lp = lp->next) {
				if (FTPChdir(&fi, lp->line) != 0) {
					fprintf(stderr, "ncftpget: cannot chdir to %s: %s.\n", lp->line, FTPStrError(fi.errno));
					es = kExitChdirFailed;
					exit(es);
				}
			}

			errstr = "could not read file from remote host";
			es = kExitXferTimedOut;
			if (FTPGetFiles2(&fi, urlfile, ".", rflag, kGlobYes, xtype, resumeflag, appendflag) < 0) {
				fprintf(stderr, "ncftpget: file retrieval error: %s.\n", FTPStrError(fi.errno));
				es = kExitXferFailed;
			} else {
				es = kExitSuccess;
			}
		} else {
			errstr = "could not read file from remote host";
			es = kExitXferTimedOut;
			if (Copy(&fi, dstdir, flist, rflag, xtype, resumeflag, appendflag) < 0)
				es = kExitXferFailed;
			else
				es = kExitSuccess;
		}
	} else {
		signal(SIGALRM, SIG_IGN);
		signal(SIGINT, SIG_IGN);
		signal(SIGTERM, SIG_IGN);
		if (gGotSig == SIGALRM) {
			fprintf(stderr, "ncftpget: %s: timed-out.\n", errstr);
		} else {
			fprintf(stderr, "\nncftpget: caught signal, cleaning up...\n");
		}
	}

	if (setjmp(gJmp) == 0) {
		signal(SIGINT, Abort);
		signal(SIGTERM, Abort);
		signal(SIGALRM, Abort);
		gCanJmp = 1;
		errstr = "could not close remote host";
		FTPCloseHost(&fi);
	} else {
		/* couldn't close, but don't change exit status. */
		signal(SIGALRM, SIG_IGN);
		signal(SIGINT, SIG_IGN);
		signal(SIGTERM, SIG_IGN);
		if (gGotSig == SIGALRM) {
			fprintf(stderr, "ncftpget: %s: timed-out.\n", errstr);
		} else {
			fprintf(stderr, "\nncftpget: caught signal, cleaning up...\n");
		}
	}
	exit(es);
}	/* main */
