/* monkey.c
 *
 * The FTP Monkey can be used to simulate a FTP user.  It opens a host,
 * randomly changes directories and fetches files.
 */

#ifdef HAVE_CONFIG_H
#	include <Config.h>
#endif

#include <unistd.h>
#include <sys/types.h>
#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <signal.h>
#include <setjmp.h>

#include <ncftp.h>				/* Library header. */
#include <Strn.h>				/* Library header. */

#define kYesCloseYesQuit 0
#define kYesCloseNoQuit 1
#define kNoCloseNoQuit 2

jmp_buf gJmp;
LineList gDir;
int gHaveDir = 0;
int gQuitMode = kYesCloseYesQuit;
int gPrintLists = 0;
int gLoops = 0;
int gGotSig = 0;
int gMaxLoops = 0;
int gCloseAndReconnect = 0;
time_t gAlarmClock = 0;
char *gMainDir = "/pub";

#define kErrStream stdout

#define c_quit 0
#define c_cd 1
#define c_pwd 2
#define c_dir 3
#define c_get 4
#define c_max 5

int gWeights[c_max] = {
	10,	/* quit */
	20,	/* cd */
	5,	/* pwd */
	15,	/* dir */
	50,	/* get */
};

extern char *optarg;
extern int optind;

static void
Usage(void)
{
	fprintf(kErrStream, "FTP Monkey, copyright 1996 by Mike Gleason, NCEMRSoft.\n");
	fprintf(kErrStream, "Usage: ftp_monkey hostname\n");
	fprintf(kErrStream, "Library version: %s.\n", gLibNcFTPVersion + 5);
#ifdef UNAME
	fprintf(kErrStream, "System: %s.\n", UNAME);
#endif
	exit(2);
}	/* Usage */


static int
Ri(int a)
{
	int b;

	b = rand() % a;
	return (b);
}	/* Ri */


static int
Rp(int a)
{
	int b;

	b = rand() % 100;
	if (b < a)
		return (1);
	return (0);
}	/* Rp */



static int
FTPReadOneFile(const FTPCIPtr cip, const char *file)
{
	char *buf;
	size_t bufSize;
	int tmpResult, result;
	int nread;

	if (cip == NULL)
		return (kErrBadParameter);
	if (strcmp(cip->magic, kLibraryMagic))
		return (kErrBadMagic);
	
	if (file == NULL)
		return (kErrBadParameter);
	if (file[0] == '\0')
		return (kErrBadParameter);

	result = kNoErr;
	FTPStartIOTimer(cip);
	tmpResult = FTPStartDataCmd(cip, kNetReading, kTypeBinary, 0, "RETR %s", file);
	if (tmpResult < 0) {
		result = tmpResult;
		if (result == kErrGeneric)
			result = kErrRETRFailed;
		if ((cip->errno == kErrAcceptDataSocket) || (cip->errno == kErrNewStreamSocket))
			FTPCloseHost(cip);
		cip->errno = result;
		return (result);
	}

	buf = cip->buf;
	bufSize = cip->bufSize;
	while (1) {
		nread = read(cip->dataSocket, buf, bufSize);
		if (nread < 0) {
			/* Error(cip, kDoPerror, "Remote read failed.\n"); */
			result = kErrSocketReadFailed;
			cip->errno = result;
			break;
		} else if (nread == 0) {
			break;
		}
		cip->bytesTransferred += (long) nread;
	}

	tmpResult = FTPEndDataCmd(cip, 1);
	if ((tmpResult < 0) && (result == 0)) {
		result = kErrRETRFailed;
		if ((cip->errno == kErrAcceptDataSocket) || (cip->errno == kErrNewStreamSocket))
			FTPCloseHost(cip);
		cip->errno = result;
	}

	FTPStopIOTimer(cip);
	return (result);
}	/* FTPReadOneFile */



static int
FileLIST(const FTPCIPtr cip, LineListPtr lines, const char *lsflags)
{
	int result, nread;
	char secondaryBuf[512];
	char *secBufPtr, *secBufLimit;
	char line[128];
	char *tok;
	int ftype;
	char fname[128];

	InitLineList(lines);

	FTPStartIOTimer(cip);
	result = FTPStartDataCmd(cip, kNetReading, kTypeAscii, 0, "%s%s", "LIST", lsflags);
	if (result == 0) {
		/* This line sets the buffer pointer so that the first thing
		 * BufferGets will do is reset and fill the buffer using
		 * real I/O.
		 */
		secBufPtr = secondaryBuf + sizeof(secondaryBuf);
		secBufLimit = (char *) 0;

		while (1) {
skip:
			nread = BufferGets(line, sizeof(line), cip->dataSocket, secondaryBuf, &secBufPtr, &secBufLimit, sizeof(secondaryBuf));
			if (nread <= 0) {
				if (nread < 0)
					break;
			} else {
				cip->bytesTransferred += (long) nread;
				ftype = line[0];
				if ((ftype == '-') || (ftype == 'd')) {
					tok = line + strlen(line);
					while (*--tok != ' ') {
						if (tok < line)
							goto skip;
					}
					fname[0] = ftype;
					fname[1] = ' ';
					fname[2] = '\0';
					STRNCAT(fname, tok + 1);
					AddLine(lines, fname);
				}
			}
		}
		result = FTPEndDataCmd(cip, 1);
		FTPStopIOTimer(cip);
		if (result < 0) {
			result = kErrLISTFailed;
			cip->errno = result;
		}
		result = kNoErr;
	} else if (result == kErrGeneric) {
		result = kErrLISTFailed;
		if ((cip->errno == kErrAcceptDataSocket) || (cip->errno == kErrNewStreamSocket))
			FTPCloseHost(cip);
		cip->errno = result;
	}
	return (result);
}	/* FileLIST */



static int
FileNLST(const FTPCIPtr cip, LineListPtr lines, const char *lsflags)
{
	int result, nread;
	char secondaryBuf[512];
	char *secBufPtr, *secBufLimit;
	char line[128];
	char *cp1, *cp2, *tok;
	int ftype;
	char fname[128];

	InitLineList(lines);

	FTPStartIOTimer(cip);
	result = FTPStartDataCmd(cip, kNetReading, kTypeAscii, 0, "%s%s", "NLST", lsflags);
	if (result == 0) {
		/* This line sets the buffer pointer so that the first thing
		 * BufferGets will do is reset and fill the buffer using
		 * real I/O.
		 */
		secBufPtr = secondaryBuf + sizeof(secondaryBuf);
		secBufLimit = (char *) 0;

		while (1) {
			nread = BufferGets(line, sizeof(line), cip->dataSocket, secondaryBuf, &secBufPtr, &secBufLimit, sizeof(secondaryBuf));
			if (nread <= 0) {
				if (nread < 0)
					break;
			} else {
				cip->bytesTransferred += (long) nread;
				/* Parse files out of possibly
				 * multicolumn output.
				 */
				for (cp1 = line; ; cp1 = NULL) {
					tok = strtok(cp1, " \r\n\t");
					if (tok == NULL)
						break;
					cp2 = tok + strlen(tok) - 1;
					switch (*cp2) {
						case '*':
						case '@':
							*cp2 = '\0';
							ftype = '-';
							break;
						case '/':
							*cp2 = '\0';
							ftype = 'd';
							break;
						default:
							ftype = '-';
							break;
					}
					fname[0] = ftype;
					fname[1] = ' ';
					fname[2] = '\0';
					STRNCAT(fname, tok);
					AddLine(lines, fname);
				}
			}
		}
		result = FTPEndDataCmd(cip, 1);
		FTPStopIOTimer(cip);
		if (result < 0) {
			result = kErrLISTFailed;
			cip->errno = result;
		}
		result = kNoErr;
	} else if (result == kErrGeneric) {
		result = kErrLISTFailed;
		if ((cip->errno == kErrAcceptDataSocket) || (cip->errno == kErrNewStreamSocket))
			FTPCloseHost(cip);
		cip->errno = result;
	}
	return (result);
}	/* FileNLST */




static double
Duration(struct timeval *t0)
{
	struct timeval t1;
	double sec;

	gettimeofday(&t1, NULL);
	if (t0->tv_usec > t1.tv_usec) {
		t1.tv_usec += 1000000;
		t1.tv_sec--;
	}
	sec = ((double) (t1.tv_usec - t0->tv_usec) * 0.000001)
		+ (t1.tv_sec - t0->tv_sec);

	return (sec);
}	/* Duration */



static int
FileList(const FTPCIPtr cip)
{
	int p, rc;
	LinePtr lp;
	struct timeval t0;
	double dura;

	if (gHaveDir == 1) {
		DisposeLineListContents(&gDir);
		gHaveDir = 0;
	}
	p = Ri(3);
	if (p == 0) {
		printf("%-5d ls -F", gLoops);
		fflush(stdout);
		gettimeofday(&t0, NULL);
		rc = FileNLST(cip, &gDir, " -F");
	} else if (p == 1) {
		printf("%-5d dir", gLoops);
		fflush(stdout);
		gettimeofday(&t0, NULL);
		rc = FileLIST(cip, &gDir, "");
	} else {
		printf("%-5d ls -CF", gLoops);
		fflush(stdout);
		gettimeofday(&t0, NULL);
		rc = FileNLST(cip, &gDir, " -CF");
	}
	if (rc >= 0) {
		gHaveDir = 1;
		dura = Duration(&t0);
		printf("\t%.3f sec\t%ld bytes %.3f sec %.3f kB/sec\n", dura, (long) cip->bytesTransferred, cip->sec, cip->kBytesPerSec);
		if (gPrintLists != 0) {
			for (lp=gDir.first; lp != NULL; lp=lp->next)
				printf("    %s\n", lp->line);
		}
	}
	return (rc);
}	/* FileList */




static char *
RandomFile(void)
{
	LinePtr lp;
	int nfiles, pick, i;

	if (gHaveDir == 0)
		return NULL;

	for (lp=gDir.first, nfiles=0; lp != NULL; lp=lp->next)
		if (lp->line[0] == '-')
			nfiles++;
	if (nfiles == 0)
		return NULL;

	pick = Ri(nfiles);
	for (lp=gDir.first, i=0; lp != NULL; lp=lp->next) {
		if (lp->line[0] == '-') {
			if (pick == i)
				return (lp->line + 2);
			i++;
		}
	}
	return NULL;
}	/* RandomFile */




static char *
RandomDir(void)
{
	LinePtr lp;
	int ndirs, pick, i;

	if (gHaveDir == 0)
		return NULL;

	for (lp=gDir.first, ndirs=0; lp != NULL; lp=lp->next)
		if (lp->line[0] == 'd')
			ndirs++;
	if (ndirs == 0)
		return NULL;

	pick = Ri(ndirs);
	for (lp=gDir.first, i=0; lp != NULL; lp=lp->next) {
		if (lp->line[0] == 'd') {
			if (pick == i)
				return (lp->line + 2);
			i++;
		}
	}
	return NULL;
}	/* RandomDir */



static void
Shell(const FTPCIPtr cip)
{
	int i, maxw, needlist = 1;
	int p;
	char *randdir, *randfile;
	char s1[128];
	struct timeval t0;
	double dura;
	time_t now;

	for (i=1; i<c_max; i++) {
		gWeights[i] += gWeights[i-1];
	}
	maxw = gWeights[i-1];
	for (gLoops = 0; ((gMaxLoops < 1) || (gLoops < gMaxLoops)); gLoops++) {
		time(&now);
		if ((gAlarmClock != 0) && (now >= gAlarmClock)) {
			printf("%-5d time up!\n", gLoops);
			return;
		}
		p = Ri(maxw);
		for (i=0; i<c_max; i++) {
			if (p < gWeights[i])
				break;
		}

		/* Lost connection? */
		if (cip->connected == 0)
			i = c_quit;

		switch (i) {
			case c_quit:
				if (gQuitMode == kYesCloseYesQuit) {
					printf("%-5d quit\n", gLoops);
					return;
				} else if (gQuitMode == kYesCloseNoQuit) {
					printf("%-5d close", gLoops);
					fflush(stdout);
					gettimeofday(&t0, NULL);
					if ((FTPCloseHost(cip)) < 0) {
						dura = Duration(&t0);
						printf("\t%.3f sec\n", dura);
						fflush(stdout);
						fprintf(kErrStream, "Cannot close host.\n");
						exit(5);
					}
					dura = Duration(&t0);
					printf("\t%.3f sec\n", dura);
					sleep(Ri(3) + 3);
					needlist = 1;
					printf("\n%-5d open", ++gLoops);
					fflush(stdout);
					gettimeofday(&t0, NULL);
					if ((FTPOpenHost(cip)) < 0) {
						dura = Duration(&t0);
						printf("\t%.3f sec\n", dura);
						fflush(stdout);
						fprintf(kErrStream, "Cannot open host.\n");
						exit(5);
					}
					dura = Duration(&t0);
					printf("\t%.3f sec\n", dura);
				} /* else kNoCloseNoQuit */
				break;
			case c_cd:
				if (Rp(20)) {
					printf("%-5d cd ..", gLoops);
					fflush(stdout);
					gettimeofday(&t0, NULL);
					(void) FTPChdir(cip, "..");
					dura = Duration(&t0);
					printf("\t%.3f sec\n", dura);
					needlist = 1;
				} else if (Rp(20)) {
					printf("%-5d cd %s", gLoops, gMainDir);
					fflush(stdout);
					gettimeofday(&t0, NULL);
					(void) FTPChdir(cip, gMainDir);
					dura = Duration(&t0);
					printf("\t%.3f sec\n", dura);
					needlist = 1;
				} else {
cd:
					if (needlist) {
						if (FileList(cip) >= 0)
							needlist = 0;
						++gLoops;
					}
					if ((needlist == 0) && ((randdir = RandomDir()) != NULL)) {
						printf("%-5d cd %s", gLoops, randdir);
						fflush(stdout);
						gettimeofday(&t0, NULL);
						if (FTPChdir(cip, randdir) == 0)
							needlist = 1;
						dura = Duration(&t0);
						printf("\t%.3f sec\n", dura);
					} else {
						printf("%-5d cd %s", gLoops, gMainDir);
						fflush(stdout);
						gettimeofday(&t0, NULL);
						(void) FTPChdir(cip, gMainDir);
						dura = Duration(&t0);
						printf("\t%.3f sec\n", dura);
						needlist = 1;
					}
				}
				break;
			case c_pwd:
				printf("%-5d pwd", gLoops);
				fflush(stdout);
				gettimeofday(&t0, NULL);
				(void) FTPGetCWD(cip, s1, sizeof(s1));	/* pwd */
				dura = Duration(&t0);
				printf("\t%.3f sec\n", dura);
				break;
			case c_dir:
				if (FileList(cip) >= 0)
					needlist = 0;
				break;
			case c_get:
				if (needlist) {
					if (FileList(cip) >= 0)
						needlist = 0;
					++gLoops;
				}
				if (needlist == 0) {
					randfile = RandomFile();
					if (randfile == NULL)
						goto cd;
					/* pick a random file to get */
					printf("%-5d get %s", gLoops, randfile);
					fflush(stdout);
					gettimeofday(&t0, NULL);
					(void) FTPReadOneFile(cip, randfile);
					dura = Duration(&t0);
					printf("\t%.3f sec\t%ld bytes %.3f sec %.3f kB/sec\n", dura, (long) cip->bytesTransferred, cip->sec, cip->kBytesPerSec);
				}
				break;
		}
		sleep(Ri(4));
	}
}	/* Shell */



static void
Abort(int sigNum)
{
	static int onceOnly = 0;

	if (onceOnly == 0) {
		++onceOnly;
		gGotSig = sigNum;
		longjmp(gJmp, 1);
	}
}	/* Abort */



void
main(int argc, char **argv)
{
	int result, c;
	FTPLibraryInfo li;
	FTPConnectionInfo fi;
	int seed;
	time_t now;
	struct tm *ltp;
	char dstr[64];
#ifdef HAVE_GETPASS
	char *password;
#endif

	result = FTPInitLibrary(&li);
	if (result < 0) {
		fprintf(kErrStream, "Init library error %d.\n", result);
		exit(3);
	}
	result = FTPInitConnectionInfo(&li, &fi, kDefaultFTPBufSize);
	if (result < 0) {
		fprintf(kErrStream, "Init connection info error %d.\n", result);
		exit(4);
	}

	fi.debugLog = NULL;
	fi.errLog = kErrStream;
	STRNCPY(fi.user, "anonymous");
	STRNCPY(fi.pass, "monkey@redwing.probe.net");
	seed = -1;

	while ((c = getopt(argc, argv, "a:QRD:m:ls:d:e:P:p:u:")) > 0) switch(c) {
		case 'a':
			time(&gAlarmClock);
			gAlarmClock += (time_t) atoi(optarg);
			break;
		case 'm':
			gMaxLoops = atoi(optarg);
			break;
		case 's':
			seed = atoi(optarg);
			break;
		case 'D':
			gMainDir = optarg;
			break;
		case 'l':
			gPrintLists = !gPrintLists;
			break;
		case 'P':
			fi.port = atoi(optarg);	
			break;
		case 'u':
			STRNCPY(fi.user, optarg);
			break;
		case 'Q':
			gQuitMode = kNoCloseNoQuit;
			break;
		case 'R':
			gQuitMode = kYesCloseNoQuit;
			break;
		case 'p':
			STRNCPY(fi.pass, optarg);	/* Don't recommend doing this! */
			break;
		case 'e':
			fi.errLog = fopen(optarg, "a");
			break;
		case 'd':
			fi.debugLog = fopen(optarg, "a");
			break;
		default:
			Usage();
	}
	if (optind > argc - 1)
		Usage();

#ifdef HAVE_GETPASS
	if (strcmp(fi.user, "anonymous") && strcmp(fi.user, "ftp")) {
		if (fi.pass[0] == '\0') {
			password = getpass("Password: ");		
			if (password != NULL) {
				STRNCPY(fi.pass, password);
				/* Don't leave cleartext password in memory. */
				memset(password, 0, strlen(fi.pass));
			}
		}
	}
#endif

	if (seed < 0) {
		seed = time(NULL) & 0x7fff;
		fprintf(stdout, "Seed is %d.\n", seed);
	}
	srand((unsigned int) seed);

	STRNCPY(fi.host, argv[optind]);

	if (setjmp(gJmp) == 0) {
		signal(SIGINT, Abort);
		signal(SIGTERM, Abort);
		if ((result = FTPOpenHost(&fi)) < 0) {
			fprintf(kErrStream, "Cannot open host.  Err %d.\n", result);
			exit(5);
		}
		time(&now);
		ltp = localtime(&now);
		strftime(dstr, sizeof(dstr), "%m/%y %H:%M:%S", ltp);
		printf("-- Started at %s --\n", dstr);
		Shell(&fi);
	} else {
		signal(SIGINT, SIG_IGN);
		signal(SIGTERM, SIG_IGN);
		fprintf(kErrStream, "\nCaught signal %d, cleaning up...\n", gGotSig);
	}

	if (setjmp(gJmp) == 0) {
		signal(SIGINT, Abort);
		signal(SIGTERM, Abort);
		FTPCloseHost(&fi);
	} else {
		signal(SIGINT, SIG_IGN);
		signal(SIGTERM, SIG_IGN);
		fprintf(kErrStream, "\nCaught signal %d, cleaning up...\n", gGotSig);
	}

	time(&now);
	ltp = localtime(&now);
	strftime(dstr, sizeof(dstr), "%m/%y %H:%M:%S", ltp);
	printf("-- Finished at %s --\n", dstr);

	exit(0);
}	/* main */
