/* gpshare.c
 *
 * Shared routines for ncftpget and ncftpput.
 */

#ifdef HAVE_CONFIG_H
#	include <Config.h>
#endif

#include <unistd.h>
#include <sys/types.h>
#include <stdio.h>
#include <string.h>
#include <ctype.h>
#include <stdlib.h>

#include <ncftp.h>				/* Library header. */
#include <Strn.h>				/* Library header. */
#include "gpshare.h"

/* This will abbreviate a string so that it fits into max characters.
 * It will use ellipses as appropriate.  Make sure the string has
 * at least max + 1 characters allocated for it.
 */
void
AbbrevStr(char *dst, const char *src, size_t max, int mode)
{
	int len;

	len = (int) strlen(src);
	if (len > (int) max) {
		if (mode == 0) {
			/* ...Put ellipses at left */
			strcpy(dst, "...");
			Strncat(dst, (char *) src + len - (int) max + 3, max + 1);
		} else {
			/* Put ellipses at right... */
			Strncpy(dst, (char *) src, max + 1);
			strcpy(dst + max - 3, "...");
		}
	} else {
		Strncpy(dst, (char *) src, max + 1);
	}
}	/* AbbrevStr */



double
FileSize(double size, char **uStr0, double *uMult0)
{
	double uMult, uTotal;
	char *uStr;

	if (size > kGigabyte) {
		uStr = "GB";
		uMult = kGigabyte;
	} else if (size > kMegabyte) {
		uStr = "MB";
		uMult = kMegabyte;
	} else if (size > kKilobyte) {
		uStr = "kB";
		uMult = 1024;
	} else {
		uStr = "B";
		uMult = 1;
	}
	if (uStr0 != NULL)
		*uStr0 = uStr;
	if (uMult0 != NULL)
		*uMult0 = uMult;
	uTotal = size / ((double) uMult);
	return (uTotal);
}	/* FileSize */




void
PrSizeAndRateMeter(const FTPCIPtr cip, int mode)
{
	double rate;
	char *rStr;
	static char *uStr;
	static double uTotal, uMult;
	char localName[32];
	char line[128];
	int i;

	switch (mode) {
		case kPrInitMsg:
			if (cip->expectedSize != kSizeUnknown) {
				cip->progress = PrStatBar;
				PrStatBar(cip, mode);
				return;
			}
			uTotal = FileSize((double) cip->expectedSize, &uStr, &uMult);

			if (cip->lname == NULL) {
				localName[0] = '\0';
			} else {
				AbbrevStr(localName, cip->lname, sizeof(localName) - 2, 0);
				STRNCAT(localName, ":");
			}

			fprintf(stderr, "%-32s", localName);
			break;

		case kPrUpdateMsg:
			rate = FileSize(cip->kBytesPerSec * 1024.0, &rStr, NULL);

			if (cip->lname == NULL) {
				localName[0] = '\0';
			} else {
				AbbrevStr(localName, cip->lname, sizeof(localName) - 2, 0);
				STRNCAT(localName, ":");
			}

			sprintf(line, "%-32s  %10lu bytes  %5.1f %s/s",
				localName,
				(unsigned long) (cip->bytesTransferred + cip->startPoint),
				rate,
				rStr
			);

			/* Pad the rest of the line with spaces, to erase any
			 * stuff that might have been left over from the last
			 * update.
			 */
			for (i=strlen(line); i < 80 - 2; i++)
				line[i] = ' ';
			line[i] = '\0';

			/* Print the updated information. */
			fprintf(stderr, "\r%s", line);
			break;

		case kPrEndMsg:
			fprintf(stderr, "\n\r");
			break;
	}
}	/* PrSizeAndRateMeter */




void
PrStatBar(const FTPCIPtr cip, int mode)
{
	double rate, done;
	int secLeft, minLeft;
	char *rStr;
	static char *uStr;
	static double uTotal, uMult;
	char localName[32];
	char line[128];
	int i;

	switch (mode) {
		case kPrInitMsg:
			if (cip->expectedSize == kSizeUnknown) {
				cip->progress = PrSizeAndRateMeter;
				PrSizeAndRateMeter(cip, mode);
				return;
			}
			uTotal = FileSize((double) cip->expectedSize, &uStr, &uMult);

			if (cip->lname == NULL) {
				localName[0] = '\0';
			} else {
				/* Leave room for a ':' and '\0'. */
				AbbrevStr(localName, cip->lname, sizeof(localName) - 2, 0);
				STRNCAT(localName, ":");
			}
			fprintf(stderr, "%-32s", localName);
			break;

		case kPrUpdateMsg:
			secLeft = (int) (cip->secLeft + 0.5);
			minLeft = secLeft / 60;
			secLeft = secLeft - (minLeft * 60);
			if (minLeft > 999) {
				minLeft = 999;
				secLeft = 59;
			}

			rate = FileSize(cip->kBytesPerSec * 1024.0, &rStr, NULL);
			done = (double) (cip->bytesTransferred + cip->startPoint) / uMult;

			if (cip->lname == NULL) {
				localName[0] = '\0';
			} else {
				AbbrevStr(localName, cip->lname, sizeof(localName) - 2, 0);
				STRNCAT(localName, ":");
			}

			sprintf(line, "%-32s   %5.1f/%.1f %s   %5.1f %s/s   ETA %3d:%02d",
				localName,
				done,
				uTotal,
				uStr,
				rate,
				rStr,
				minLeft,
				secLeft
			);

			/* Pad the rest of the line with spaces, to erase any
			 * stuff that might have been left over from the last
			 * update.
			 */
			for (i=strlen(line); i < 80 - 2; i++)
				line[i] = ' ';
			line[i] = '\0';

			/* Print the updated information. */
			fprintf(stderr, "\r%s", line);
			break;

		case kPrEndMsg:
			rate = FileSize(cip->kBytesPerSec * 1024.0, &rStr, NULL);
			done = (double) (cip->bytesTransferred + cip->startPoint) / uMult;

			if (cip->lname == NULL) {
				localName[0] = '\0';
			} else {
				AbbrevStr(localName, cip->lname, sizeof(localName) - 2, 0);
				STRNCAT(localName, ":");
			}

			sprintf(line, "%-32s   %5.1f/%.1f %s   %5.1f %s/s",
				localName,
				done,
				uTotal,
				uStr,
				rate,
				rStr
			);

			/* Pad the rest of the line with spaces, to erase any
			 * stuff that might have been left over from the last
			 * update.
			 */
			for (i=strlen(line); i < 80 - 2; i++)
				line[i] = ' ';
			line[i] = '\0';

			/* Print the updated information. */
			fprintf(stderr, "\r%s\n\r", line);
			break;
	}
}	/* PrStatBar */




void
ReadConfigFile(const char *fn, FTPCIPtr cip)
{
	FILE *fp;
	char line[128];
	char *cp;
	int goodfile = 0;

	fp = fopen(fn, "r");
	if (fp == NULL) {
		perror(fn);
		exit(kExitBadConfigFile);
	}

	line[sizeof(line) - 1] = '\0';
	while (fgets(line, sizeof(line) - 1, fp) != NULL) {
		if ((line[0] == '#') || (isspace(line[0])))
			continue;
		cp = line + strlen(line) - 1;
		if (*cp == '\n')
			*cp = '\0';
		if (strncmp(line, "user", 4) == 0) {
			STRNCPY(cip->user, line + 5);
			goodfile = 1;
		} else if (strncmp(line, "password", 8) == 0) {
			STRNCPY(cip->pass, line + 9);
			goodfile = 1;
		} else if ((strncmp(line, "pass", 4) == 0) && (isspace(line[4]))) {
			STRNCPY(cip->pass, line + 5);
			goodfile = 1;
		} else if (strncmp(line, "host", 4) == 0) {
			STRNCPY(cip->host, line + 5);
			goodfile = 1;
		}
	}
	fclose(fp);

	if (goodfile == 0) {
		fprintf(stderr, "%s doesn't contain anything useful.\n", fn);
		fprintf(stderr, "%s should look something like this:\n", fn);
		fprintf(stderr, "# Comment lines starting with a hash character\n# and blank lines are ignored.\n\n");
		fprintf(stderr, "host Bozo.probe.net\n");
		fprintf(stderr, "user gleason\n");
		fprintf(stderr, "pass mypasswd\n");
		exit(kExitBadConfigFile);
	}
}	/* ReadConfigFile */

/* eof */
