#ifdef	LOGIN_ACCESS

 /*
  * This module implements a simple but effective form of login access
  * control based on login names and on host (or domain) names, internet
  * addresses (or network numbers), or on terminal line names in case of
  * non-networked logins. Diagnostics are reported through syslog(3).
  * 
  * Author: Wietse Venema, Eindhoven University of Technology, The Netherlands.
  */

#ifndef lint
static char sccsid[] = "%Z% %M% %I% %E% %U%";

#endif

#include <stdio.h>
#include <syslog.h>
#include <ctype.h>

extern char *fgets();
extern char *strchr();
extern char *strtok();
extern void exit();

 /* Path name of the access control file. */

#ifndef	TABLE
#define TABLE	"/etc/login.access"
#endif

 /* Delimiters for fields and for lists of users, ttys or hosts. */

static char fs[] = ":";			/* field separator */
static char sep[] = ", \t";		/* list-element separator */

 /* Constants to be used in assignments only, not in comparisons... */

#define YES             1
#define NO              0

/* login_access - match username and (host or tty) with access control file */

login_access(user, from)
char   *user;
char   *from;
{
    FILE   *fp;
    char    line[BUFSIZ];
    char   *perm;			/* becomes permission field */
    char   *users;			/* becomes list of login names */
    char   *froms;			/* becomes list of terminals or hosts */
    int     match = NO;
    int     end;
    int     lineno = 0;			/* for diagnostics */

    /*
     * Process the table one line at a time and stop at the first match.
     * Lines that begin with a '#' character are ignored. Non-comment lines
     * are broken at the ':' character. All fields are mandatory. The first
     * field should be a "+" or "-" character. A non-existing table means no
     * access control.
     */

    if (fp = fopen(TABLE, "r")) {
	while (!match && fgets(line, sizeof(line), fp)) {
	    lineno++;
	    if (line[end = strlen(line) - 1] != '\n')
		syslog(LOG_ERR, "%s: line %d: exceeds STDIO buffer size",
		       TABLE, lineno);
	    else
		line[end] = 0;			/* strip trailing newline */
	    if (line[0] == '#')
		continue;			/* comment line */
	    if (!(perm = strtok(line, fs))
		|| !(users = strtok((char *) 0, fs))
		|| !(froms = strtok((char *) 0, fs))
		|| strtok((char *) 0, fs)) {
		syslog(LOG_ERR, "%s: line %d: bad field count", TABLE, lineno);
		continue;
	    }
	    if (perm[0] != '+' && perm[0] != '-') {
		syslog(LOG_ERR, "%s: line %d: bad first field", TABLE, lineno);
		continue;
	    }
	    match = (list_match(users, user) && list_match(froms, from));
	}
	(void) fclose(fp);
    }
    return (match == 0 || (line[0] == '+'));
}

/* list_match - match a string against a list of tokens */

list_match(list, string)
char   *list;
char   *string;
{
    char   *tok;
    int     tok_len;
    int     str_len;

    /*
     * Process tokens one at a time. If a token has the magic value "ALL" the
     * match always succeeds. If the token is a domain name, return YES if it
     * matches the last fields of the string. If the token has the magic
     * value "LOCAL", return YES if the string does not contain a "."
     * character. If the token is a network number, return YES if it matches
     * the head of the string. Otherwise, return YES if the token fully
     * matches the string. Note: we assume that a daemon process name never
     * begins or ends with a "." character.
     */

    for (tok = strtok(list, sep); tok; tok = strtok((char *) 0, sep)) {
	if (strcasecmp(tok, "ALL") == 0) {	/* all: always matches */
	    return (YES);
	} else if (tok[0] == '.') {		/* domain: match last fields */
	    if ((str_len = strlen(string)) > (tok_len = strlen(tok))
		&& strcasecmp(tok, string + str_len - tok_len) == 0)
		return (YES);
	} else if (strcasecmp(tok, "LOCAL") == 0) {	/* local: no dots */
	    if (strchr(string, '.') == 0)
		return (YES);
	} else if (!strcasecmp(tok, string)) {	/* match host name or address */
	    return (YES);
	} else if (tok[(tok_len = strlen(tok)) - 1] == '.'	/* network */
		   && strncmp(tok, string, tok_len) == 0) {
	    return (YES);
	}
    }
    return (NO);
}

#endif
