/* Copyright (c) 1992 The Geometry Center; University of Minnesota
   1300 South Second Street;  Minneapolis, MN  55454, USA;
   
This file is part of geomview/OOGL. geomview/OOGL is free software;
you can redistribute it and/or modify it only under the terms given in
the file COPYING, which you should have received along with this file.
This and other related software may be obtained via anonymous ftp from
geom.umn.edu; email: software@geom.umn.edu. */

/* Authors: Stuart Levy, Tamara Munzner, Mark Phillips */

#include "ooglutil.h"
#include "windowP.h"
#include "streampool.h"

int WnStreamIn( Pool *, Handle **, WnWindow ** );
int WnStreamOut( Pool *, Handle *, WnWindow * );

HandleOps WindowOps = {
	"window",
	WnStreamIn,
	WnStreamOut,
	WnDelete,
	NULL,
	NULL
};

static struct winkeyword {
	char *kw;
	int flag;
} wn_kw[] = {
	{ "window", 0 },
	{ "define", 0 },
	{ "size", WNF_HASSIZE },
	{ "position", WNF_HASPREF },
	{ "noborder", WNF_NOBORDER },
	{ "pixelaspect", 0 },
	{ "resize", WNF_ENLARGE|WNF_SHRINK },
	{ "curpos", WNF_HASCUR },
	{ "border", WNF_NOBORDER },
};

int
WnStreamIn( Pool *p, Handle **hp, WnWindow **wp )
{
    Handle *h = NULL;
    Handle *hname = NULL;
    register WnWindow *win = NULL;
    FILE *inf;
    char *w, *err;
    int i, c;
    int more = 0;
    int brack = 0;
    static char poserr[] = "xmin xmax ymin ymax";

    if(p == NULL || (inf = PoolInputFile(p)) == NULL)
	return 0;

    win = WnCreate(WN_END);

    do {
	more = 0;
	switch(c = fnextc(inf, 0)) {
	case '{': brack++; fgetc(inf); break;
	case '}': if(brack--) fgetc(inf); break;

	case '<':
	case ':':
	case '@':
	    fgetc(inf);
	    w = fdelimtok("{}()", inf, 0);
	    if(c == '<' && HandleByName(w, &WindowOps) == NULL)
		w = findfile(PoolName(p), 0);
	    h = HandleReferringTo(c, w, &WindowOps, NULL);
	    if(h)
		win = (WnWindow *)HandleObject(h);
	    break;

	default:
	    w = fdelimtok("{}()", inf, 0);
	    if(w == NULL) goto error;
	    for(i = sizeof(wn_kw)/sizeof(wn_kw[0]); strcmp(w, wn_kw[--i].kw); )
		if(i == 0) {
		    OOGLError(1,
			"%s: unknown window keyword %s", PoolName(p), w);
		    goto error;
		}

	    win->flag |= wn_kw[i].flag;
	    win->changed |= wn_kw[i].flag;
	    switch(i) {
	    case 0: more = 1; break;		/* window */
	    case 1:				/* define */
		hname = HandleAssign(ftoken(inf, 0), &WindowOps, NULL);
		break;
	    case 2:				/* size */
		err = "xsize ysize";
		if(fgetni(inf, 1, (int *)&win->xsize, 0) <= 0 ||
		   fgetni(inf, 1, (int *)&win->ysize, 0) <= 0)
		    goto expect;
		break;
	    case 3:				/* position */
		err = poserr;
		if(fgetni(inf, 4, (int *)&win->pref, 0) != 4)
		    goto expect;
		break;

		/* "noborder" and "resize" are just flag manipulation */

	    case 5:				/* aspect */
		err = "pixel-aspect-ratio(X/Y)";
		if(fgetnf(inf, 1, &win->aspect, 0) <= 0)
		    goto expect;
		break;
	    case 7:				/* curpos */
		err = poserr;
		if(fgetni(inf, 4, (int *)&win->cur, 0) != 4)
		    goto expect;
		break;
	    case 8:				/* border */
		win->flag &= ~wn_kw[i].flag;
		break;
	    }
	    break;

	
	 expect:
	    OOGLError(1, "window: %s expects %s", w, err);
	 error:
	    if(win && !h)
		WnDelete(win);
	    return NULL;
	}
    } while(brack || more);

    if(hname != NULL) {
	if(win)
	    HandleSetObject(hname, (Ref *)win);
	h = hname;
    }


    if(h != NULL && hp != NULL && *hp != h) {
	if(*hp != NULL)
	    HandlePDelete(hp);
	*hp = h;
    }

    if(win != NULL && wp != NULL)
	*wp = win;

    if(h != NULL && win != NULL)
	HandleSetObject(h, (Ref *)win);

    return (h != NULL || win != NULL);
}

int
WnStreamOut( Pool *p, Handle *h, WnWindow *win )
{
    int i;
    register WnPosition *wp;
    FILE *f;

    if((f = PoolOutputFile(p)) == NULL)
	return 0;

    fprintf(f, "window {");

    if(PoolStreamOutHandle( p, h, win != NULL)) {
	for(i = 2; i < sizeof(wn_kw)/sizeof(wn_kw[0]); i++) {
	    if(win->flag & wn_kw[i].flag) {
		fprintf(f, " %s", wn_kw[i].kw);
		switch(i) {
		case 2: fprintf(f, " %d %d", win->xsize, win->ysize);
			break;
		case 3: wp = &win->pref; goto dowp;
		case 7: wp = &win->cur;
		    dowp:
			fprintf(f, " %d %d %d %d",
				wp->xmin, wp->xmax, wp->ymin, wp->ymax);
			break;
		}
	    }
	}
	if(win->pixaspect != 1)
	    fprintf(f, " pixelaspect %g", win->pixaspect);
    }
    fputs(" }\n", f);
}
