/* Copyright (c) 1992 The Geometry Center; University of Minnesota
   1300 South Second Street;  Minneapolis, MN  55454, USA;
   
This file is part of geomview/OOGL. geomview/OOGL is free software;
you can redistribute it and/or modify it only under the terms given in
the file COPYING, which you should have received along with this file.
This and other related software may be obtained via anonymous ftp from
geom.umn.edu; email: software@geom.umn.edu. */

/* Authors: Charlie Gunn, Stuart Levy, Tamara Munzner, Mark Phillips */

#include <string.h>
#include <sys/stat.h>
#include <fcntl.h>
#include "handleP.h"

static Pool *AllPools = NULL;

static fd_set poolwatchfds;
static int poolmaxfd = 0;
static fd_set poolreadyfds;
static int poolnready = 0;


/*
 * Does the whole job of handling stream references to named objects.
 * Interprets strings of the forms:
 *	name:file
 *		referring to a specific named object to be read from a
 *		specific file, or
 *	file
 *		referring to the nameless content of that file, or
 *	name:
 *		referring to a specific named object from any source.
 *
 * In the first two cases, we open (as a Pool) the specified file and
 * attempt to read it using the functions in *ops.
 */
Handle *
HandleReferringTo(int prefixch, char *str, HandleOps *ops, Handle **hp)
{
    Pool *p = NULL;
    Handle *h = NULL;
    Handle *hknown;
    char *sep;
    char *fname;
    char *name;
    char nb[128];

    if(str == NULL || ops == NULL)
	return 0;

    sep = strrchr(str, ':');
    if(prefixch == ':') {	/*   :  name   -- take 'name' from anywhere */
	name = str;
	fname = NULL;
    } else if(sep == NULL) {	/*   <  file   -- read from file 'name' */
	fname = str;
	name = str;
    } else {			/*   <  file:name */
	name = sep+1;
	fname = nb;
	if(sep-str >= sizeof(nb))
	    sep = &str[sizeof(nb)-1];
	memcpy(fname, str, sep-str);
	fname[sep-str] = '\0';
    }
    h = hknown = HandleByName(name, ops);

    if(fname != NULL)
	p = PoolStreamOpen(fname, NULL, 0, ops);

    if((p || prefixch == ':') && h == NULL) {
	h = HandleAssign(name, ops, NULL);
	if(h == NULL) {
	    OOGLError(1, "Can't make handle for '%s'", str);
	    return NULL;
	}
    }
    if(p && (((p->flags & (PF_ANY|PF_REREAD)) != PF_ANY) || hknown != NULL)) {
	Handle *th = PoolIn(p);	/* Try reading one item. */
	if(th) {		/* Read anything? */
	   h = th;		/* Return that if so */
	   h->whence = p;
	}
    }
    if(hp && h) {
	if(*hp && *hp != h)
	    HandlePDelete(hp);
	*hp = h;
    }
    return h;
}

char *
PoolName(Pool *p) { return p ? p->poolname : NULL; }

Pool *
PoolByName(char *fname)
{
    register Pool *p;

    for(p = AllPools; p != NULL; p = p->next)
	if(strcmp(fname, p->poolname) == 0)
	    return p;
    return NULL;
}


static
watchfd(int fd)
{
    if(fd < 0 || fd >= FD_SETSIZE || FD_ISSET(fd, &poolwatchfds))
	return;

    FD_SET(fd, &poolwatchfds);
    if(poolmaxfd <= fd)
	poolmaxfd = fd+1;
}

static
unwatchfd(int fd)
{
    register Pool *p;
    register int i;

    if(fd < 0 || fd >= FD_SETSIZE)
	return;

    if(FD_ISSET(fd, &poolwatchfds)) {
	FD_CLR(fd, &poolwatchfds);
    }
    if(fd+1 >= poolmaxfd) {
	for(i = poolmaxfd; --i >= 0 && !FD_ISSET(i, &poolwatchfds); )
	    ;
	poolmaxfd = i+1;
    }
    if(FD_ISSET(fd, &poolreadyfds)) {
	FD_CLR(fd, &poolreadyfds);
	poolnready--;
    }
}

Pool *
PoolStreamTemp(char *name, FILE *f, int rw, HandleOps *ops)
{
    register Pool *p = OOGLNewE(Pool, "new scratch Pool");

    p->ops = ops;
    p->poolname = name ? strdup(name) : NULL;
    p->type = P_STREAM;
    if(f == NULL && name != NULL)
	f = fopen(name, rw ? (rw>1 ? "w+":"w") : "r");
    p->inf = p->outf = f;
    if(f) {
	switch(rw) {
	    case 0: p->outf = NULL; break;
	    case 1: p->inf = NULL; break;
	    case 2: p->outf = fdopen(dup(fileno(f)), "w");
	}
    }
    p->handles = NULL;
    p->resyncing = NULL;
    p->otype = PO_ALL;
    p->next = NULL;
    p->mode = rw;
    p->level = 0;
    p->flags = PF_TEMP;
    return p;
}

Pool *
PoolStreamOpen(char *name, FILE *f, int rw, HandleOps *ops)
{
    register Pool *p;
    struct stat st;

    p = PoolByName(name);

    if(p == NULL) {
	p = OOGLNewE(Pool, "new Pool");
	p->ops = ops;
	p->poolname = strdup(name);
	p->type = P_STREAM;
	p->inf = p->outf = NULL;
	p->mode = rw;
	p->handles = NULL;
	p->resyncing = NULL;
	p->otype = PO_ALL;
	p->level = 0;
	p->flags = 0;
	p->next = AllPools;
    } else {
	if(rw == 0 && p->mode == 0 && p->inf != NULL
		&& p->softEOF == 0
		&& (p->flags & PF_REREAD) == 0
		&& stat(name, &st) == 0
		&& st.st_mtime == p->inf_mtime) {
	    rewind(p->inf);
	    return p;
	}

	/*
	 * Combine modes.  Allows e.g. adding write stream to read-only pool.
	 */
	p->mode = ((p->mode+1) | (rw+1)) - 1;
	if(p->inf && rw != 1) {
	    fclose(p->inf);
	    p->inf = NULL;
	}
    }

    if(f == NULL || f == (FILE *)-1) {
	if(rw != 1) {
	    if(strcmp(name, "-") == 0) {
		p->inf = stdin;
	    } else {
		int fd;

		fd = open(name, O_RDONLY | O_NDELAY);
		if(fd < 0)
		    OOGLError(1, "%s: cannot open: %s", name, sperror());
		else {
#ifdef FNONBLK
		    fcntl(fd, F_SETFL, fcntl(fd, F_GETFL) & ~(FNDELAY|FNONBLK));
#else
		    fcntl(fd, F_SETFL, fcntl(fd, F_GETFL) & ~FNDELAY);
#endif
		    p->inf = fdopen(fd, "r");
		}
	    }
	}
	if(rw > 0) {
	    if(strcmp(name, "-") == 0)
		p->outf = stdout;
	    else if((p->outf = fopen(name, "w")) == NULL)
		OOGLError(1, "%s: cannot create: %s", name, sperror());
	}
    } else {
	if(rw != 1)
	    p->inf = f;
	if(rw > 0)
	    p->outf = (rw == 2) ? fdopen(dup(fileno(f)), "w") : f;
    }

    if(p->inf == NULL && p->outf == NULL) {
	PoolDelete(p);
	return NULL;
    }

	/* We're committed now. */
    if(p->next == AllPools)
	AllPools = p;
    p->seekable = 0;
    p->softEOF = 0;
    if(p->inf != NULL) {
	if(isatty(fileno(p->inf))) {
	    p->softEOF = 1;
	} else if(tell(fileno(p->inf)) != -1) {
	    p->seekable = 1;
	}
	if(fstat(fileno(p->inf), &st) < 0 || (st.st_mode & S_IFMT) == S_IFIFO)
	    p->softEOF = 1;
	p->inf_mtime = st.st_mtime;

	watchfd(fileno(p->inf));
    }
    return p;
}

int
PoolIncLevel(Pool *p, int incr)
{
    return (p->level += incr);
}

int
PoolOType(Pool *p, int otype)
{
    return p->otype;
}

void
PoolSetOType(Pool *p, int otype)
{
    p->otype = otype;
}

FILE *
PoolInputFile(Pool *p)
{
    return (p && p->type == P_STREAM) ? p->inf : NULL;
}

FILE *
PoolOutputFile(Pool *p)
{
    return (p && p->type == P_STREAM) ? p->outf : NULL;
}

void
PoolDoReread(Pool *p)
{
    p->flags |= PF_REREAD;	/* Reread when asked */
}

PoolClose(register Pool *p)
{
    if(p->type != P_STREAM)
	return;

    if(p->inf != NULL) {
	unwatchfd(fileno(p->inf));
	fclose(p->inf);
	p->inf = NULL;
    }
    if(p->outf != NULL) {
	fclose(p->outf);
	p->outf = NULL;
    }
}

PoolDelete(Pool *p)
{
    register Pool **pp;

    if(p == NULL) return;
    if((p->flags & PF_TEMP) == 0) {
	for(pp = &AllPools; *pp != NULL; pp = &(*pp)->next) {
	    if(*pp == p) {
		*pp = p->next;
		OOGLFree(p);
		return;
	    }
	}
    }
    OOGLFree(p);
}

/*
 * PoolInputFDs(fds, maxfd)
 * supplies a list of file descriptors for select() to watch for input.
 * Sets *fds to the set of pool input file descriptors
 * Sets *maxfd to the max+1 file descriptor in fds.
 * Returns 1 if some pool(s) have input immediately available
 * (so select() shouldn't block for long),
 * 0 if none do.
 */
int
PoolInputFDs(fd_set *fds, int *maxfd)
{
    *fds = poolwatchfds;
    *maxfd = poolmaxfd;
    return poolnready != 0;
}

/*
 * PoolInAll(fd_set *fds, int nfds)
 * Read from all available pools, given a set of file descriptors
 * which claim to have some input.  We also import from all those which
 * already have buffered input.
 * nfds is an advisory upper bound on the number of fd's in *fds;
 * if the caller doesn't keep the return value from select(), just use
 * FD_SETSIZE or other huge number.
 * Each fd used is removed from *fds, and *nfds is decremented.
 * The return value is 1 if anything was read from any pool, 0 otherwise.
 */
int
PoolInAll(register fd_set *fds, int *nfds)
{
    register Pool *p;
    int got = 0;

    for(p = AllPools; p != NULL; p = p->next) {
	if(p->type != P_STREAM || p->inf == NULL /* || ignoring(p) */)
	    continue;

	if(FD_ISSET(fileno(p->inf), &poolreadyfds)) {
	    FD_CLR(fileno(p->inf), &poolreadyfds);
	    poolnready--;
	    if(PoolIn(p))
		got++;
	} else if(FD_ISSET(fileno(p->inf), fds)) {
	    FD_CLR(fileno(p->inf), fds);
	    (*nfds)--;
	    if(PoolIn(p))
		got++;
	}
    }
    return got;
}

static
poolresync(Pool *p, int (*resync)())
{ /* XXX implement this */
}


#define	CBRA	'{'
#define	CKET	'}'

Handle *
PoolIn(Pool *p)
{
    register int c;
    int fd;
    Handle *h = NULL;
    Ref *r = NULL;
    int ok;

    if(p->type != P_STREAM)
	return NULL;		/* Do shared memory here someday XXX */
    if(p->inf == NULL || p->ops == NULL || p->ops->strmin == NULL)
	return NULL;		/* No way to read */


    c = async_fnextc(p->inf, 0);
    if(c == EOF) {
	if(p->softEOF) {
	    rewind(p->inf);
	} else {
	    p->ops->close ? (*p->ops->close)(p) : PoolClose(p);
	}
	return NULL;
    }
    if(c != NODATA) {
	/* Kludge.  The interface to TransStreamIn really needs to change. */
	extern int TransStreamIn(/* Pool *, Handle **, Transform */);

	if((*p->ops->strmin)(p, &h,
			(p->ops->strmin == TransStreamIn) ? NULL : &r)) {
	    /*
	     * Attach nameless objects to a handle named for the Pool.
	     * Putting this code here in PoolIn() ensures we just bind names to
	     * top-level objects, not those nested inside a hierarchy.
	     */
	    if(h == NULL && r != NULL) {
		h = HandleAssign(p->poolname, p->ops, r);
		    /* Decrement reference count since we're handing
		     * ownership of the object to the Handle.
		     */
		RefDecr(r);
	    }

	    /* Delete r since stream-in routine called RefIncr for us. */
	
	    if(r && p->ops->delete)
		(*p->ops->delete)(r);

	    /* Remember whether we've read (PF_ANY) at least one and
	     * (PF_REREAD) at least two objects from this pool.
	     * There'll be a nontrivial side effect of rereading a file
	     * containing multiple objects, so we actually do reread if asked.
	     */
	    p->flags |= (p->flags & PF_ANY) ? PF_REREAD : PF_ANY;
	} else {
	    if(p->ops->resync) {
		(*p->ops->resync)(p);
	    } else if(p->softEOF) {
		rewind(p->inf);
	    } else {
		if(FD_ISSET(fileno(p->inf), &poolreadyfds)) {
		    FD_CLR(fileno(p->inf), &poolreadyfds);
		    poolnready--;
		}
		p->ops->close ? (*p->ops->close)(p) : PoolClose(p);
		return NULL;
	    }
	}
    }

    /*
     * Anything left in stdio buffer?
     * If so, remember to try reading next time without waiting for select().
     */
    if(p->inf && p->inf->_cnt > 0) {
	if(!FD_ISSET(fileno(p->inf), &poolreadyfds)) {
	    FD_SET(fileno(p->inf), &poolreadyfds);
	    poolnready++;
	}
    } else {
	if(FD_ISSET(fileno(p->inf), &poolreadyfds)) {
	    FD_CLR(fileno(p->inf), &poolreadyfds);
	    poolnready--;
	}
    }
    return h;
}

int
PoolStreamOutHandle(Pool *p, Handle *h, int havedata)
{
    if(p == NULL || p->outf == NULL)
	return 0;

    if(h == NULL || p->otype == PO_DATA ||
				(p->otype == PO_HANDLES && havedata))
	return havedata;

    if(h->whence != NULL && h->whence->seekable) {
	fprintf(p->outf, " < \"");
	if(strcmp(h->name, p->poolname) == 0) {
	    fprintf(p->outf, "%s\"\n", h->whence->poolname);
	} else {
	    fprintf(p->outf, "%s:%s\"\n", h->whence->poolname, h->name);
	}
    } else {
	fprintf(p->outf, ": \"%s\"\n", h->name);
    }
    return (p->otype == PO_ALL);
}
