/*
 * XGL Appearance stuff
 */

#include "mgP.h"	/* mg private definitions */
#include "mg_xglP.h"	/* mg xgl-specific stuff */


/*
 * Thanks to mg_xglP.h, you can use
 *  MGC->astk->whatever to get at the corresponding field.
 *  e.g. MGC->astk->ap.linewidth, MGC->astk->mat.kd, etc.
 * The XGL drawing context is MGXGL->xglctx.
 * The XGL raster object is (MGXGL->membuf ? MGXGL->xglmem : MGXGL->xglwin).
 *
 * For Appearance structures & flag bits see shade/appearance.h.
 */             

/*
 * Incorporate appearance *ap into current context's appearance.
 * If "merge" set, just change those fields valid in *ap.
 * If "merge == 0, completely rewrite current appearance.
 * Of course, the current appearance's Material and Lighting change too...
 */

mgxgl_setappearance(Appearance *ap, int merge)
{
    int changed;		/* Mask of fields to change */

    /*
     * Handle changes to Appearance structure proper.
     */
    /* What fields will change?  Same code as in shade/appearance.c. */

    /* We incorporate everything except where the existing  appearance
     * says its value overrides all newcomers.
     */
    changed = merge ? ap->valid & ~MGC->astk->ap.override :
		      ap->valid;

    /*
     * Where XGL should know about the option, tell it.
     * E.g. we tell it about face & edge drawing, but not
     * about normal-vector drawing since we do that ourselves.
     */



    if(changed & APF_FACEDRAW) {
   /* We should think about making changes for back and front
    * differences here! 
    */
       if  (ap->flag & APF_FACEDRAW) {
   /* This is the command to fill in the faces */
          xgl_object_set (MGXGL->xglctx,
            XGL_CTX_SURF_FRONT_FILL_STYLE,XGL_SURF_FILL_SOLID,
            XGL_3D_CTX_SURF_BACK_FILL_STYLE,XGL_SURF_FILL_SOLID,0);
   /* Both front and back must be specified unless 
    * XGL_SURF_FACE_DISTINGUISH is set to FALSE
    */
	  }
       else {
   /* Nothing is drawn unless edges is specified */
          xgl_object_set (MGXGL->xglctx,
            XGL_CTX_SURF_FRONT_FILL_STYLE,XGL_SURF_FILL_EMPTY,
            XGL_3D_CTX_SURF_BACK_FILL_STYLE,XGL_SURF_FILL_EMPTY,0);
	}
    /*
     * There is also a possibility of taking advantage of
     *  XGL_SURF_FILL_HOLLOW , ... STIPPLE, ... OPAQUE_STIPPLE in
     * xgl..
     */
   }
    if (changed & APF_EDGEDRAW) {
       if (ap->flag & APF_EDGEDRAW)
         /* This turns on the edge drawing (FALSE turns it off)*/
         xgl_object_set(MGXGL->xglctx,XGL_CTX_SURF_EDGE_FLAG,TRUE,0);
       else
         xgl_object_set(MGXGL->xglctx,XGL_CTX_SURF_EDGE_FLAG,FALSE,0);
    }

   
    /* Maybe other APF_ things, e.g. APF_TRANSP, APF_EVERT, APF_NORMSCALE */

    if (changed & APF_SHADING) {
	switch(ap->shading) {
	case APF_CONSTANT:
          xgl_object_set (MGXGL->xglctx,
            XGL_3D_CTX_SURF_FRONT_ILLUMINATION,XGL_ILLUM_NONE,
            XGL_3D_CTX_SURF_BACK_ILLUMINATION,XGL_ILLUM_NONE,
            0);
	break;
	case APF_FLAT:
          xgl_object_set (MGXGL->xglctx,
            XGL_3D_CTX_SURF_FRONT_ILLUMINATION,XGL_ILLUM_PER_FACET,
            XGL_3D_CTX_SURF_BACK_ILLUMINATION,XGL_ILLUM_PER_FACET,
            0);
	  break;
	case APF_SMOOTH:
          xgl_object_set (MGXGL->xglctx,
            XGL_3D_CTX_SURF_FRONT_ILLUMINATION,XGL_ILLUM_PER_VERTEX,
            XGL_3D_CTX_SURF_BACK_ILLUMINATION,XGL_ILLUM_PER_VERTEX,
            0);
	}
     }

    if (changed & APF_LINEWIDTH ) {
    /* This has some room for interpretation...
     * The mg appearance linewidth is an integer.
     * (it is not clear to me what the stands for, pixels
     * on the screen or some relative scale.)
     * xgl uses a float (width scale factor) that determines
     * the width of lines and surface edges .. 
     * dev line width = line scale factor * nominal line width
     */

    /* we are going to convert linewidth directly to a float..
     * The XGL manual recommends odd line widths
     */

     xgl_object_set (MGXGL->xglctx,
       XGL_CTX_EDGE_WIDTH_SCALE_FACTOR, (float)ap->linewidth,
       XGL_CTX_LINE_WIDTH_SCALE_FACTOR, (float)ap->linewidth,
     0);
   /* line_width is for the lines in multilines
    * edge_width is for the edges of polygons, these must be defined seperatedly
    */

   }
    /*
     * Incorporate Material changes
     */
    if(ap->mat != NULL) {
	changed = merge ? ap->mat->valid & ~MGC->astk->mat.override :
			  ap->mat->valid;
    /* Maybe we should incorporate something for different
     * front and back materials, since XGL has it..
     */
	if (changed & MTF_Ka )
     /* This sets the xgl's ambient reflection coefficient used in 3d calculations */
          xgl_object_set (MGXGL->xglctx,
            XGL_3D_CTX_SURF_FRONT_AMBIENT, ap->mat->ka,
            XGL_3D_CTX_SURF_BACK_AMBIENT,  ap->mat->ka,
          0);

	if (changed & MTF_Kd )
     /* This sets the xgl's diffuse reflection coefficient used in 3d calculations */
          xgl_object_set (MGXGL->xglctx,
            XGL_3D_CTX_SURF_FRONT_DIFFUSE, ap->mat->kd,
            XGL_3D_CTX_SURF_BACK_DIFFUSE,  ap->mat->kd,
          0);

	if (changed & MTF_Ks ) {
     /* This sets the xgl's specular reflection coefficient used in 3d calculations */
          xgl_object_set (MGXGL->xglctx,
            XGL_3D_CTX_SURF_FRONT_SPECULAR, ap->mat->ks,
            XGL_3D_CTX_SURF_BACK_SPECULAR,  ap->mat->ks,
	    XGL_3D_CTX_SURF_FRONT_LIGHT_COMPONENT, 
		XGL_LIGHT_ENABLE_COMP_AMBIENT|XGL_LIGHT_ENABLE_COMP_DIFFUSE|XGL_LIGHT_ENABLE_COMP_SPECULAR,
	    XGL_3D_CTX_SURF_BACK_LIGHT_COMPONENT, 
		XGL_LIGHT_ENABLE_COMP_AMBIENT|XGL_LIGHT_ENABLE_COMP_DIFFUSE|XGL_LIGHT_ENABLE_COMP_SPECULAR,
          0);
	}


    /* Note: front and back must be specified unless
     *  XGL_3D_CTX_SURF_FACE_DISTINGUISH is FALSE
     */

	if(changed & MTF_DIFFUSE) {
	    xgl_object_set(MGXGL->xglctx,
		XGL_CTX_SURF_FRONT_COLOR, &ap->mat->diffuse,
		XGL_3D_CTX_SURF_BACK_COLOR, &ap->mat->diffuse,
		0);
	}
	if(changed & MTF_SPECULAR) {
	    xgl_object_set(MGXGL->xglctx,
		XGL_3D_CTX_SURF_FRONT_SPECULAR_COLOR, &ap->mat->specular,
		XGL_3D_CTX_SURF_BACK_SPECULAR_COLOR, &ap->mat->specular,
		0);
	}
	if(changed & MTF_EDGECOLOR) {
	    xgl_object_set(MGXGL->xglctx,
		XGL_CTX_EDGE_COLOR, &ap->mat->edgecolor, 0);
	}
    } else if(!merge) {
	/* Not merging, but there's no new material.  Invalidate Material */
	MGC->astk->mat.valid = MGC->astk->mat.override = 0;
    }

    /*
     * Incorporate Lighting changes
     */
    
    if(ap->lighting != NULL) {
	changed = merge ? ap->lighting->valid & ~MGC->astk->lighting.override :
			  ap->lighting->valid;

	if(changed) {
	    MGC->astk->light_seq++;
	    mgxgl_installlights(ap->lighting->lights,
		!merge || (ap->lighting->valid & LMF_REPLACELIGHTS) ?
			1 : MGXGL->xglnlights);
	    mgxgl_installlmodel(ap->lighting);
	}
    }

    /*
     * Apply changes to our Appearance structure on stack.
     */
    mg_setappearance(ap, merge);
}

mgxgl_installlights(LtLight *lt, int first)
{
    register LtLight *l;
    int nlights = first;
    int i;
    static int pushme[] = {
		    XGL_CTX_LOCAL_MODEL_TRANS, /*XGL_CTX_GLOBAL_MODEL_TRANS,*/ 0 };

    xgl_context_push(MGXGL->xglctx, pushme);
    /*xgl_transform_identity(MGXGL->xglcam);*/
    xgl_transform_identity(MGXGL->xglmodel);
    for(l = lt; l != NULL; l = l->next)
	nlights++;
    if(nlights > MGXGL_MAXLIGHTS)
	nlights = MGXGL_MAXLIGHTS;
    if(nlights != MGXGL->xglnlights) {
	xgl_object_set(MGXGL->xglctx, XGL_3D_CTX_LIGHT_NUM, nlights, 0);
	if(nlights > MGXGL->xglnlights)
	    xgl_object_get(MGXGL->xglctx, XGL_3D_CTX_LIGHTS, MGXGL->xgllgts);
	MGXGL->xglnlights = nlights;
    }
    for(l = lt, i = first; i < nlights; l = l->next, i++) {
	Color c;
	Point3 p;
	float w;

	c.r = l->intensity * l->color.r;
	c.g = l->intensity * l->color.g;
	c.b = l->intensity * l->color.b;
	w = l->position.w == 0 ? -1 : 1 / l->position.w;
	p.x = l->position.x * w;
	p.y = l->position.y * w;
	p.z = l->position.z * w;
	
	if(l->position.w == 0) {
	    xgl_object_set(MGXGL->xgllgts[i],
		XGL_LIGHT_COLOR, &c,
		XGL_LIGHT_TYPE, XGL_LIGHT_DIRECTIONAL,
		XGL_LIGHT_DIRECTION, &p,
		0);
	} else {
	    xgl_object_set(MGXGL->xgllgts[i],
		XGL_LIGHT_COLOR, &c,
		XGL_LIGHT_TYPE, XGL_LIGHT_POSITIONAL,
		XGL_LIGHT_POSITION, &p,
		0);
	}
    }
    xgl_context_pop(MGXGL->xglctx);
}

mgxgl_installlmodel(LmLighting *lm)
{
    if(lm->valid & LMF_AMBIENT) {
	if(MGXGL->xglnlights == 0)
	    mgxgl_installlights(NULL, 1);

	xgl_object_set(MGXGL->xgllgts[0], 
		XGL_LIGHT_TYPE, XGL_LIGHT_AMBIENT,
		XGL_LIGHT_COLOR, &lm->ambient,
		0);
    }
}
