#include "mg_xglP.h"

/*
 * Some geometric primitives in XGL
 */

static
copyrow(char *to, Point3 *p, Point3 *n, ColorA *c, int npts, int istep, int ostep)
{
    register int step;
    register int k;
    register char *ip, *op;

    k = npts;
    ip = (char *)p;
    op = to;
    step = sizeof(Point3) * istep;
    do {
	*(Xgl_pt_f3d *)op = *(Xgl_pt_f3d *)ip;
	ip += step;
	op += ostep;
    } while(--k > 0);
    op = MGXGL->xbuf + sizeof(Xgl_pt_f3d);
    if(c) {
	k = npts;
	ip = (char *)c;
	step = istep * sizeof(ColorA);
	do {
	    *(Xgl_color *)op = *(Xgl_color *)ip;
	    ip += step;
	    op += ostep;
	} while(--k > 0);
	op = MGXGL->xbuf + sizeof(Xgl_pt_f3d) + sizeof(Xgl_color);
    }
    if(n) {
	k = npts;
	ip = (char *)n;
	step = istep * sizeof(Point3);
	do {
	    *(Xgl_pt_f3d *)op = *(Xgl_pt_f3d *)ip;
	    ip += step;
	    op += ostep;
	} while(--k > 0);
    }
}

void
mgxgl_mesh(int wrap, int nu, int nv, Point3 *p, Point3 *n, ColorA *c)
{ 
    int perfacet, perpoint;		/* Bytes per facet and per point */
    int pt_type;			/* Xgl_pt_type : per-vertex data type */
    int vertc = 0, vertn = 0;
    int facetc = 0, facetn = 0;
    Xgl_facet_list fl, *flp;
    Xgl_pt_list pl;
    int npts = nu * nv;
    int wanted = 0;

    if(nu <= 1 || nv <= 1)
	return;

    /*
     * Determine format of points and facets based on available & relevant data.
     * Strip irrelevant bits off plflags; set "perfacet" and "perpoint" sizes.
     */
    if(c != NULL)
	vertc = facetc = 1;
    if(n != NULL)
	vertn = facetn = 1;

    switch(MGC->astk->ap.shading) {
	case APF_CONSTANT:	facetn = 0;	/* fall into FLAT */
	case APF_FLAT:		vertc = vertn = 0; break;
	case APF_SMOOTH:	facetc = facetn = 0; break;
    }

    /*
     * Determine size and datatype of vertices
     */
    if(vertc) {
	perpoint = sizeof(Xgl_pt_color_f3d);
	pl.pt_type = XGL_PT_COLOR_F3D;
	if(vertn) {
	    perpoint = sizeof(Xgl_pt_color_normal_f3d);
	    pl.pt_type = XGL_PT_COLOR_NORMAL_F3D;
	}
    } else if(vertn) {
	perpoint = sizeof(Xgl_pt_normal_f3d);
	pl.pt_type = XGL_PT_NORMAL_F3D;
    } else {
	perpoint = sizeof(Xgl_pt_f3d);
	pl.pt_type = XGL_PT_F3D;
    }

    flp = &fl;
    if(facetc) {
	fl.facet_type = XGL_FACET_COLOR;
	perfacet = sizeof(Xgl_color_facet);
	if(facetn) {
	    fl.facet_type = XGL_FACET_COLOR_NORMAL;
	    perfacet = sizeof(Xgl_color_normal_facet);
	}
    } else if(facetn) {
	fl.facet_type = XGL_FACET_NORMAL;
	fl.facets.normal_facets = (Xgl_normal_facet *)n;
    } else {
	fl.facet_type = XGL_FACET_NONE;
	flp = NULL;
    }

    /*
     * Cases:
     *  points only -- just feed them to XGL
     *  points + something else per vertex -- copy all points (sigh)
     *	point only per vertex, something per facet -- copy other info
     * Also, wrapped meshes get an extra 1 or 2 lumps.
     */
    
    if(pl.pt_type == XGL_PT_F3D) {
	/*
	 * points only, or points plus facet data.  Don't copy points.
	 * Feed simple N array directly to XGL; copy if we need C's or both.
	 * We may need multiple passes if it's a wrapped mesh.
	 */
	if(facetc) {
	    register char *ip;
	    register char *op;
	    register int k;

	    wanted = perfacet;
	    mgxgl_wantbuf(npts * perfacet);
	    fl.facets.color_facets = (Xgl_color_facet *)MGXGL->xbuf;
	    k = npts;
	    ip = (char *)c;
	    op = MGXGL->xbuf;
	    do {
		*(Xgl_color *)op = *(Xgl_color *)ip;
		ip += sizeof(ColorA);
		op += perfacet;
	    } while(--k > 0);

	    if(facetn) {
		k = npts;
		ip = (char *)n;
		op = MGXGL->xbuf + sizeof(Xgl_color);
		do {
		    *(Xgl_pt_f3d *)op = *(Xgl_pt_f3d *)ip;
		    ip += sizeof(Xgl_pt_f3d);
		    op += sizeof(Xgl_color_normal_facet);
		} while(--k > 0);
	    }
	}
	pl.pts.f3d = (Xgl_pt_f3d *)p;

    } else {
	    /* Must copy all vertices */
	wanted = perpoint;
	mgxgl_wantbuf(npts * perpoint);
	copyrow(MGXGL->xbuf, p, n, c,  npts, 1, perpoint);
	pl.pts.f3d = (Xgl_pt_f3d *)MGXGL->xbuf;
    }

    /*
     * Draw the bulk of the mesh
     */
    pl.bbox = NULL;
    pl.num_pts = npts;
    fl.num_facets = npts;
    xgl_quadrilateral_mesh(MGXGL->xglctx, nv, nu, flp, &pl);

    /*
     * Is it a wrapped mesh?
     * If so, make recursive call(s) for remaining strip(s).
     * We must copy p,n,c edges.
     */
     if(wrap & MM_UWRAP) {
	int newv = nv;
	Point3 *poff, *noff;
	ColorA *coff;
	int k;

	if(wrap & MM_VWRAP)
	    newv++;
	mgxgl_wantbuf((wanted + 2*sizeof(Point3) + sizeof(ColorA)) * 2*newv);
	poff = (Point3 *)(MGXGL->xbuf + wanted * 2*newv);
	noff = poff + 2*newv;
	coff = (ColorA *)(noff + 2*newv);
	copypnc(p,n,c, nu-1, nu, 2, nv, poff,  noff,  coff);
	copypnc(p,n,c, 0,    nu, 2, nv, poff+1,noff+1,coff+1);
	if(wrap & MM_VWRAP) {
	    k = 2*nv;
	    copypnc(p,n,c, nu-1, 1-nu, 1, 2, poff+k, noff+k, coff+k);
	}
	mgxgl_mesh(0, 2, newv, poff, n?noff:0, c?coff:0);
    }
    if(wrap & MM_VWRAP) {
	Point3 *poff, *noff;
	ColorA *coff;
	int k;

	copypnc(p,n,c, nu*(nv-1), 1, 1, nu,  poff,noff,coff);
	copypnc(p,n,c, 0,	  1, 1, nu,  poff+nu, noff+nu, coff+nu);
	mgxgl_mesh(0, nu, 2, poff, n?noff:NULL, c?coff:NULL);
    }

    if(MGC->astk->ap.flag & APF_NORMSCALE && n != NULL)
	mgxgl_normals(npts, p, n, sizeof(Point3), sizeof(Point3));
}

copypnc(Point3 *p,  Point3 *n,  ColorA *c,
	int ioff,   int istep,  int ostep, int count,
	Point3 *op, Point3 *on, ColorA *oc)
{
    register int k;
    register char *in, *out;
    register int instep, outstep;

    k = count;
    in = (char *)(p + ioff);
    out = (char *)op;
    instep = istep * sizeof(Point3);
    outstep = ostep * sizeof(Point3);
    do {
	*(Point3 *)out = *(Point3 *)in;
	out += outstep;  in += instep;
    } while(--k > 0);
    if(n) {
	k = count;
	in = (char *)(n + ioff);
	out = (char *)on;
	do {
	    *(Point3 *)out = *(Point3 *)in;
	    out += outstep;  in += instep;
	} while(--k > 0);
    }
    if(c) {
	k = count;
	in = (char *)(c + ioff);
	out = (char *)oc;
	instep = istep * sizeof(ColorA);
	outstep = ostep * sizeof(ColorA);
	do {
	    *(ColorA *)out = *(ColorA *)in;
	    out += outstep;  in += instep;
	} while(--k > 0);
    }
}
