/*
 * Main module: XGL driver for mg library
 */
#include <stdarg.h>
#include "mg_xglP.h"
#include "windowP.h"			/* friend of Window */
extern Appearance *_ApSet(Appearance *, int, va_list *); /* XXX put in ap.h */

#undef __STDC__
#include <xview/frame.h>
#include <xview/canvas.h>
#include <xview/xv_xrect.h>
#include <X11/Xlib.h>

#ifdef SUNBUG
#include <X11/Xatom.h>			/* For XA_WINDOW XView magic */
#endif /*SUNBUG*/

#define __STDC__
#include <stdarg.h>



extern struct mgfuncs mgxglfuncs;	/* Forward */

		/*
		 * XView default callback routines, user-replaceable.
		 * These just set flags readable with MG_REPAINT and MG_RESIZE.
		 */
void _mgxgl_xv_repaint();		/* Forward */
void _mgxgl_xv_resize();
void mgxgl_sync();
void mgxgl_worldend();

extern Appearance *mgxgl_setappearance(Appearance *ap, int mergeflag);
	/*{ return mg_setappearance(ap, mergeflag); }*/

/*
 * static data
 */

Xgl_sys_state	XGLstate = NULL;


int
mgdevice_XGL()
{
    _mgf = mgxglfuncs;
    if(MGC != NULL && MGC->devno != MGD_XGL)
	MGC = NULL;
    return 0;
}


int
mgxgl_feature(int feature)  /* XXX No features defined yet */
{ return -1;
}

/*
 * mgcontext *mgxgl_ctxcreate(attr, ..., MG_END)
 * Create a new XGL context.  Initialize any device-specific parts.
 * Pass attribute list off to mgxgl_set().
 * The latter will create a window when done, unless attr MG_SHOW is set to 0.
 * slevy, Tue Jul 23 16:06:46 CDT 1991
 */
mgcontext *
mgxgl_ctxcreate(int attr, ... /* , MG_END */)
{
    va_list alist;
    register mgxglcontext *mgxgl;

    MGC = (mgcontext *)OOGLNewE(mgxglcontext,
				"mgxglcontext mgxgl_ctxcreate");
    mg_newcontext(MGC);

    MGC->shown = 1;
    mgxgl = MGXGL;
    mgxgl->born = 0;
    mgxgl->repaint = 0;		/* Set by expose-like events */
    mgxgl->resize = 1;		/* Set by resize events */
    mgxgl->xglctx = mgxgl->ctx_win = mgxgl->ctx_mem = NULL;
    mgxgl->useX11 = 0;		/* Use XView by default */
    mgxgl->frame = 0;		/* No XView frame yet */
    mgxgl->canvas = NULL;	/* No XView canvas either */
    mgxgl->x11win = 0;
    mgxgl->x11dpy = NULL;
    mgxgl->x11scrn = -1;
    mgxgl->dorepaint = _mgxgl_xv_repaint;
    mgxgl->doresize = _mgxgl_xv_resize;

    mgxgl->bufsiz = 128;
    mgxgl->bufused = 0;
    mgxgl->flushfunc = NULL;
    mgxgl->xbuf = (void *)OOGLNewNE(char, mgxgl->bufsiz, "MG XGL misc buffer");

    mgxgl->weshade = 0;
    mgxgl->wedither = 0;

    va_start(alist, attr);
    _mgxglset(attr, &alist);
    va_end(alist);
}

void
mgxgl_ctxdelete()
{
    free(MGXGL);		/* XXX free more than this!  & close window */
    MGC = NULL;
}

int
mgxgl_ctxselect(mgcontext *ctx)
{
    if(ctx->devno != MGD_XGL)
	return mg_ctxselect(ctx);
    MGC = ctx;
    return 0;
}

mgcontext *
mgxgl_findctxbywin(int x11win)
{
    register mgcontext *mgc;

    for(mgc = _mgclist; mgc != NULL; mgc = mgc->next) {
	if(mgc->devno == MGD_XGL && ((mgxglcontext *)mgc)->x11win == x11win)
	    return mgc;
    }
    return NULL;
}


mgxgl_ctxset(int attr, ...)
{
    va_list alist;

    va_start(alist, attr);
    _mgxglset(attr, &alist);
    va_end(alist);
}

_mgxglset(int attr, register va_list *alist)
{
    int a;
    int oopt;
    WnWindow *win;
    Appearance *ap;

    a = attr;
    while(a != MG_END) {
	switch(a) {
	case MG_SHOW:   MGC->shown = va_arg(*alist, int); break;
	case MG_APPEAR:
		mgxgl_setappearance(va_arg(*alist, Appearance *), 0);
		break;
	case MG_ApSet:
		a = va_arg(*alist, int);
		ap = _ApSet(NULL, a, alist);
		mgxgl_setappearance(ap, 1);
		ApDelete(ap);
		break;
		
	case MG_PARENT: MGC->parent = va_arg(*alist, mgcontext *); break;
	case MG_BACKGROUND: MGC->background = *va_arg(*alist, ColorA *); break;

	case MG_WINDOW:
	    win = va_arg(*alist, WnWindow *);
	    if(win == NULL) break;
	    RefIncr((Ref *)win);
	    if(MGC->win) WnDelete(MGC->win);
	    if(MGXGL->born) {
		mgxgl_movewin(win);	/* Window already exists: move it */
		if(MGXGL->frame)
		    xv_set(MGXGL->frame, FRAME_LABEL, win->win_name, 0);
		/* else if(MGXGL->useX11) ... */
	    }
	    MGC->win = win;
	    break;

	    /*
	     * Could object to changing some option bits after window "born".
	     * Some require reconfiguration if already born; do that here.
	     */
	case MG_SETOPTIONS:
		oopt = MGC->opts;
		MGC->opts |= va_arg(*alist, int);
		if(MGXGL->born && (oopt ^ MGC->opts) & MGO_HIDDEN)
		    mgxgl_doZbuffer();
		break;

	case MG_UNSETOPTIONS:
		oopt = MGC->opts;
		MGC->opts &= ~va_arg(*alist, int);
		if(MGXGL->born && (oopt ^ MGC->opts) & MGO_HIDDEN)
		    mgxgl_dontZbuffer();
		break;

	/*
	 * XGL-specific options
	 * Most should be unchangable after "born"; prohibit this! XXX
	 */
	case MG_XV_FRAME: MGXGL->frame = va_arg(*alist, Frame); break;

	case MG_XV_CANVAS:
		MGXGL->canvas = va_arg(*alist, Canvas); 
		MGXGL->xvwin = canvas_paint_window(MGXGL->canvas);
		if(MGXGL->xvwin) {
		    MGXGL->useX11 = 0;
		    MGXGL->frame = (Frame)xv_get(MGXGL->xvwin, WIN_FRAME);
		    MGXGL->canvas = (Canvas)xv_get(MGXGL->xvwin,
						CANVAS_PAINT_CANVAS_WINDOW);
		}
		break;

	case MG_XV_WINDOW:
		MGXGL->xvwin = va_arg(*alist, Xv_Window);
		if(MGXGL->xvwin) {
		    MGXGL->useX11 = 0;
		    MGXGL->canvas = (Canvas)xv_get(MGXGL->xvwin,
					CANVAS_PAINT_CANVAS_WINDOW);
		}
		break;

	case MG_XWINDOW:
		MGXGL->x11win = va_arg(*alist, Window);
		MGXGL->useX11 = 1;
		break;

	case MG_XDISPLAY:	MGXGL->x11dpy = va_arg(*alist, Display *); break;
	case MG_XSCREEN:	MGXGL->x11scrn = va_arg(*alist, int); break;
	case MG_REPAINT_PROC:	MGXGL->dorepaint=(void(*)())va_arg(*alist,void*); break;
	case MG_RESIZE_PROC:	MGXGL->doresize=(void(*)())va_arg(*alist,void*); break;
	case MG_REPAINT:	MGXGL->repaint = va_arg(*alist, int); break;
	case MG_RESIZE:		MGXGL->resize = va_arg(*alist, int); break;
	case MG_XGL_STATE:	XGLstate = va_arg(*alist, Xgl_sys_state); break;

	default:
	    OOGLError(1, "mgxgl_set: unknown option %d = 0x%x", a, a);
	    return;
	}
	a = va_arg(*alist, int);
    }

    if(MGC->shown && !MGXGL->born) {
	if(mgxgl_newwindow() < 0)
	    OOGLError(1, "mgctxset/mgctxcreate: Error creating XGL window");
	else
	    MGXGL->born = 1;
    }
}

int
mgxgl_ctxget(int attr, void *valp)
{
    switch(attr) {
    case MG_SHOW:	*(int *)valp = MGC->shown; break;
    case MG_APPEAR:	*(Appearance **)valp = &MGC->astk->ap; break;
    case MG_PARENT:	*(mgcontext **)valp = MGC->parent; break;
    case MG_WINDOW:	*(WnWindow **)valp = MGC->win; break;
    case MG_CAMERA:	*(Camera **)valp = MGC->cam; break;
    case MG_SETOPTIONS:	*(int *)valp = MGC->opts; break;
    case MG_XV_FRAME:	*(Frame *)valp = MGXGL->frame; break;
    case MG_XV_CANVAS:	*(Canvas *)valp = MGXGL->canvas; break;
    case MG_XV_WINDOW:	*(Xv_Window *)valp = MGXGL->xvwin; break;
    case MG_XV_RASTER:	*(Xgl_object *)valp = MGXGL->xglwin; break;
    case MG_XWINDOW:	*(Window *)valp = MGXGL->x11win; break;
    case MG_XDISPLAY:	*(Display **)valp = MGXGL->x11dpy; break;	
    case MG_XSCREEN:	*(int *)valp = MGXGL->x11scrn; break;
    case MG_REPAINT_PROC: *(void (**)())valp = MGXGL->dorepaint; break;
    case MG_RESIZE_PROC:  *(void (**)())valp = MGXGL->doresize; break;
    case MG_REPAINT:	*(int *)valp = MGXGL->repaint; break;
    case MG_RESIZE:	*(int *)valp = MGXGL->resize; break;
    case MG_XGL_STATE:	*(Xgl_sys_state *)valp = XGLstate; break;
    default:
	return -1;
    }
    return 1;
}

/*
 * mgxgl_window() does the work when a new XGL XView window is born.
 * Interprets options, opens & positions window, etc.
 * Takes advantage of user-supplied window (supplied via yet-to-be-defined
 * set() attribute).
 *
 * mgxgl_setupXVwin() is not a public entry point.
 */
int
mgxgl_setupXVwin()
{
    Atom catom;
    register mgxglcontext *mgxgl = MGXGL;
    register WnWindow *win = MGC->win;


    if(mgxgl->frame == NULL) {
	mgxgl->frame = xv_create(NULL,  FRAME, 
		    WIN_DYNAMIC_VISUAL, TRUE, /* -- may not need this */
		    0);
	if(mgxgl->frame == 0) {
	    OOGLError(1, "mgxgl_newwindow: Can't create XView window");
	    return -1;
	}
    }
    xv_set(mgxgl->frame, FRAME_LABEL, win->win_name, 0);
    if(mgxgl->canvas == NULL) {
	mgxgl->canvas = xv_create(mgxgl->frame, CANVAS,
			CANVAS_AUTO_CLEAR, FALSE,
			CANVAS_RETAINED, FALSE,
			CANVAS_FIXED_IMAGE, FALSE,
			NULL);
    }


    if(mgxgl->x11dpy == NULL)
	mgxgl->x11dpy = (Display *)xv_get(mgxgl->frame, XV_DISPLAY);
    mgxgl->xvwin = (Xv_Window)canvas_paint_window(mgxgl->canvas);

    mgxgl->x11win = (Window)xv_get(mgxgl->xvwin, XV_XID);

    xv_set(mgxgl->canvas,
	    CANVAS_REPAINT_PROC, mgxgl->dorepaint,
	    CANVAS_RESIZE_PROC,  mgxgl->doresize,
	    0);

#ifdef SUNBUG
    /*
     * What the heck does this do?
     * Mysterious code taken from demo/xgl/WS_macros.h.
     * A comment explains it's a workaround for an OpenWindows 1.0 bug.
     * Well, it seems to invoke an MIT X11R4/twm bug, so let's toss it.
     */
    catom = XInternAtom(mgxgl->x11dpy, "WM_COLORMAP_WINDOWS", False);
    XChangeProperty(mgxgl->x11dpy,
		(Window)xv_get(mgxgl->frame, XV_XID),	/* Frame window */
		catom,
		XA_WINDOW, 32,
		PropModeAppend,
		&mgxgl->x11win, 1);
#endif /*SUNBUG*/

    /*
     * Tell window manager about requested window size, etc.
     */
    if(win->flag & WNF_HASPREF)
	xv_set(mgxgl->frame, FRAME,
		/* WIN_DYNAMIC_VISUAL, TRUE,  -- may not need this */
		XV_X, win->pref.xmin,
		XV_Y, win->pref.ymin,
		XV_HEIGHT, win->pref.ymax - win->pref.ymin + 1,
		XV_WIDTH, win->pref.xmax - win->pref.xmin + 1,
		0);

    if(win->flag & WNF_HASSIZE)
	xv_set(mgxgl->frame, FRAME,
		XV_HEIGHT, win->ysize, XV_WIDTH, win->xsize, 0);

#ifdef notyet
    /*
     * How deep is our colormap?
     * If all we've got is 1-bit monochrome, XGL won't handle it,
     * so we enable software shading and dithering.
     */
    {
	Xv_cmsdata *colormap;
	colormap = (Xv_cmsdata *)xv_get(mgxgl->canvas, WIN_CMS_DATA);
	if(colormap == NULL) {
	    OOGLError(1, "mgxgl_setupXVwin: WIN_CMS_DATA gives NULL");
	} else {
#ifdef notdef
	    /* Don't do this until/unless we've thoroughly installed colorindex
	     * code in this library -- otherwise we get strange crashes
	     * when XGL tries to use a float R value as an int color index!
	     */
	    if(colormap->size < 16) {
		/*
		 * Pretty arbitrary.  Can't tell just what XGL won't do! 
		 */
		mgxgl->wedither = 1;
		mgxgl->weshade = 1;
		mgxgl->colorindex = 1;
	    }
#endif
	}
    }
#endif
}

/*
 * Internal routines: choose whether we do hidden-surface removal by Z-buffer.
 */
mgxgl_doZbuffer()
{
    Xgl_hlhsr_data z0;

    z0.z_buffer.z_value = 2.0;
    xgl_object_set(MGXGL->xglctx,
	XGL_3D_CTX_HLHSR_MODE, XGL_HLHSR_ZBUFFER,
	/*XGL_3D_CTX_HLHSR_DATA, &z0,*/
	0);
}

mgxgl_dontZbuffer()
{
    xgl_object_set(MGXGL->xglctx, XGL_3D_CTX_HLHSR_MODE, XGL_HLHSR_NONE, 0);
}

/*
 * mgxgl_setupX11win() creates a new XGL X11 (non-XView) window is born.
 */
int
mgxgl_setupX11win()
{
    return -1;		/* XXX implement mgxgl_setupX11win() */
}


/*
 * Do XGL-specific work in establishing an XGL window.
 * Hand off the XView or Xlib work.
 * Returns: -1 on failure, 1 on success.
 * Errors:  Reports failure from mgxgl_setup<*>win() or from xgl_3d_ctx_create.
 */
mgxgl_newwindow()
{
    register mgxglcontext *mgxgl = MGXGL;
    Xgl_X_window xgl_x_win;
    int ok;

    mgxgl->weshade = mgxgl->colorindex = mgxgl->wedither = 0;
    mgxgl->xglnlights = 0;
    ok = mgxgl->useX11 ? mgxgl_setupX11win() : mgxgl_setupXVwin();
    if(ok < 0) {
	OOGLError(1, "mgxgl_newwindow: can't create %s window",
		mgxgl->useX11 ? "X11" : "XView");
	return -1;
    }

    xgl_x_win.X_display = mgxgl->x11dpy;
    xgl_x_win.X_window  = mgxgl->x11win;
    xgl_x_win.X_screen  = mgxgl->x11scrn >= 0 ?
			    mgxgl->x11scrn : DefaultScreen(mgxgl->x11dpy);

    if(XGLstate == NULL) {
	XGLstate = xgl_open(XGL_SYS_ST_ERROR_DETECTION, TRUE, 0);
	if(XGLstate == NULL) {
	    OOGLError(1, "mgxgl_newwindow: can't xgl_open()");
	    return -1;
	}
    }

    mgxgl->xglwin = xgl_window_raster_device_create(XGL_WIN_X, &xgl_x_win,
		XGL_RAS_COLOR_TYPE,
		mgxgl->colorindex ? XGL_COLOR_INDEX : XGL_COLOR_RGB,
		0);

    if(mgxgl->xglwin == NULL) {
	OOGLError(1, "mgxgl_newwindow: xgl_window_raster_device_create failed");
	return -1;
    }

    mgxgl->xglctx = mgxgl->ctx_win = xgl_3d_context_create(
			XGL_CTX_DEVICE,		mgxgl->xglwin,
			XGL_CTX_DEFERRAL_MODE,	XGL_DEFER_ASTI,
			XGL_CTX_VDC_ORIENTATION,XGL_Y_DOWN_Z_AWAY,
			NULL);
    mgxgl->already = 0;

    if(mgxgl->ctx_win == NULL) {
	OOGLError(1, "mgxgl_newwindow: xgl_3d_context_create failed");
	return -1;
    }

    xgl_object_get(MGXGL->xglctx, XGL_CTX_VIEW_TRANS, &MGXGL->xglview);
    xgl_object_get(MGXGL->xglctx, XGL_CTX_LOCAL_MODEL_TRANS, &MGXGL->xglmodel);
    xgl_object_get(MGXGL->xglctx, XGL_CTX_GLOBAL_MODEL_TRANS, &MGXGL->xglcam);

    MGXGL->xglidentity = xgl_transform_create(0);

    mgxgl->born = 1;
    return 1;
}

void
mgxgl_worldbegin()
{
    register struct mgxglcontext *mgxgl = MGXGL;
    register int frameacts;

    if(!mgxgl->born) {
	OOGLError(1, "mgxgl_worldbegin: window not created yet");
	return;
    }

    if(!mgxgl->useX11) {
	register Rect *r = (Rect *)xv_get(mgxgl->xvwin, XV_RECT);
	if(r) {
	    WnPosition wp;
	    wp.xmin = r->r_left;
	    wp.ymin = r->r_top;
	    wp.xmax = wp.xmin + r->r_width - 1;
	    wp.ymax = wp.ymin + r->r_height - 1;
	    WnSet(MGC->win, WN_CURPOS, &wp, WN_END);
	}
    }

    if(MGC->opts & MGO_DOUBLEBUFFER && mgxgl->already) {
	/*
	 * Double-buffering, and not the first frame.
	 * Let's write in the back-buffer, i.e. memory raster.
	 * We need it to have the same colormap and size as the window.
	 */

	if(mgxgl->xglmem == NULL) {
	    xgl_object_get(mgxgl->xglwin, XGL_RAS_COLOR_MAP, &mgxgl->xglcmap);
	    mgxgl->xglmem = xgl_memory_raster_device_create(
			XGL_RAS_COLOR_TYPE,
			mgxgl->colorindex ? XGL_COLOR_INDEX : XGL_COLOR_RGB,
			XGL_RAS_COLOR_MAP, mgxgl->xglcmap,
			XGL_RAS_DEPTH, 32,
			0);
	}
	xgl_object_set(mgxgl->xglmem,
		XGL_RAS_WIDTH, MGC->win->cur.xmax - MGC->win->cur.xmin + 1,
		XGL_RAS_HEIGHT, MGC->win->cur.ymax - MGC->win->cur.ymin + 1,
		0);
	if(mgxgl->ctx_mem == NULL) {
	    mgxgl->ctx_mem = xgl_3d_context_create(
			XGL_CTX_DEVICE,		mgxgl->xglmem,
			XGL_CTX_DEFERRAL_MODE,	XGL_DEFER_ASTI,
			XGL_CTX_VDC_ORIENTATION,XGL_Y_DOWN_Z_AWAY,
			XGL_CTX_VIEW_TRANS, mgxgl->xglview,
			XGL_CTX_LOCAL_MODEL_TRANS, mgxgl->xglmodel,
			0);
	}
	mgxgl->xglctx = mgxgl->ctx_mem;
    } else {
	mgxgl->xglctx = mgxgl->ctx_win;
    }

    /*
     * Code this to allow choosing a viewport less than full window.
     * Default XGL_CTX_VDC_MAP := XGL_VDC_MAP_ASPECT (i.e. map entire win).
     */
    if(MGC->win->flag & WNF_HASVP) {
	Xgl_bounds_f2d xvport;

	xvport.xmin = MGC->win->viewport.xmin;
	xvport.xmax = MGC->win->viewport.xmax;
	xvport.ymin = MGC->win->viewport.ymin;
	xvport.ymax = MGC->win->viewport.ymax;
	xgl_object_set(mgxgl->xglctx,
			XGL_CTX_DC_VIEWPORT, &xvport,
			XGL_CTX_VDC_MAP, XGL_VDC_MAP_OFF,
			NULL);
    } else {
	xgl_object_set(mgxgl->xglctx,
			XGL_CTX_VDC_MAP, XGL_VDC_MAP_ASPECT,
			NULL);
    }

    /*
     * Prepare to start new frame.  Clear?  Initialize XGL Z-buffer?
     */
    frameacts = XGL_CTX_NEW_FRAME_CLEAR;

    if(MGC->opts & MGO_INHIBITBACKGROUND)
	frameacts = 0;

    if(MGC->opts & MGO_HIDDEN) {
	mgxgl_doZbuffer();
	frameacts |= XGL_CTX_NEW_FRAME_HLHSR_ACTION;
    } else { 
	mgxgl_dontZbuffer();
    }

    xgl_object_set(mgxgl->xglctx,
		XGL_CTX_BACKGROUND_COLOR, &MGC->background,
		XGL_CTX_NEW_FRAME_ACTION, frameacts,
		0);

    mgxgl->repaint = 0;		/* We're repainting, cancel flag */
    /*
     * If single-buffered (or first time), clear window to background color.
     * Possibly initialize primitive list if sorting.
     */

    xgl_context_new_frame( mgxgl->xglctx );

    mgxgl_install_camera();
}


/*
 * mgxgl_install_camera()
 * Not a public entry point; called from mgxgl_worldbegin().
 * Interprets the current Camera structure and configures XGL appropriately.
 * Specifically:
 *	establish clipping planes
 *	establish camera projection
 *	set initial identity object->world transformation
 */
mgxgl_install_camera()
{
    Xgl_bounds_f3d	vdc;
    Transform proj;
    Transform model;
    static Xgl_bounds_f3d vdc_bounds = { -1,1,  -1,1,  -1,1  };

    CamViewProjection(MGC->cam, proj);
    CamViewWorld(MGC->cam, model);

    xgl_transform_write(MGXGL->xglcam, model);
    xgl_transform_write(MGXGL->xglview, proj);

    xgl_object_set(MGXGL->xglctx,
		XGL_CTX_VIEW_CLIP_BOUNDS, &vdc_bounds,
		XGL_CTX_VDC_WINDOW, &vdc_bounds,
		0);
#ifdef notyet
    CamGet(MGC->cam, CAM_W2C, MGXGL->w2c);
#endif notyet
}

/*
 * Swap buffers if double-buffering.
 * When implemented, this'll mean copying the image (rendered into
 * a raster buffer) onto the window.
 */
void
mgxgl_worldend()
{
    mgxgl_sync();
    if(MGXGL->xglctx == MGXGL->ctx_mem && !(MGC->opts & MGO_INHIBITSWAP) ) {
	/*
	 * Double-buffering -- time to "swap" buffers.
	 * Copy memory raster to window.
	 */
	xgl_context_copy_raster(MGXGL->ctx_win, NULL,NULL, MGXGL->xglmem);
    }
    MGXGL->already = 1;
}

void
mgxgl_sync()
{
    xgl_context_post(MGXGL->xglctx, FALSE);	/* Send & wait */
}


void
mgxgl_identity()
{
    xgl_transform_identity(MGXGL->xglmodel);
    mg_identity();
}

void
mgxgl_settransform(Transform T)
{
    xgl_transform_write(MGXGL->xglmodel, T);
    TmCopy(T, MGC->xstk->T);
}

void
mgxgl_transform(Transform T)
{
    mg_transform(T);
    xgl_transform_write(MGXGL->xglmodel, MGC->xstk->T);
}


int
mgxgl_poptransform()
{
    register int ok = mg_poptransform();
    xgl_transform_write(MGXGL->xglmodel, MGC->xstk->T);
    return ok;
}


int
mgxgl_pushappearance()
{
    xgl_context_push(MGXGL->xglctx, NULL);
    mg_pushappearance();
}

int
mgxgl_popappearance()
{
    register struct mgastk *n;

    if((n = MGC->astk->next) == NULL)
	return;
    xgl_context_pop(MGXGL->xglctx);
    if(MGC->astk->light_seq != n->light_seq) {
	mgxgl_installlights(n->lighting.lights, 1);
	mgxgl_installlmodel(&n->lighting);
    }
    mg_popappearance();
}


int
mgxgl_setCamera(Camera *cam)
{
    /*
     * Use camera by reference.
     * Be sure to RefIncr() before CamDelete() -- they might pass us
     * the same camera we're already using.
     */
    RefIncr((Ref *)cam);
    CamDelete(MGC->cam);
    MGC->cam = cam;
}


extern void mgxgl_polygon();
extern void mgxgl_mesh();
extern void mgxgl_polylist();
extern void mgxgl_line();
extern void mgxgl_polyline();

/*
 * Utility routines.
 */
void
_mgxgl_xv_resize() {
    MGXGL->resize = 1;
    xgl_window_raster_resize(MGXGL->xglwin);
}

void
_mgxgl_xv_repaint() {
    MGXGL->repaint = 1;
    write(2, "XView repaint time.\n", 20);
}

void
mgxgl_wantbuf(int nbytes)
{
    if(MGXGL->bufsiz < nbytes && nbytes > 0) {
	MGXGL->xbuf = (void *)OOGLRenewNE(char, MGXGL->xbuf, nbytes, "MG buffer for XGL data");
	MGXGL->bufsiz = nbytes;
    }
}

struct mgfuncs mgxglfuncs = {
	MGD_XGL,
	mgdevice_XGL,
	mgxgl_feature,
	(mgcontext *(*)())mgxgl_ctxcreate,
	mgxgl_ctxdelete,
	(void (*)())mgxgl_ctxset,
	mgxgl_ctxget,
	mgxgl_ctxselect,
	mgxgl_sync,
	mgxgl_worldbegin,
	mgxgl_worldend,
	mg_reshapeviewport,	/* inherit reshapeviewport from mg.c */
	mgxgl_settransform,
	mg_gettransform,	/* and mg_gettransform */
	mgxgl_identity,
	mgxgl_transform,
	mg_pushtransform,	/* and mg_pushtransform */
	mgxgl_poptransform,
	mgxgl_pushappearance,
	mgxgl_popappearance,
	mgxgl_setappearance,
	mg_getappearance,
	mgxgl_setCamera,
	mgxgl_polygon,
	mgxgl_polylist,
	mgxgl_mesh,
	mgxgl_line,
	mgxgl_polyline
};
