/*
 * NOTES:
 * 1. mrti() assumes unsigned short is 2 bytes, MSB first
 *    this may have to change for other architecture
 * 2. will it be possible to increase the encoded float array length
 *    by "adding l" to create 256 aggregates of 3 or 4 floats each, as
 *    opposed to 256 straight floats? (see appendix C, pg 163 of the
 *    'big' renderman manual)
 * 3. the name mrti should be changed to mgri (it sounds better)
 */

#include "mgribtoken.h"
#include <stdio.h>

#include "mg.h"
#include "mgribP.h"
#include "mgrib.h"

#ifdef test
#include <stdio.h>
#endif

#define BUFFERSIZE 1024*128
#define TMPSIZE 8192
#define SECTIONWIDTH 70        /* width of a section header */

#define STRINGDEF	0315
#define STRINGREF	0317
#define LNGSTRINGENCODE	0241   /* length is unsigned short */
#define STRINGENCODE	0220
#define RIREQDEF	0314
#define RIREQREF	0246
#define FLOATARRAYDEF	0310
#define INTEGER		0201   /* unsigned short */
#define FLOAT		0244

void binary_token(int a1, va_list *alist);
void ascii_token(int a1, va_list *alist);

unsigned char tokenbuffer[BUFFERSIZE];
char tmp[TMPSIZE];
unsigned char *ptr;

struct _table table[] = {
    {"", 0, 0, 0},		/* mr_NULL */

    /* Ri Requests */
    {"AttributeBegin",		14,  0, 0},
    {"AttributeEnd",		12,  1, 0},
    {"TransformBegin",		14,  2, 0},
    {"TransformEnd",		12,  3, 0},
    {"Identity",		 8,  4, 0},
    {"ConcatTransform",		15,  5, 0},
    {"Surface",			 7,  6, 0},
    {"ShadingInterpolation",	20,  7, 0},
    {"Color",		 	 5,  8, 0},
    {"Opacity",			 7,  9, 0},
    {"Polygon",			 7, 10, 0},
    {"PatchMesh",		 9, 11, 0},
    {"Format",			 6, 12, 0},
    {"Projection",		10, 13, 0},
    {"Clipping",		 8, 14, 0},
    {"WorldBegin",		10, 15, 0},
    {"WorldEnd",		 8, 16, 0},
    {"Display",			 7, 17, 0},
    {"ScreenWindow",		12, 19, 0},
    {"LightSource",		11, 20, 0},
    {"Sphere",			 6, 21, 0},
    {"Translate",		 9, 22, 0},
    {"Rotate",			 6, 23, 0},
    {"Cylinder",		 8, 24, 0},
    
    /* following are reserved - do not add */
    /* or remove fields, just change them! */
    {"", 0, 255, 0},
    {"", 0, 255, 0},
    {"", 0, 255, 0},
    {"", 0, 255, 0},
    {"", 0, 255, 0},

    /* Strings - we start these out at position 30 */
    {"P",			 1,  0, 0},
    {"N",			 1,  1, 0},
    {"Cs",            	 	 2,  2, 0},
    {"Pw",			 2,  3, 0},
    {"Os",			 2,  4, 0},
    {"plastic",			 7,  5, 0},
    {"hplastic",		 8,  6, 0},
    {"eplastic",		 8,  7, 0},
    {"heplastic",		 9,  8, 0},
    {"constant",		 8,  9, 0},
    {"ambientlight",		12, 10, 0},
    {"lightcolor",		10, 11, 0},
    {"distantlight",		12, 12, 0},
    {"intensity",		 9, 13, 0},
    {"file",			 4, 14, 0},
    {"rgb",			 3, 15, 0},
    {"rgba",			 4, 16, 0},
    {"Ka",			 2, 17, 0},
    {"Kd",			 2, 18, 0},
    {"Ks",			 2, 19, 0},
    {"specularcolor",		13, 20, 0}, 
    {"roughness",		 9, 21, 0},
    {"fov",			 3, 22, 0},
    {"perspective",		11, 23, 0},
    {"to",			 2, 24, 0}
};


/* quick copy routine w/ ptr update */
void cat(unsigned char *s, char *a)
{
    while(*(s++)=(*(a++))) ptr++;
}


/* reset the ptr & tokkenbuffer */
void mrti_reset()
{
	ptr = tokenbuffer;
	tokenbuffer[0] = (char)0;
}


/* process variable size token list */
void mrti(int a1, ... )
{
    va_list alist;
    static unsigned char *halfWayPoint = NULL;
    
    if(!halfWayPoint)
        halfWayPoint = (unsigned char *)((int)tokenbuffer + BUFFERSIZE/2);
    
    /* when the buffer is about half full, flush it */
    if(ptr>halfWayPoint) {
    	mgrib_flushbuffer();
    }
    
    va_start(alist, a1);
 
    switch(_mgribc->render_device & (RMD_BINARY|RMD_ASCII)) {
	case RMD_BINARY: binary_token(a1, &alist); break;
	case RMD_ASCII:  ascii_token(a1, &alist); break;
    }
    va_end(alist);
}

/* return true when supplied token requires its own line (ascii) */
int lineInitializer(int token)
{
    if(token<STRINGBASE||token==mr_comment||token==mr_section||
      (token>=mr_P && token<=mr_Os)) return 1;
    else return 0;
}

/* ASCII SUBPROCESSING */
void ascii_token(int token, va_list *alist)
{
int i;
int count, number;
int subsize;
static int arraysize;
static int expectSubArray=0;
static char astring[128];
double nextfloat; /* va_arg converts floats to doubles */
float *floatptr;
char *s;
int len, add;

    do {
    
    if(expectSubArray && (token!=mr_subarray3)) {
    	/* ERROR */
    }
    
    /* check to see if we need to start a new line */
    if(lineInitializer(token)) *(ptr++)='\n';
    
    switch(token) {
	
    case mr_section:
        s = va_arg(*alist, char*);
        len = strlen(s);
	*(ptr++)='\n';
	cat(ptr,"# ");
        cat(ptr,s);
	cat(ptr," ");
	
	len = SECTIONWIDTH - 3 - len; /* 3 is from 3 added characters */
        while(len-->0) *(ptr++)='*';
	break;

    case mr_comment:
	s = va_arg(*alist, char*);
	cat(ptr,"# ");
	cat(ptr,s);
	break;

    case mr_nl:
    	/* check for space */
	if(*(ptr-1)=' ') ptr--;
    	*(ptr++)='\n';
	break;
	
    case mr_array:
    	arraysize = va_arg(*alist, int);
	*(ptr++)='[';
	for(i=0;i<arraysize;i++) {
	    nextfloat = va_arg(*alist, double);
	    sprintf(astring,"%g ",nextfloat);
	    cat(ptr, astring);
	}
	*(--ptr)=0; /* get rid of space */
	cat(ptr,"] ");
	break;
	
    case mr_buildarray:
    	arraysize = va_arg(*alist, int);
	expectSubArray = 1;
	*(ptr++)='[';
	break;

    case mr_parray:
    {
    	int size;
    	size = va_arg(*alist, int);
	*(ptr++)='[';
	/* if arraysize<0 then ERROR */
	floatptr = va_arg(*alist, float*);
	for(i=0;i<size;i++) {
	    sprintf(astring,"%g ",*(floatptr++));
	    cat(ptr,astring);
	}
	*(--ptr)=0; /* get rid of unwanted space */
	cat(ptr,"] ");
	break;
    }
    
    case mr_subarray3:
	arraysize-=3;
	/* if arraysize<0 then ERROR */
    	floatptr = va_arg(*alist, float*);
    	sprintf(astring,"%g %g %g   ",
	    *(floatptr),*(floatptr+1),*(floatptr+2));
	cat(ptr,astring);
	if(arraysize<=0) {
	    expectSubArray = 0;
	    *(ptr-=3)=0; /* get rid of unwanted spaces */
	    cat(ptr,"] ");
	}
	break;

    case mr_int:
        number = va_arg(*alist, int);
	sprintf(astring,"%d ",number);
	cat(ptr,astring);
	break;

    case mr_float:
        nextfloat = va_arg(*alist, double);
	sprintf(astring,"%g ",nextfloat);
	cat(ptr,astring);
	break;
	
    case mr_intseq:
    	/* OBSOLETE */
    	count = va_arg(*alist, int);
	for(i=0;i<count;i++) {
	    number = va_arg(*alist, int);
	    sprintf(astring,"%d ",number);
	    cat(ptr,astring);
	}
	*(--ptr)=0;
	break;

    case mr_string:
    {
    	char *string;
	string = va_arg(*alist, char *);
	*(ptr++)='"';
	cat(ptr,string);
	*(ptr++)='"';
	*(ptr++)=' ';
	break;
    }

    /* these rquests require a trailing line feed */
    case mr_attributebegin:
    case mr_attributeend:
    case mr_transformbegin:
    case mr_transformend:
    case mr_identity:
    case mr_polygon:
    case mr_worldbegin:
    case mr_worldend:
	cat(ptr,table[token].name);
	/*
	*(ptr++) = '\n';
	*/
	break;

    /* these requests require a trailing space */
    case mr_color:
    case mr_opacity:
    case mr_format:
    case mr_display:
    case mr_screenwindow:
    case mr_clipping:
    case mr_concattransform:
    case mr_projection:
    case mr_lightsource:
    case mr_surface:
    case mr_sphere:
    case mr_patchmesh:
    case mr_rotate:
    case mr_translate:
    case mr_cylinder:
    	cat(ptr,table[token].name);
	*(ptr++)=' ';
	break;
	
    /* anything left over should be a string  */
    /* which requires quotes & trailing space */
    /* (THIS SHOULD BE MOVED OUT OF SWITCH)  */
    default:
        *(ptr++)='"';
    	cat(ptr,table[token].name);
	*(ptr++)='"';
	*(ptr++)=' ';
	break;

    }
    		
    } while ((token=va_arg(*alist, int))!=mr_NULL);
}


/* BINARY SUBPROCESSING */
/* NOTE: some hardware will not support copying data types > 1 byte into an
 * arbitrary (unaligned) location. i.e.
 *    *(unsigned short *)ptr = anUnsignedVariable
 * may be an illegal instruction. Therefore, macros which will copy these values
 * one byte at a time follow here and are used where appropriate
 */

#define COPYUSHORT( value )	*ptr++ = ((char *)&value)[0];\
				*ptr++ = ((char *)&value)[1]
#define COPYFLOAT( value )	*ptr++ = ((char *)&value)[0];\
				*ptr++ = ((char *)&value)[1];\
				*ptr++ = ((char *)&value)[2];\
				*ptr++ = ((char *)&value)[3]

void binary_token(int token, va_list *alist)
{
int i;
static int expectSubArray=0;
static int arraysize;
float *floatptr;

    if(expectSubArray && (token!=mr_subarray3)) {
    	/* ERROR */
    }

    do {
    
    if(token>=STRINGBASE & token<mr_array) {
    	/* The token is a string defintion/request */
	if(!table[token].defined) {
	    *(ptr++)=STRINGDEF;
	    *(ptr++)=table[token].reqn;
	    if(table[token].len<16) {
	    	*(ptr++)=STRINGENCODE+table[token].len;
	    } else {
	    	unsigned short length;
		length = (unsigned short) table[token].len;
	    	*(ptr++)=LNGSTRINGENCODE;
		COPYUSHORT(length);
	    }
	    cat(ptr,table[token].name);
	    table[token].defined = 1;
	}
	*(ptr++)=STRINGREF;
	*(ptr++)=table[token].reqn;
    }
    
    else switch(token) {
    
    /* THIS SHOULD BE MOVE BEFORE SWITCH */
    /* AND TURNED INTO A SPECIAL CASE    */
    case mr_attributebegin:
    case mr_attributeend:
    case mr_transformbegin:
    case mr_transformend:
    case mr_identity:
    case mr_polygon:
    case mr_worldbegin:
    case mr_worldend:
    case mr_color:
    case mr_format:
    case mr_display:
    case mr_screenwindow:
    case mr_clipping:
    case mr_concattransform:
    case mr_projection:
    case mr_lightsource:
    case mr_surface:
    case mr_sphere:
    case mr_patchmesh:
    case mr_translate:
    case mr_rotate:
    case mr_cylinder:
	if(!table[token].defined) {
	    /* we must define it */
	    *(ptr++)=RIREQDEF;
	    *(ptr++)=table[token].reqn;
	    if(table[token].len<16) {
	    	*(ptr++)=STRINGENCODE+table[token].len;
	    } else {
	    	unsigned short length;
		length = (unsigned short) table[token].len;
		*(ptr++)=LNGSTRINGENCODE;
		COPYUSHORT(length);
	    }
	    cat(ptr,table[token].name);
	    table[token].defined = 1;
	}
	*(ptr++)=RIREQREF;
	*(ptr++)=table[token].reqn;
    	break;
    
    case mr_array:
    {
        unsigned char arraycount;
	unsigned char *p;
	float f;
    	arraycount = (unsigned char)va_arg(*alist, int);
	*(ptr++)=FLOATARRAYDEF;
	*(ptr++)=arraycount;
	for(i=0;i<arraycount;i++) {
	    /* will float be the same on each platform? (ieee?)*/
	    /* IS THIS NEXT STEP LEGAL?!? */
	    f = (float) va_arg(*alist, double);
	    COPYFLOAT(f);
	}
	break;
    }
    
    case mr_buildarray:
    {
    	unsigned char arraycount;
	arraysize = va_arg(*alist, int);
	arraycount = (unsigned char)arraysize;
	*(ptr++)=FLOATARRAYDEF;
	*(ptr++)=arraycount;
	expectSubArray = 1;
	break;
    }
    
    case mr_subarray3:
    {
    	arraysize-=3;
	if(arraysize<0) /* error */;
    	floatptr = va_arg(*alist, float*);
	bcopy((char *)floatptr, (char *)ptr, 3*sizeof(float));
	ptr+=3*sizeof(float);
	if(arraysize<=0) {
	    expectSubArray = 0;
	}
	break;
    }
  
    case mr_parray:
    {
    	unsigned char arraycount;
	arraycount = (unsigned char)va_arg(*alist, int);
	*(ptr++)=FLOATARRAYDEF;
	*(ptr++)=arraycount;
	floatptr = va_arg(*alist, float*);
	bcopy((char *)floatptr, (char *)ptr, arraycount*sizeof(float));
	ptr+=arraycount*sizeof(float);
	if(arraysize<=0) {
	    expectSubArray = 0;
	}
	break;
    }

    case mr_int:
    {
    	unsigned short number;
        number = (unsigned short)va_arg(*alist, int);
	*(ptr++) = INTEGER;
	COPYUSHORT(number);
	break;
    }

    case mr_float:
    {
    	float afloat;
        afloat = (float)va_arg(*alist, double);
	*(ptr++) = FLOAT;
	COPYFLOAT(afloat);
	break;
    }
	
    case mr_intseq:
    {
    	unsigned short number;
	int  count;
	count = va_arg(*alist, int);
	for(i=0;i<count;i++) {
	    number = (unsigned short)va_arg(*alist, int);
	    *(ptr++) = INTEGER;
	    COPYUSHORT(number);
	}
	break;
    }

    case mr_string:
    {
        unsigned short length;
	char *string;
	string = va_arg(*alist, char *);
	length = (unsigned short) strlen(string);
	if(length<16) {
	    *(ptr++)=STRINGENCODE+(char)length;
	} else {
	    *(ptr++)=LNGSTRINGENCODE;
	    COPYUSHORT(length);
	}
	cat(ptr,string);
	break;
    }
    
    case mr_nl:
    	/* temporary for debugging (puts tokens on spereate lines */
        printf("\n");
	break;
    
    } /* switch */    
    } while ((token=va_arg(*alist, int))!=mr_NULL);
}

#ifdef test
main()
{
	int l,i;
	unsigned char *scan;
	float f[3];
	
	float m1[16]={1.,0.,0.,0.,0.,1.,0.,0.,0.,0.,-1.,0.,0.,0.,0.,1.};
	
	float m2[16]={1.,0.,0.,0.,0.,1.,0.,0.,0.,0.,1.,0.,0.,0.,-1.3,1.};

	f[0]=0; f[1]=2; f[2]=4;
	
	mrti_reset();

	mrti(mr_display, mr_string, "g.tiff", mr_file,
		mr_rgb, mr_nl, mr_NULL);
	
	mrti(mr_format, mr_int, 200, mr_int, 150,
		mr_float, 1., mr_NULL);
	
	mrti(mr_screenwindow, mr_float, -1.33, mr_float, 1.33,
		mr_float, -1., mr_float, 1., mr_NULL);
	
	mrti(mr_clipping, mr_float, .1, mr_float, 10., mr_NULL);
	
	mrti(mr_projection, mr_perspective, mr_fov, mr_float, 60., mr_NULL);
	
	mrti(mr_identity, mr_NULL);
	
	mrti(mr_concattransform, mr_parray, 16, m1);
	
	mrti(mr_concattransform, mr_parray, 16, m2);
	
	mrti(mr_lightsource, mr_ambientlight, mr_int, 0,
		mr_lightcolor, mr_array, 3, .6, .6, .6, mr_NULL);
	
	mrti(mr_lightsource, mr_distantlight, mr_int, 1,
		mr_intensity, mr_float, 1.,
		mr_lightcolor, mr_array, 3, 1., 1., 1.,
		mr_to, mr_array, 3, 0., 0., -1., mr_NULL);
	
	mrti(mr_color, mr_array, 3, 1., 1., 1., mr_NULL);
	
	mrti(mr_surface, mr_plastic, mr_Ka, mr_float, 1.,
		mr_Kd, mr_float, 1., mr_Ks, mr_float, 1.,
		mr_specularcolor, mr_array, 3, .6, .2, 0.,
		mr_roughness, mr_float, 1., mr_NULL);
	
	mrti(mr_worldbegin, mr_NULL);
	
	mrti(mr_concattransform, mr_array, 16,
		1., 0., 0., 0.,
		0., 1., 0., 0.,
		0., 0., 1., 0.,
		0., 0., 0., 1., mr_NULL);
		
	#ifdef ascii
	printf("test is:\n%s\n", tokenbuffer);
	#endif
	
	#ifdef 0
	#ifdef binary
	printf("enter length:");
	scanf("%d",&l);
	for(i=0;i<l;i++)
	    printf("%d\t%d\t%o\t%c\n",i,(int)tokenbuffer[i],
	    	(int)tokenbuffer[i], tokenbuffer[i]);
	#endif
	#endif
	
	scan = tokenbuffer;
	while(scan<ptr) {
	   unsigned char c;
	   c=*(scan++);
	   fprintf(stderr,"%o\t%c\n",c,c);
	   putc(c,stdout);
	}
	    
}
#endif

