/* Copyright (c) 1992 The Geometry Center; University of Minnesota
   1300 South Second Street;  Minneapolis, MN  55454, USA;
   
This file is part of geomview/OOGL. geomview/OOGL is free software;
you can redistribute it and/or modify it only under the terms given in
the file COPYING, which you should have received along with this file.
This and other related software may be obtained via anonymous ftp from
geom.umn.edu; email: software@geom.umn.edu. */

/* Authors: Stuart Levy, Tamara Munzner, Mark Phillips */

/*
 * lisp.c: minimal lisp interpreter
 */

#include <stdio.h>
#include <string.h>
#include "ooglutil.h"
#include "fsa.h"
#include "lisp.h"
#include "handleP.h"

static LObject _nil = { LSTRING, NULL };
static LObject *nil = &_nil;
LObject *Lnil = &_nil;

#define MAXFUNCS 200
static LObjectFunc	funcs[MAXFUNCS];

static Fsa		func_fsa;
static LObjectFunc	funcval(LObject *obj);
static LObject *	nilfunc(Pool *p);
static char *		verbose;

typedef int (*PFI)();
typedef char String[1024];

static int noop(){}

#define MAXTYPES 20

PFI freefunc(int type);
PFI copyfunc(int type);
PFI stringfunc(int type);

static PFI freefuncs[MAXTYPES];
static PFI copyfuncs[MAXTYPES];
static PFI stringfuncs[MAXTYPES];

int typeok(int type);
int *intpdup(int *ptr);
float *floatpdup(float *ptr);

static char *delims = "()";
#define NEXTTOKEN(tok,fp) tok = fdelimtok( delims, fp, 0 )

/*-----------------------------------------------------------------------
 * Function:	NewLObject
 * Description:	allocate & initialize a new LObject
 * Args:	type: type of LObject
 * Returns:	ptr to new LObject
 * Author:	mbp
 * Date:	Wed Jan 15 15:12:20 1992
 * Notes:	
 */
LObject *NewLObject(int type)
{
  LObject *obj = (LObject*)malloc(sizeof(LObject));
  obj->type = type;
  obj->val = NULL;
}

/*-----------------------------------------------------------------------
 * Function:	FreeLObject
 * Description:	(deep) free storage associated with an LObject
 * Args:	*obj: the object to free
 * Returns:	
 * Author:	mbp
 * Date:	Wed Jan 15 15:13:01 1992
 * Notes:	
 */
void FreeLObject(LObject *obj)
{
  PFI f;

  if (obj && obj != Lnil) {
    if (obj->val && (f=freefunc(obj->type)))
      (*f)(obj->val);
    free(obj);
  }
}

/*-----------------------------------------------------------------------
 * Function:	NewLList
 * Description:	allocate & initialize a new LList
 * Returns:	ptr to new LList
 * Author:	mbp
 * Date:	Wed Jan 15 15:13:34 1992
 * Notes:	
 */
LList *NewLList()
{
  LList *new = (LList*)malloc(sizeof(LList));
  new->cdr = NULL;
}

/*-----------------------------------------------------------------------
 * Function:	FreeLList
 * Description:	(deep) free storage associated with an LList
 * Args:	*list: LList to free
 * Returns:	
 * Author:	mbp
 * Date:	Wed Jan 15 15:13:53 1992
 * Notes:	
 */
void FreeLList(LList *list)
{
  if (!list) return;
  if (list->cdr) FreeLList(list->cdr);
  FreeLObject(list->car);
  free(list);
}

/*-----------------------------------------------------------------------
 * Function:	LListAppend
 * Description:	append an LObject to an LList
 * Args:	*list: the LList to append to 
 *		*obj: the LObject to be appended
 * Returns:	new head of LList
 * Author:	mbp
 * Date:	Wed Jan 15 15:14:09 1992
 * Notes:	return value equals list unless list was NULL on entry
 */
LList *LListAppend(LList *list, LObject *obj)
{
  LList *l, *new = NewLList();

  new->car = obj;
  l = list;
  if (l) {
    while (l->cdr) l = l->cdr;
    l->cdr = new;
    return list;
  }
  return new;
}

/*-----------------------------------------------------------------------
 * Function:	LListCopy
 * Description:	(deep) copy an LList
 * Args:	*list: the LList to copy
 * Returns:	ptr to new copy of list
 * Author:	mbp
 * Date:	Wed Jan 15 15:17:05 1992
 * Notes:	
 */
LList *LListCopy(LList *list)
{
  LList *new;

  if (! list) return NULL;
  new = NewLList();
  if (list->car)
    new->car = LObjectCopy(list->car);
  new->cdr = LListCopy(list->cdr);
  return new;
}

/*-----------------------------------------------------------------------
 * Function:	LObjectCopy
 * Description:	(deep) copy an LObject
 * Args:	*obj: LObject to copy
 * Returns:	ptr to copy of obj
 * Author:	mbp
 * Date:	Wed Jan 15 15:17:28 1992
 * Notes:	
 */
LObject *LObjectCopy(LObject *obj)
{
  LObject *new;
  PFI f;

  if (!obj) return NULL;
  new = NewLObject(obj->type);
  if (f=copyfunc(obj->type))
    new->val = (void*)((*f)(obj->val));
  return new;
}

/*-----------------------------------------------------------------------
 * Function:	*LSexpr
 * Description:	parse a sexpr from a pool
 * Args:	*p: the pool
 * Returns:	ptr to parsed LObject
 * Author:	mbp
 * Date:	Wed Jan 15 15:17:47 1992
 * Notes:	reads one complete sexpr from p
 */
LObject *LSexpr(Pool *p)
{
  LObject *obj;
  char *tok;
  int c;

  NEXTTOKEN(tok,p->inf);
  if(tok == NULL)
    return Lnil;
  if (*tok == '(' && tok[1] == '\0') {
    obj = NewLObject(LLIST);
    while ( (c = fnextc(p->inf,0)) != ')' && c != EOF )
      obj->val = (void*) LListAppend((LList*)(obj->val), LSexpr(p));
    NEXTTOKEN(tok,p->inf);
  } else {
    obj = NewLObject(LSTRING);
    obj->val = strdup(tok);
  }
  return obj;
}

/*-----------------------------------------------------------------------
 * Function:	LEvalSexpr
 * Description:	Parse and evaluate a sexpr from a pool
 * Args:	*p: the pool
 * Returns:	
 * Author:	mbp
 * Date:	Wed Jan 15 18:57:51 1992
 * Notes:	
 */
LObject *LEvalSexpr(Pool *p)
{
  LObject *obj, *lcar;
  LObjectFunc f;
  char *tok;

  NEXTTOKEN(tok,p->inf);
  if(tok == NULL)
    return Lnil;
  if (*tok == '(' && tok[1] == '\0') {
    if (fnextc(p->inf,0) == ')') {
      NEXTTOKEN(tok,p->inf);
      return nil;
    }
    lcar = LEvalSexpr(p);
    f = funcval(lcar);
    if (f) obj = (*f)(p);
    if (fexpecttoken(p->inf,")"))
      printf("Error: ')' missing\n");
  } else {
    obj = NewLObject(LSTRING);
    obj->val = strdup(tok);
  }
  return obj;
}

/*-----------------------------------------------------------------------
 * Function:	LObjectString
 * Description:	return a string representation of an LObject
 * Args:	*obj: the LObject
 * Returns:	ptr to dynamically allocated copy of string
 * Author:	mbp
 * Date:	Wed Jan 15 15:20:37 1992
 * Notes:	the caller owns the space pointed to by returned value
 *		WARNING: present implementation will barf if more
 *		than 1024 chars are required!
 */
char *LObjectString(LObject *obj)
{
  String buf;
  switch (obj->type) {
  case LLIST:
    return LListString(LLISTVAL(obj));
  case LSTRING:
    if (obj->val)
      return strdup(LSTRINGVAL(obj));
    else
      return "nil";
  case LINT:
    sprintf(buf, "%1d", LINTVAL(obj));
    return strdup(buf);
  case LFLOAT:
    sprintf(buf, "%1f", LFLOATVAL(obj));
    return strdup(buf);
  default:
    {
      PFI f;
      if (f=stringfunc(obj->type))
	return (char*)((*f)(obj->val));
      else
	return "<val>";
    }
  }
}

/*-----------------------------------------------------------------------
 * Function:	LListString
 * Description:	return a string representation of an LList
 * Args:	*l: the LList
 * Returns:	ptr to dynamically allocated copy of string
 * Author:	mbp
 * Date:	Wed Jan 15 15:21:28 1992
 * Notes:	the caller owns the space pointed to by returned value
 *		WARNING: present implementation will barf if more
 *		than 1024 chars are required!
 */
char *LListString(LList *l)
{
  String buf;
  char *s;
  sprintf(buf,"(");
  while (l) {
    strcat(buf, " ");
    s = LObjectString(l->car);
    strcat(buf, s);
    free(s);
    strcat(buf, " ");
    l = l->cdr;
  }
  strcat(buf, ")");
  return strdup(buf);
}

#if LEVAL_WORKS

/*-----------------------------------------------------------------------
 * Function:	LEval
 * Description:	Evaluate an LObject
 * Args:	*obj: the LObject to evaluate
 * Returns:	obj's value (another LObject)
 * Author:	mbp
 * Date:	Wed Jan 15 15:22:05 1992
 * Notes:	
 */
LObject *LEval(LObject *obj)
{
  if (obj->type==LLIST)
    return LListEval(LLISTVAL(obj));
  else
    return obj;
}

/*-----------------------------------------------------------------------
 * Function:	LListEval
 * Description:	evaluate an LList
 * Args:	*list: the LList to evaluate
 * Returns:	list's value
 * Author:	mbp
 * Date:	Wed Jan 15 15:22:39 1992
 * Notes:	list is evaluated by treating the car as the name
 *		of a function and passing the cdr to that function
 *		as args.  Returns nil if car is not a known function
 *		(as defined by a call to LDefun).
 *
 *  WARNING:  This is currently broken!  Don't use it !!!
 *   It passes a list to the function, but functions are
 *   currently defined to take Pool *'s !!
 */
LObject *LListEval(LList *list)
{
  LObjectFunc func;
  if (!list) return nil;
  return funcval(LEval(list->car))(list->cdr);
}

#endif /* LEVAL_WORKS */

/*-----------------------------------------------------------------------
 * Function:	funcval
 * Description:	return the function-value of an LSTRING LObject
 * Args:	*obj:
 * Returns:	ptr to the function-value
 * Author:	mbp
 * Date:	Wed Jan 15 15:24:29 1992
 * Notes:	internal use only
 */
static LObjectFunc funcval(LObject *obj)
{
  int i;
  if (obj->type==LSTRING) {
    i = fsa_parse( func_fsa, LSTRINGVAL(obj) );
    if (i>=0) return funcs[i];
  }
  OOGLError(0,"unknown function: %s", LSTRINGVAL(obj));
  return nilfunc;
}
    
/*-----------------------------------------------------------------------
 * Function:	nilfunc
 * Description:	function which always returns nil
 * Args:	*args:
 * Returns:	nil
 * Author:	mbp
 * Date:	Wed Jan 15 15:25:20 1992
 * Notes:	takes any number of args, skips over them without
 *		evaluating them.
 */
static LObject *nilfunc(Pool *p)
{
  LObject *obj;
  int c;

  while ( (c=fnextc(p->inf,0)) != ')' && (c != EOF) ) {
    obj = LSexpr(p);
  }
  return Lnil;
}

/*-----------------------------------------------------------------------
 * Function:	LDefun
 * Description:	define a function
 * Args:	*name: name of function
 *		func: ptr to corresponding C function
 * Returns:	1 if successful, 0 otherwise
 * Author:	mbp
 * Date:	Wed Jan 15 15:25:57 1992
 * Notes:	may only LDefun MAXFUNCS functions
 */
int LDefun(char *name, LObjectFunc func)
{
  int i;
  for (i=0; i<MAXFUNCS; ++i)
    if (funcs[i] == NULL) {
      fsa_install( func_fsa, name, i );
      funcs[i] = func;
      return 1;
    }
  return 0;
}

int LTypeDefun(int type, PFI copy, PFI free, PFI string)
{
  if (typeok(type)) {
    copyfuncs[type] = copy;
    freefuncs[type] = free;
    stringfuncs[type] = string;
    return 1;
  }
  else
    return 0;
}

static LObject *car(Pool *p)
{
  LObject *list, *ans;

  list = LEvalSexpr(p);
  if (list->type!=LLIST) return nil;
  ans = LObjectCopy(LLISTVAL(list)->car);
  FreeLObject(list);
  return ans;
}

static LObject *cdr(Pool *p)
{
  LObject *list, *ans;

  list = LEvalSexpr(p);
  if (list->type!=LLIST) return nil;
  ans = NewLObject(LLIST);
  ans->val = LListCopy(LLISTVAL(list)->cdr);
  FreeLObject(list);
  return ans;
}

static LObject *quote(Pool *p)
{
  LObject *ans;

  ans = LSexpr(p);
  return ans;
}

/*-----------------------------------------------------------------------
 * Function:	LInit
 * Description:	initialize the lisp system
 * Returns:	
 * Author:	mbp
 * Date:	Wed Jan 15 15:29:10 1992
 * Notes:	must be called before any other lisp functions
 */
void LInit()
{
  int i;

  for (i=0; i<MAXFUNCS; ++i) funcs[i] = NULL;
  for (i=0; i<MAXTYPES; ++i) {
    freefuncs[i] = NULL;
    copyfuncs[i] = NULL;
    stringfuncs[i] = NULL;
  }
  func_fsa = fsa_initialize( NULL, -1 );
  {
    extern char *getenv();
    verbose = getenv("VERBOSE");
  }
  LDefun("car",	  car);
  LDefun("cdr",   cdr);
  LDefun("q",     quote);
  LDefun("quote", quote);
}

PFI freefunc(int type)
{
  switch (type) {
  case LLIST:
    return (PFI)FreeLList;
  case LSTRING:
  case LINT:
  case LFLOAT:
    return (PFI)OOGLFree;
  }
  if (typeok(type) && freefuncs[type]) return freefuncs[type];
  return noop;
}

PFI copyfunc(int type)
{
  switch (type) {
  case LLIST:
    return (PFI)LListCopy;
  case LSTRING:
    return (PFI)strdup;
  case LINT:
    return (PFI)intpdup;
  case LFLOAT:
    return (PFI)floatpdup;
  }
  if (typeok(type) && copyfuncs[type]) return copyfuncs[type];
  return noop;
}

char *intstring(int *i)
{
  String buf;
  sprintf(buf,"%1d",*i);
  return strdup(buf);
}

char *floatstring(float *f)
{
  String buf;
  sprintf(buf,"%1f",*f);
  return strdup(buf);
}

PFI stringfunc(int type)
{
  switch (type) {
  case LLIST:
    return (PFI)LListString;
  case LSTRING:
    return (PFI)strdup;
  case LINT:
    return (PFI)intstring;
  case LFLOAT:
    return (PFI)floatstring;
  }
  if (typeok(type) && copyfuncs[type]) return copyfuncs[type];
  return noop;
}

int *intpdup(int *ptr)
{
  int *new = (int*)malloc(sizeof(int));
  *new = *ptr;
  return new;
}

float *floatpdup(float *ptr)
{
  float *new = (float*)malloc(sizeof(float));
  *new = *ptr;
  return new;
}

int typeok(int type)
{
  return (type >= 0 && type < MAXTYPES);
}
