/* Copyright (c) 1992 The Geometry Center; University of Minnesota
   1300 South Second Street;  Minneapolis, MN  55454, USA;
   
This file is part of geomview/OOGL. geomview/OOGL is free software;
you can redistribute it and/or modify it only under the terms given in
the file COPYING, which you should have received along with this file.
This and other related software may be obtained via anonymous ftp from
geom.umn.edu; email: software@geom.umn.edu. */

/* Authors: Charlie Gunn, Stuart Levy, Tamara Munzner, Mark Phillips */

 /*
  * Geometry Routines
  * 
  * Geometry Supercomputer Project
  * 
  * ROUTINE DESCRIPTION:  Create and load a vert object from a file.
  * 
  */

#include <string.h>
#include "vectP.h"


Vect *
VectFLoad(file)
	FILE	*file;
{
    register Vect *v;
    int	 binary = 0, dimn = 3;
    char *p;
    int i;

    if (file == NULL) return NULL;

    /* new format possibility: 4-d vectors */
    if (fnextc(file, 1) == '4')	{
	dimn = 4;	
	if (fexpectstr(file,"4VECT")) return NULL;
	}

    else if (fnextc(file, 0) != 'V' || fexpectstr(file, "VECT"))
	return NULL;

    if(fnextc(file, 1) == 'B') {
	if(fexpectstr(file, "BINARY"))
	    return NULL;
	binary = 1;
	if(fnextc(file, 1) == '\n')
	    (void) fgetc(file);		/* Toss \n */
    }

    v = OOGLNewE(Vect, "VectFLoad: Vect");

    GGeomInit(v, VectMethods(), VECTMAGIC, NULL);
    v->flag = (dimn == 4) ? 1 : 0;
    v->vnvert = NULL;
    v->vncolor = NULL;
    v->p = NULL;
    v->c = NULL;

    if(fgetni(file, 1, &v->nvec, binary) <= 0 ||
       fgetni(file, 1, &v->nvert, binary) <= 0 ||
       fgetni(file, 1, &v->ncolor, binary) <= 0) {
	OOGLError(1, "Reading VECT header counts");
	goto bogus;
    }
    if(!vSane(v)) {
	OOGLError(1, "inconsistent VECT header counts %d %d %d", v->nvec, v->nvert, v->ncolor);
	goto bogus;
    }

    v->vnvert = OOGLNewNE(short, 2*v->nvec, "VECT nvec counts");
    v->p = OOGLNewNE(HPoint3, v->nvert, "VECT vertices");
    v->c = OOGLNewNE(ColorA, (v->ncolor>0) ? v->ncolor : 1, "VECT colors");

    v->vncolor = v->vnvert + v->nvec;

    if((i = fgetns(file, v->nvec, v->vnvert, binary)) < v->nvec) {
	OOGLError(1, "Reading VECT polyline count %d/%d", i, v->nvec);
	goto bogus; 
    }
    if((i = fgetns(file, v->nvec, v->vncolor, binary)) < v->nvec) {
	OOGLError(1, "Reading VECT color count %d/%d", i, v->nvec);
	goto bogus;
    }
    /* if the points are 3D points, we have to convert them to the native
	4D data structure */
    if (dimn == 3)	{
	register HPoint3 *p;

	for(i = v->nvert, p = v->p; --i >= 0; p++) {
	    if (fgetnf(file, 3, (float *)p, binary) < 3) {
		OOGLError(1, "Reading %dth VECT vertex", v->nvert-i);
		goto bogus;
	    }
	    p->w = 1;
	  }
	}
    else {
	i = fgetnf(file, 4*v->nvert, (float *)v->p, binary);
	if(i < 4*v->nvert) {
	    OOGLError(1, "Reading 4VECT vertex %d/%d", i/4, v->nvert);
	    goto bogus;
	}
      }
    if (v->ncolor > 0 &&
	(i = fgetnf(file, 4*v->ncolor, (float *)v->c, binary)) < 4*v->ncolor) {
	    OOGLError(1, "Reading VECT color %d/%d", i/4, v->ncolor);
	    goto bogus;
	}

    if(!VectSane(v)) {
	OOGLError(1, "VECT polyline/color counts inconsistent with header");
	goto bogus;
    }

    return v;

  bogus:
    GeomDelete((Geom *)v);
    return NULL;
}


Vect *
VectLoad(name)
char           *name;
{
    Vect *v;
    FILE *inf;

    if((inf = fopen(name, "r")) == NULL) {
	perror(name);
	return NULL;
    }
    v = VectFLoad(inf);
    fclose(inf);

    return (v);
}
