/* Copyright (c) 1992 The Geometry Center; University of Minnesota
   1300 South Second Street;  Minneapolis, MN  55454, USA;
   
This file is part of geomview/OOGL. geomview/OOGL is free software;
you can redistribute it and/or modify it only under the terms given in
the file COPYING, which you should have received along with this file.
This and other related software may be obtained via anonymous ftp from
geom.umn.edu; email: software@geom.umn.edu. */
static char *copyright = "Copyright (C) 1992 The Geometry Center";

/* Authors: Charlie Gunn, Stuart Levy, Tamara Munzner, Mark Phillips */

#include "vectP.h"

/*
 * Vect object creation, editing, copying, deletion.
 */

Vect *
VectCopy(register Vect *ov)
{
	register Vect  *v;

	if (ov == NULL) return (NULL);

	v = OOGLNewE(Vect, "new Vect");

	*v = *ov;
	v->p = OOGLNewNE(HPoint3, ov->nvert, "Vect vertices");
	v->c = ov->ncolor ? OOGLNewNE(ColorA, ov->ncolor, "Vect colors")
			  : NULL;
	v->vnvert = OOGLNewNE(short, 2*ov->nvec, "Vect nverts");
	v->vncolor = v->vnvert + ov->nvec;

	memcpy(v->p, ov->p, ov->nvert * sizeof(HPoint3));
	memcpy(v->c, ov->c, ov->ncolor * sizeof(ColorA));
	memcpy(v->vnvert, ov->vnvert, ov->nvec * sizeof(short));
	memcpy(v->vncolor, ov->vncolor, ov->nvec * sizeof(short));
	return(v);
}

void
VectDelete( register Vect *v )
{
    if (v) {
	if(v->p != NULL) OOGLFree(v->p);
	if(v->c != NULL) OOGLFree(v->c);
	if(v->vnvert != NULL) OOGLFree(v->vnvert);
    }
}

Vect *
VectCreate ( Vect *exist, GeomClass *classp, va_list a_list )
{
    register Vect *vect;
    int attr, copy = 1;

    short *vectcounts;	/* vectcounts[0..nvec-1] */
    short *colorcounts;	/* colorcounts[0..nvec-1] */
    Point3	*v3;
    HPoint3	*v4;			/* vertices[0..nvert-1] */
    ColorA	*colors;		/* colors[0..ncolor-1] */

    if (exist == NULL) {
	vect = OOGLNewE(Vect, "new vect");
	vect->flag = 0;
	vect->nvec = 0;
	vect->nvert = 0;
	vect->ncolor = 0;
    } else {
	/* Should check that exist is a vect */
	vect = exist;
    }

    while (attr = va_arg (a_list, int)) switch (attr) {
	case CR_FLAG:
	    vect->flag = va_arg (a_list, int);
	    break;
	case CR_NVECT:
	    vect->nvec = va_arg (a_list, int);
	    break;
	case CR_NVERT:
	    vect->nvert = va_arg (a_list, int);
	    break;
	case CR_NCOLR:
	    vect->ncolor = va_arg (a_list, int);
	    break;
	case CR_VECTC:
	    vectcounts = va_arg (a_list, short *);
	    break;
	case CR_COLRC:
	    colorcounts = va_arg (a_list, short *);
	    break;
	case CR_4D:
	    vect->flag |= VECT_4D;
	    break;
	case CR_POINT:
	    v3 = va_arg (a_list, Point3 *);
	    break;
	case CR_POINT4:
	    v4 = va_arg (a_list, HPoint3 *);
	    break;
	case CR_COLOR:
	    colors = va_arg (a_list, ColorA *);
	    break;
	default:
	    if (GeomDecorate (vect, &copy, attr, &a_list)) {
		OOGLError (0, "VectCreate: Undefined option: %d\n", attr);
		OOGLFree (vect);
		return NULL;
	    }
    }

    vect->vnvert = OOGLNewNE(short, 2*vect->nvec, "vect vert counts");
    vect->p = OOGLNewNE(HPoint3, vect->nvert, "vect points");
    vect->c = OOGLNewNE(ColorA, vect->ncolor, "vect colors");

    if(0 > vect->ncolor || vect->ncolor > vect->nvec || vect->nvec > vect->nvert || vect->nvert > 99999999) {
	OOGLError (0, "VectCreate: nvec=%d, nvert=%d, ncolor=%d; need 0<=ncolor<=nvec<=nvert<=1e8", vect->nvec, vect->nvert, vect->ncolor);
	return(NULL);
    }

    vect->vncolor = vect->vnvert + vect->nvec;
 
    memcpy(vect->vnvert, vectcounts, vect->nvec*sizeof(*vect->vnvert));
    memcpy(vect->vncolor, colorcounts, vect->nvec*sizeof(*vect->vncolor));
    if (vect->flag & VECT_4D)
        memcpy(vect->p, v4, vect->nvert*sizeof(Point3));
    else
	Pt3ToPt4(v3, vect->p, vect->nvert);
    memcpy(vect->c, colors, vect->ncolor*sizeof(ColorA));

    if(!VectSane(vect)) {
	OOGLError (0, "VectCreate: Bogus data supplied");
	GeomDelete((Geom *)vect);
	return NULL;
    }

    if (exist != NULL) return exist;

    GGeomInit (vect, classp, VECTMAGIC, NULL);
    return vect;
}

int
VectSane(vect)
	register Vect *vect;
{
	register int i;
	register int vleft, cleft;
	register short *p, *c;

	if(!vSane(vect))
		return(0);
	vleft = vect->nvert;
	cleft = vect->ncolor;
	for(i = vect->nvec, p = vect->vnvert, c = vect->vncolor; --i >= 0; p++, c++) {
		if(*p == 0 || (vleft -= vcount(*p)) < 0)
			return(0);
		if(*c < 0 || (cleft -= *c) < 0)
			return(0);
	}
	if(vleft || cleft)
		return(0);
	return(1);
}
